"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defaultMessages = exports.announceToScreenReader = exports.getLiveRegion = void 0;

/**
 * This helper creates a live region that announces the results of certain
 * actions (e.g. searching, paging, etc.), that are otherwise invisible
 * to screen reader users.
 *
 * @see https://developer.mozilla.org/en-US/docs/Web/Accessibility/ARIA/ARIA_Live_Regions
 */
var regionId = "search-ui-screen-reader-notifications";
var hasDOM = typeof document !== "undefined"; // Prevent errors in SSR apps

var getLiveRegion = function getLiveRegion() {
  if (!hasDOM) return;
  var region = document.getElementById(regionId);
  if (region) return region;
  region = document.createElement("div");
  region.id = regionId;
  region.setAttribute("role", "status");
  region.setAttribute("aria-live", "polite");
  /**
   * Visually-hidden CSS that's still available to screen readers.
   * We're avoiding putting this in a stylesheet to ensure that this
   * still works for users that opt for custom views & CSS. We're
   * also opting to use CSSOM instead of inline styles to avoid
   * Content Security Policy warnings.
   *
   * @see https://accessibility.18f.gov/hidden-content/
   */

  region.style.position = "absolute";
  region.style.width = "1px";
  region.style.height = "1px";
  region.style.margin = "-1px";
  region.style.padding = "0";
  region.style.border = "0";
  region.style.overflow = "hidden";
  region.style.clip = "rect(0 0 0 0)";
  document.body.appendChild(region);
  return region;
};

exports.getLiveRegion = getLiveRegion;

var announceToScreenReader = function announceToScreenReader(announcement) {
  if (hasDOM) {
    var region = getLiveRegion();
    region.textContent = announcement;
  }
};

exports.announceToScreenReader = announceToScreenReader;
var defaultMessages = {
  searchResults: function searchResults(_ref) {
    var start = _ref.start,
        end = _ref.end,
        totalResults = _ref.totalResults,
        searchTerm = _ref.searchTerm;
    var message = "Showing ".concat(start, " to ").concat(end, " results out of ").concat(totalResults);
    if (searchTerm) message += ", searching for \"".concat(searchTerm, "\".");
    return message;
  }
};
exports.defaultMessages = defaultMessages;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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