/*
 * Decompiled with CFR 0.152.
 */
package com.cburch.logisim.fpga.download;

import com.cburch.logisim.fpga.Strings;
import com.cburch.logisim.fpga.data.BoardInformation;
import com.cburch.logisim.fpga.data.DriveStrength;
import com.cburch.logisim.fpga.data.FpgaClass;
import com.cburch.logisim.fpga.data.FpgaIoInformationContainer;
import com.cburch.logisim.fpga.data.IoStandards;
import com.cburch.logisim.fpga.data.MapComponent;
import com.cburch.logisim.fpga.data.MappableResourcesContainer;
import com.cburch.logisim.fpga.data.PullBehaviors;
import com.cburch.logisim.fpga.designrulecheck.Netlist;
import com.cburch.logisim.fpga.download.Download;
import com.cburch.logisim.fpga.download.DownloadBase;
import com.cburch.logisim.fpga.download.VendorDownload;
import com.cburch.logisim.fpga.file.FileWriter;
import com.cburch.logisim.fpga.gui.Reporter;
import com.cburch.logisim.fpga.settings.VendorSoftware;
import com.cburch.logisim.util.LineBuffer;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class XilinxDownload
implements VendorDownload {
    private final VendorSoftware xilinxVendor = VendorSoftware.getSoftware('\u0001');
    private final String scriptPath;
    private final String projectPath;
    private final String sandboxPath;
    private final String ucfPath;
    private final Netlist rootNetList;
    private MappableResourcesContainer mapInfo;
    private final BoardInformation boardInfo;
    private final List<String> entities;
    private final List<String> architectures;
    private final String HdlType;
    private final String bitfileExt;
    private final boolean isCpld;
    private final boolean writeToFlash;
    private static final String VHDL_LIST_FILE = "XilinxVHDLList.prj";
    private static final String SCRIPT_FILE = "XilinxScript.cmd";
    private static final String UCF_FILE = "XilinxConstraints.ucf";
    private static final String DOWNLOAD_FILE = "XilinxDownload";
    private static final String MCS_FILE = "XilinxProm.mcs";
    private static final Integer BUFFER_SIZE = 16384;

    public XilinxDownload(String projectPath, Netlist rootNetList, BoardInformation boardInfo, List<String> entities, List<String> architectures, String hdlType, boolean writeToFlash) {
        this.projectPath = projectPath;
        this.sandboxPath = DownloadBase.getDirectoryLocation(projectPath, DownloadBase.SANDBOX_PATH);
        this.scriptPath = DownloadBase.getDirectoryLocation(projectPath, DownloadBase.SCRIPT_PATH);
        this.ucfPath = DownloadBase.getDirectoryLocation(projectPath, DownloadBase.UCF_PATH);
        this.rootNetList = rootNetList;
        this.boardInfo = boardInfo;
        this.entities = entities;
        this.architectures = architectures;
        this.HdlType = hdlType;
        this.writeToFlash = writeToFlash;
        String part = boardInfo.fpga.getPart().toUpperCase();
        this.isCpld = part.startsWith("XC2C") || part.startsWith("XA2C") || part.startsWith("XCR3") || part.startsWith("XC9500") || part.startsWith("XA9500");
        this.bitfileExt = this.isCpld ? "jed" : "bit";
    }

    @Override
    public int getNumberOfStages() {
        return 5;
    }

    @Override
    public String getStageMessage(int stage) {
        return switch (stage) {
            case 0 -> Strings.S.get("XilinxSynth");
            case 1 -> Strings.S.get("XilinxContraints");
            case 2 -> Strings.S.get("XilinxMap");
            case 3 -> Strings.S.get("XilinxPAR");
            case 4 -> Strings.S.get("XilinxBit");
            default -> "unknown";
        };
    }

    @Override
    public ProcessBuilder performStep(int stage) {
        return switch (stage) {
            case 0 -> this.stage0Synth();
            case 1 -> this.stage1Constraints();
            case 2 -> this.stage2Map();
            case 3 -> this.stage3Par();
            case 4 -> this.stage4Bit();
            default -> null;
        };
    }

    @Override
    public boolean readyForDownload() {
        return new File(this.sandboxPath + "logisimTopLevelShell." + this.bitfileExt).exists();
    }

    @Override
    public ProcessBuilder downloadToBoard() {
        BufferedInputStream bitfileIn;
        if (!this.boardInfo.fpga.isUsbTmcDownloadRequired().booleanValue()) {
            ArrayList<String> command = new ArrayList<String>();
            command.add(this.xilinxVendor.getBinaryPath(5));
            command.add("-batch");
            command.add(this.scriptPath.replace(this.projectPath, "../") + File.separator + DOWNLOAD_FILE);
            ProcessBuilder xilinx = new ProcessBuilder(command);
            xilinx.directory(new File(this.sandboxPath));
            return xilinx;
        }
        Reporter.report.clearConsole();
        boolean usbtmcdevice = new File("/dev/usbtmc0").exists();
        if (!usbtmcdevice) {
            Reporter.report.addFatalError(Strings.S.get("XilinxUsbTmc"));
            return null;
        }
        File bitfile = new File(this.sandboxPath + "logisimTopLevelShell." + this.bitfileExt);
        byte[] bitfileBuffer = new byte[BUFFER_SIZE.intValue()];
        int bitfileBufferSize = 0;
        try {
            bitfileIn = new BufferedInputStream(new FileInputStream(bitfile));
        }
        catch (FileNotFoundException e) {
            Reporter.report.addFatalError(Strings.S.get("XilinxOpenFailure", bitfile));
            return null;
        }
        File usbtmc = new File("/dev/usbtmc0");
        try {
            BufferedOutputStream usbtmcOut = new BufferedOutputStream(new FileOutputStream(usbtmc));
            usbtmcOut.write("FPGA ".getBytes());
            bitfileBufferSize = bitfileIn.read(bitfileBuffer, 0, BUFFER_SIZE);
            while (bitfileBufferSize > 0) {
                usbtmcOut.write(bitfileBuffer, 0, bitfileBufferSize);
                bitfileBufferSize = bitfileIn.read(bitfileBuffer, 0, BUFFER_SIZE);
            }
            usbtmcOut.close();
            bitfileIn.close();
        }
        catch (IOException e) {
            Reporter.report.addFatalError(Strings.S.get("XilinxUsbTmcError"));
        }
        return null;
    }

    @Override
    public boolean createDownloadScripts() {
        String jtagPos = String.valueOf(this.boardInfo.fpga.getFpgaJTAGChainPosition());
        File scriptFile = FileWriter.getFilePointer(this.scriptPath, SCRIPT_FILE);
        File vhdlListFile = FileWriter.getFilePointer(this.scriptPath, VHDL_LIST_FILE);
        File ucfFile = FileWriter.getFilePointer(this.ucfPath, UCF_FILE);
        File downloadFile = FileWriter.getFilePointer(this.scriptPath, DOWNLOAD_FILE);
        if (scriptFile == null || vhdlListFile == null || ucfFile == null || downloadFile == null) {
            scriptFile = new File(this.scriptPath + SCRIPT_FILE);
            vhdlListFile = new File(this.scriptPath + VHDL_LIST_FILE);
            ucfFile = new File(this.ucfPath + UCF_FILE);
            downloadFile = new File(this.scriptPath + DOWNLOAD_FILE);
            return scriptFile.exists() && vhdlListFile.exists() && ucfFile.exists() && downloadFile.exists();
        }
        LineBuffer contents = LineBuffer.getBuffer().pair("JTAGPos", jtagPos).pair("fileExt", this.bitfileExt).pair("fileBaseName", "logisimTopLevelShell").pair("mcsFile", this.scriptPath + File.separator + MCS_FILE).pair("hdlType", this.HdlType.toUpperCase());
        for (String entity : this.entities) {
            contents.add("{{hdlType}} work \"{{1}}\"", entity);
        }
        for (String arch : this.architectures) {
            contents.add("{{hdlType}} work \"{{1}}\"", arch);
        }
        if (!FileWriter.writeContents(vhdlListFile, contents.get())) {
            return false;
        }
        contents.clear().add("run -top {{1}} -ofn logisim.ngc -ofmt NGC -ifn {{2}}{{3}} -ifmt mixed -p {{4}}", "logisimTopLevelShell", this.scriptPath.replace(this.projectPath, "../"), VHDL_LIST_FILE, XilinxDownload.getFpgaDeviceString(this.boardInfo));
        if (!FileWriter.writeContents(scriptFile, contents.get())) {
            return false;
        }
        contents.clear();
        contents.add("setmode -bscan");
        if (this.writeToFlash && this.boardInfo.fpga.isFlashDefined()) {
            if (this.boardInfo.fpga.getFlashName() == null) {
                Reporter.report.addFatalError(Strings.S.get("XilinxFlashMissing", this.boardInfo.getBoardName()));
            }
            contents.pair("flashPos", String.valueOf(this.boardInfo.fpga.getFlashJTAGChainPosition())).pair("flashName", this.boardInfo.fpga.getFlashName()).add("setmode -pff\nsetSubMode -pffserial\naddPromDevice -p {{JTAGPos}} -size 0 -name {{flashName}}\naddDesign -version 0 -name \"0\"\naddDeviceChain -index 0\naddDevice -p {{JTAGPos}} -file {{fileBaseName}}.{{fileExt}}\ngenerate -format mcs -fillvalue FF -output {{mcsFile}}\nsetMode -bs\nsetCable -port auto\nidentify\nassignFile -p {{flashPos}} -file {{mcsFile}}\nprogram -p {{flashPos}} -e -v\n");
        } else {
            contents.add("setcable -p auto").add("identify");
            if (!this.isCpld) {
                contents.add("assignFile -p {{JTAGPos}} -file {{fileBaseName}}.{{fileExt}}\nprogram -p {{JTAGPos}} -onlyFpga\n");
            } else {
                contents.add("assignFile -p {{JTAGPos}} -file logisim.{{fileExt}}\nprogram -p {{JTAGPos}} -e\n");
            }
        }
        contents.add("quit");
        if (!FileWriter.writeContents(downloadFile, contents.get())) {
            return false;
        }
        contents.clear();
        if (this.rootNetList.numberOfClockTrees() > 0 || this.rootNetList.requiresGlobalClockConnection()) {
            contents.pair("clock", "fpgaGlobalClock").pair("clockFreq", Download.getClockFrequencyString(this.boardInfo)).pair("clockPin", XilinxDownload.getXilinxClockPin(this.boardInfo)).add("NET \"{{clock}}\" {{clockPin}} ;\nNET \"{{clock}}\" TNM_NET = \"{{clock}}\" ;\nTIMESPEC \"TS_{{clock}}\" = PERIOD \"{{clock}}\" {{clockFreq}} HIGH 50 % ;\n");
        }
        contents.add(this.getPinLocStrings());
        return FileWriter.writeContents(ucfFile, contents.get());
    }

    private ArrayList<String> getPinLocStrings() {
        ArrayList<String> contents = new ArrayList<String>();
        StringBuilder temp = new StringBuilder();
        for (ArrayList<String> key : this.mapInfo.getMappableResources().keySet()) {
            MapComponent map = this.mapInfo.getMappableResources().get(key);
            for (int i = 0; i < map.getNrOfPins(); ++i) {
                if (!map.isMapped(i) || map.isOpenMapped(i) || map.isConstantMapped(i) || map.isInternalMapped(i)) continue;
                temp.setLength(0);
                temp.append("NET \"");
                if (map.isExternalInverted(i)) {
                    temp.append("n_");
                }
                temp.append(map.getHdlString(i)).append("\" ");
                temp.append("LOC = \"").append(map.getPinLocation(i)).append("\" ");
                FpgaIoInformationContainer info = map.getFpgaInfo(i);
                if (info != null) {
                    if (info.getPullBehavior() != '\u00ff' && info.getPullBehavior() != '\u0000') {
                        temp.append("| ").append(PullBehaviors.getConstrainedPullString(info.getPullBehavior())).append(" ");
                    }
                    if (info.getDrive() != '\u00ff' && info.getDrive() != '\u0000') {
                        temp.append("| DRIVE = ").append(DriveStrength.getConstrainedDriveStrength(info.getDrive())).append(" ");
                    }
                    if (info.getIoStandard() != '\u00ff' && info.getIoStandard() != '\u0000') {
                        temp.append("| IOSTANDARD = ").append(IoStandards.getConstraintedIoStandard(info.getIoStandard())).append(" ");
                    }
                }
                temp.append(";");
                contents.add(temp.toString());
            }
        }
        Map<String, String> LedArrayMap = DownloadBase.getLedArrayMaps(this.mapInfo, this.rootNetList, this.boardInfo);
        for (String key : LedArrayMap.keySet()) {
            contents.add("NET \"" + LedArrayMap.get(key) + "\" LOC=\"" + key + "\";");
        }
        return contents;
    }

    @Override
    public void setMapableResources(MappableResourcesContainer resources) {
        this.mapInfo = resources;
    }

    private ProcessBuilder stage0Synth() {
        LineBuffer command = LineBuffer.getBuffer();
        command.add(this.xilinxVendor.getBinaryPath(0)).add("-ifn").add(this.scriptPath.replace(this.projectPath, "../") + File.separator + SCRIPT_FILE).add("-ofn").add("logisim.log");
        ProcessBuilder stage0 = new ProcessBuilder(command.get());
        stage0.directory(new File(this.sandboxPath));
        return stage0;
    }

    private ProcessBuilder stage1Constraints() {
        LineBuffer command = LineBuffer.getBuffer();
        command.add(this.xilinxVendor.getBinaryPath(1)).add("-intstyle").add("ise").add("-uc").add(this.ucfPath.replace(this.projectPath, "../") + File.separator + UCF_FILE).add("logisim.ngc").add("logisim.ngd");
        ProcessBuilder stage1 = new ProcessBuilder(command.get());
        stage1.directory(new File(this.sandboxPath));
        return stage1;
    }

    private ProcessBuilder stage2Map() {
        if (this.isCpld) {
            return null;
        }
        LineBuffer command = LineBuffer.getBuffer();
        command.add(this.xilinxVendor.getBinaryPath(2)).add("-intstyle").add("ise").add("-o").add("logisim_map").add("logisim.ngd");
        ProcessBuilder stage2 = new ProcessBuilder(command.get());
        stage2.directory(new File(this.sandboxPath));
        return stage2;
    }

    private ProcessBuilder stage3Par() {
        LineBuffer command = LineBuffer.getBuffer();
        if (!this.isCpld) {
            command.add(this.xilinxVendor.getBinaryPath(3)).add("-w").add("-intstyle").add("ise").add("-ol").add("high").add("logisim_map").add("logisim_par").add("logisim_map.pcf");
        } else {
            String pinPullBehavior = switch (this.boardInfo.fpga.getUnusedPinsBehavior()) {
                case '\u0001' -> "pullup";
                case '\u0002' -> "pulldown";
                default -> "float";
            };
            FpgaClass fpga = this.boardInfo.fpga;
            command.add(this.xilinxVendor.getBinaryPath(6)).add("-p").add("{{1}}-{{2}}-{{3}}", fpga.getPart().toUpperCase(), fpga.getSpeedGrade(), fpga.getPackage().toUpperCase()).add("-intstyle").add("ise").add("-terminate").add(pinPullBehavior).add("-loc").add("on").add("-log").add("logisim_cpldfit.log").add("logisim.ngd");
        }
        ProcessBuilder stage3 = new ProcessBuilder(command.get());
        stage3.directory(new File(this.sandboxPath));
        return stage3;
    }

    private ProcessBuilder stage4Bit() {
        LineBuffer command = LineBuffer.getBuffer();
        if (!this.isCpld) {
            command.add(this.xilinxVendor.getBinaryPath(4)).add("-w");
            if (this.boardInfo.fpga.getUnusedPinsBehavior() == '\u0001') {
                command.add("-g").add("UnusedPin:PULLUP");
            }
            if (this.boardInfo.fpga.getUnusedPinsBehavior() == '\u0002') {
                command.add("-g").add("UnusedPin:PULLDOWN");
            }
            command.add("-g").add("StartupClk:CCLK").add("logisim_par").add("{{1}}.bit", "logisimTopLevelShell");
        } else {
            command.add(this.xilinxVendor.getBinaryPath(7)).add("-i").add("logisim.vm6");
        }
        ProcessBuilder stage4 = new ProcessBuilder(command.get());
        stage4.directory(new File(this.sandboxPath));
        return stage4;
    }

    private static String getFpgaDeviceString(BoardInformation currentBoard) {
        FpgaClass fpga = currentBoard.fpga;
        return String.format("%s-%s-%s", fpga.getPart(), fpga.getPackage(), fpga.getSpeedGrade());
    }

    private static String getXilinxClockPin(BoardInformation currentBoard) {
        StringBuilder result = new StringBuilder();
        result.append("LOC = \"").append(currentBoard.fpga.getClockPinLocation()).append("\"");
        if (currentBoard.fpga.getClockPull() == '\u0001') {
            result.append(" | PULLUP");
        }
        if (currentBoard.fpga.getClockPull() == '\u0002') {
            result.append(" | PULLDOWN");
        }
        if (currentBoard.fpga.getClockStandard() != '\u0000' && currentBoard.fpga.getClockStandard() != '\u00ff') {
            result.append(" | IOSTANDARD = ").append(IoStandards.BEHAVIOR_STRINGS[currentBoard.fpga.getClockStandard()]);
        }
        return result.toString();
    }

    @Override
    public boolean isBoardConnected() {
        return true;
    }
}

