/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.io.retry;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.ipc.RemoteException;

public class RetryPolicies {
    private static final Log LOG = LogFactory.getLog(RetryPolicies.class);
    private static ThreadLocal<Random> RANDOM = new ThreadLocal<Random>(){

        @Override
        protected Random initialValue() {
            return new Random();
        }
    };
    public static final RetryPolicy TRY_ONCE_THEN_FAIL = new TryOnceThenFail();
    public static final RetryPolicy TRY_ONCE_DONT_FAIL = new TryOnceDontFail();
    public static final RetryPolicy RETRY_FOREVER = new RetryForever();

    public static final RetryPolicy retryUpToMaximumCountWithFixedSleep(int maxRetries, long sleepTime, TimeUnit timeUnit) {
        return new RetryUpToMaximumCountWithFixedSleep(maxRetries, sleepTime, timeUnit);
    }

    public static final RetryPolicy retryUpToMaximumTimeWithFixedSleep(long maxTime, long sleepTime, TimeUnit timeUnit) {
        return new RetryUpToMaximumTimeWithFixedSleep(maxTime, sleepTime, timeUnit);
    }

    public static final RetryPolicy retryUpToMaximumCountWithProportionalSleep(int maxRetries, long sleepTime, TimeUnit timeUnit) {
        return new RetryUpToMaximumCountWithProportionalSleep(maxRetries, sleepTime, timeUnit);
    }

    public static final RetryPolicy exponentialBackoffRetry(int maxRetries, long sleepTime, TimeUnit timeUnit) {
        return new ExponentialBackoffRetry(maxRetries, sleepTime, timeUnit);
    }

    public static final RetryPolicy retryByException(RetryPolicy defaultPolicy, Map<Class<? extends Exception>, RetryPolicy> exceptionToPolicyMap) {
        return new ExceptionDependentRetry(defaultPolicy, exceptionToPolicyMap);
    }

    public static final RetryPolicy retryByRemoteException(RetryPolicy defaultPolicy, Map<Class<? extends Exception>, RetryPolicy> exceptionToPolicyMap) {
        return new RemoteExceptionDependentRetry(defaultPolicy, exceptionToPolicyMap);
    }

    static class ExponentialBackoffRetry
    extends RetryLimited {
        private Random r = new Random();

        public ExponentialBackoffRetry(int maxRetries, long sleepTime, TimeUnit timeUnit) {
            super(maxRetries, sleepTime, timeUnit);
            if (maxRetries < 0) {
                throw new IllegalArgumentException("maxRetries = " + maxRetries + " < 0");
            }
            if (maxRetries > 30) {
                throw new IllegalArgumentException("maxRetries = " + maxRetries + " > 30");
            }
        }

        @Override
        protected long calculateSleepTime(int retries) {
            return this.sleepTime * (long)this.r.nextInt(1 << retries + 1);
        }
    }

    static class RemoteExceptionDependentRetry
    implements RetryPolicy {
        RetryPolicy defaultPolicy;
        Map<String, RetryPolicy> exceptionNameToPolicyMap;

        public RemoteExceptionDependentRetry(RetryPolicy defaultPolicy, Map<Class<? extends Exception>, RetryPolicy> exceptionToPolicyMap) {
            this.defaultPolicy = defaultPolicy;
            this.exceptionNameToPolicyMap = new HashMap<String, RetryPolicy>();
            for (Map.Entry<Class<? extends Exception>, RetryPolicy> e : exceptionToPolicyMap.entrySet()) {
                this.exceptionNameToPolicyMap.put(e.getKey().getName(), e.getValue());
            }
        }

        @Override
        public boolean shouldRetry(Exception e, int retries) throws Exception {
            RetryPolicy policy = null;
            if (e instanceof RemoteException) {
                policy = this.exceptionNameToPolicyMap.get(((RemoteException)e).getClassName());
            }
            if (policy == null) {
                policy = this.defaultPolicy;
            }
            return policy.shouldRetry(e, retries);
        }
    }

    static class ExceptionDependentRetry
    implements RetryPolicy {
        RetryPolicy defaultPolicy;
        Map<Class<? extends Exception>, RetryPolicy> exceptionToPolicyMap;

        public ExceptionDependentRetry(RetryPolicy defaultPolicy, Map<Class<? extends Exception>, RetryPolicy> exceptionToPolicyMap) {
            this.defaultPolicy = defaultPolicy;
            this.exceptionToPolicyMap = exceptionToPolicyMap;
        }

        @Override
        public boolean shouldRetry(Exception e, int retries) throws Exception {
            RetryPolicy policy = this.exceptionToPolicyMap.get(e.getClass());
            if (policy == null) {
                policy = this.defaultPolicy;
            }
            return policy.shouldRetry(e, retries);
        }
    }

    public static class MultipleLinearRandomRetry
    implements RetryPolicy {
        private final List<Pair> pairs;
        private String myString;

        public MultipleLinearRandomRetry(List<Pair> pairs) {
            if (pairs == null || pairs.isEmpty()) {
                throw new IllegalArgumentException("pairs must be neither null nor empty.");
            }
            this.pairs = Collections.unmodifiableList(pairs);
        }

        @Override
        public boolean shouldRetry(Exception e, int curRetry) throws Exception {
            Pair p = this.searchPair(curRetry);
            if (p == null) {
                throw e;
            }
            double ratio = ((Random)RANDOM.get()).nextDouble() + 0.5;
            Thread.sleep(Math.round((double)p.sleepMillis * ratio));
            return true;
        }

        private Pair searchPair(int curRetry) {
            int i;
            for (i = 0; i < this.pairs.size() && curRetry > this.pairs.get((int)i).numRetries; curRetry -= this.pairs.get((int)i).numRetries, ++i) {
            }
            return i == this.pairs.size() ? null : this.pairs.get(i);
        }

        public int hashCode() {
            return this.toString().hashCode();
        }

        public boolean equals(Object that) {
            if (this == that) {
                return true;
            }
            if (that == null || this.getClass() != that.getClass()) {
                return false;
            }
            return this.toString().equals(that.toString());
        }

        public String toString() {
            if (this.myString == null) {
                this.myString = this.getClass().getSimpleName() + this.pairs;
            }
            return this.myString;
        }

        public static MultipleLinearRandomRetry parseCommaSeparatedString(String s) {
            String[] elements = s.split(",");
            if (elements.length == 0) {
                LOG.warn((Object)("Illegal value: there is no element in \"" + s + "\"."));
                return null;
            }
            if (elements.length % 2 != 0) {
                LOG.warn((Object)("Illegal value: the number of elements in \"" + s + "\" is " + elements.length + " but an even number of elements is expected."));
                return null;
            }
            ArrayList<Pair> pairs = new ArrayList<Pair>();
            int i = 0;
            while (i < elements.length) {
                int retries;
                int sleep;
                if ((sleep = MultipleLinearRandomRetry.parsePositiveInt(elements, i++, s)) == -1) {
                    return null;
                }
                if ((retries = MultipleLinearRandomRetry.parsePositiveInt(elements, i++, s)) == -1) {
                    return null;
                }
                pairs.add(new Pair(retries, sleep));
            }
            return new MultipleLinearRandomRetry(pairs);
        }

        private static int parsePositiveInt(String[] elements, int i, String originalString) {
            int n;
            String s = elements[i].trim();
            try {
                n = Integer.parseInt(s);
            }
            catch (NumberFormatException nfe) {
                LOG.warn((Object)("Failed to parse \"" + s + "\", which is the index " + i + " element in \"" + originalString + "\""), (Throwable)nfe);
                return -1;
            }
            if (n <= 0) {
                LOG.warn((Object)("The value " + n + " <= 0: it is parsed from the string \"" + s + "\" which is the index " + i + " element in \"" + originalString + "\""));
                return -1;
            }
            return n;
        }

        public static class Pair {
            final int numRetries;
            final int sleepMillis;

            public Pair(int numRetries, int sleepMillis) {
                if (numRetries < 0) {
                    throw new IllegalArgumentException("numRetries = " + numRetries + " < 0");
                }
                if (sleepMillis < 0) {
                    throw new IllegalArgumentException("sleepMillis = " + sleepMillis + " < 0");
                }
                this.numRetries = numRetries;
                this.sleepMillis = sleepMillis;
            }

            public String toString() {
                return this.numRetries + "x" + this.sleepMillis + "ms";
            }
        }
    }

    static class RetryUpToMaximumCountWithProportionalSleep
    extends RetryLimited {
        public RetryUpToMaximumCountWithProportionalSleep(int maxRetries, long sleepTime, TimeUnit timeUnit) {
            super(maxRetries, sleepTime, timeUnit);
        }

        @Override
        protected long calculateSleepTime(int retries) {
            return this.sleepTime * (long)(retries + 1);
        }
    }

    static class RetryUpToMaximumTimeWithFixedSleep
    extends RetryUpToMaximumCountWithFixedSleep {
        public RetryUpToMaximumTimeWithFixedSleep(long maxTime, long sleepTime, TimeUnit timeUnit) {
            super((int)(maxTime / sleepTime), sleepTime, timeUnit);
        }
    }

    static class RetryUpToMaximumCountWithFixedSleep
    extends RetryLimited {
        public RetryUpToMaximumCountWithFixedSleep(int maxRetries, long sleepTime, TimeUnit timeUnit) {
            super(maxRetries, sleepTime, timeUnit);
        }

        @Override
        protected long calculateSleepTime(int retries) {
            return this.sleepTime;
        }
    }

    static abstract class RetryLimited
    implements RetryPolicy {
        final int maxRetries;
        final long sleepTime;
        final TimeUnit timeUnit;
        private String myString;

        RetryLimited(int maxRetries, long sleepTime, TimeUnit timeUnit) {
            if (maxRetries < 0) {
                throw new IllegalArgumentException("maxRetries = " + maxRetries + " < 0");
            }
            if (sleepTime < 0L) {
                throw new IllegalArgumentException("sleepTime = " + sleepTime + " < 0");
            }
            this.maxRetries = maxRetries;
            this.sleepTime = sleepTime;
            this.timeUnit = timeUnit;
        }

        @Override
        public boolean shouldRetry(Exception e, int retries) throws Exception {
            if (retries >= this.maxRetries) {
                throw e;
            }
            try {
                this.timeUnit.sleep(this.calculateSleepTime(retries));
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            return true;
        }

        protected abstract long calculateSleepTime(int var1);

        public int hashCode() {
            return this.toString().hashCode();
        }

        public boolean equals(Object that) {
            if (this == that) {
                return true;
            }
            if (that == null || this.getClass() != that.getClass()) {
                return false;
            }
            return this.toString().equals(that.toString());
        }

        public String toString() {
            if (this.myString == null) {
                this.myString = this.getClass().getSimpleName() + "(maxRetries=" + this.maxRetries + ", sleepTime=" + this.sleepTime + " " + (Object)((Object)this.timeUnit) + ")";
            }
            return this.myString;
        }
    }

    static class RetryForever
    implements RetryPolicy {
        RetryForever() {
        }

        @Override
        public boolean shouldRetry(Exception e, int retries) throws Exception {
            return true;
        }
    }

    static class TryOnceDontFail
    implements RetryPolicy {
        TryOnceDontFail() {
        }

        @Override
        public boolean shouldRetry(Exception e, int retries) throws Exception {
            return false;
        }
    }

    static class TryOnceThenFail
    implements RetryPolicy {
        TryOnceThenFail() {
        }

        @Override
        public boolean shouldRetry(Exception e, int retries) throws Exception {
            throw e;
        }
    }
}

