/* ---------------------------------------------------------------------------


  This file is part of the ``ltl2smv'' package of NuSMV version 2.
  Copyright (C) 2005 by FBK-irst.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information on NuSMV see <http://nusmv.fbk.eu>
  or email to <nusmv-users@fbk.eu>.
  Please report bugs to <nusmv-users@fbk.eu>.

  To contact the NuSMV development board, email to <nusmv@fbk.eu>. 

-----------------------------------------------------------------------------*/

/*!
  \author Andrei Tchaltsev, Marco Roveri
  \brief A function to run ltl2smv routine.

  Here we perform a convertion from LTL formula to SMV module.
  The invoker provides the LTL fromula in the form of node_ptr expression.
  The function 'ltl2smv' will convert this formula to a SMV module, which is
  also node_ptr MODULE.

  This file provides routines for reducing LTL model
  checking to CTL model checking. This work is based on the work
  presented in \[1\]<br>

  <ul><li> O. Grumberg E. Clarke and K. Hamaguchi.  <cite>Another Look
          at LTL Model Checking</cite>. <em>Formal Methods in System
          Design, 10(1):57--71, February 1997.</li>
  </ul>

*/


#ifndef __NUSMV_CORE_LTL_LTL2SMV_LTL2SMV_H__
#define __NUSMV_CORE_LTL_LTL2SMV_LTL2SMV_H__

#include "nusmv/core/node/node.h"

/*!
  \brief 
*/
typedef struct Ltl2SmvPrefixes_TAG
{
  char* pre_prefix;
  char* prefix_name;
  char* ltl_module_base_name;
} Ltl2SmvPrefixes;

/*---------------------------------------------------------------------------*/
/* MACRO declaration                                                         */
/*---------------------------------------------------------------------------*/
/**MAcro********************************************************************

  Synopsis    [This macro is a string represeting the prefix of the name
  of modules generated by ltl2smv procedure]

  Description []

  SeeAlso     [ltl2smv]

******************************************************************************/

/*!
  \brief \todo Missing synopsis

  \todo Missing description
*/
#define LTL_MODULE_BASE_NAME "ltl_spec_"

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

/*!
  \brief The routine converting an LTL formula to am SMV Module.

  The parameters are:
  uniqueId - is a positive integer number, which is converted to string
            representation and then used as a part of the generated
            SMV models name.  (_LTL_uniqueId_SPECF_N_)
  in_ltl_expr - the input LTL Formula in the form of node_ptr. The
            expression should be flattened, rewritten wrt input
            variables, negated, i.e. be ready for conversion without
            any additional transformations

  The function also performs memory-sharing on the input expression,
  since the returned module may be smaller if the memory of
  expression is shared. So DO NOT modify the output expressions.

  But the invoker can modify the declarations and module itself (but
  not their expressions). See generate_smv_module for more info.

  The return value is the created SMV module in the form of node_ptr.

  The computed module may contain more than one justice condition.
  

  \sa ltl2smv_core
*/
node_ptr ltl2smv(NuSMVEnv_ptr env,
                        unsigned int uniqueId,
                        node_ptr in_ltl_expr);

/*!
  \brief The main routine converting an LTL formula to am SMV Module.

  The parameters are:
  uniqueId - is a positive integer number, which is converted to string
            representation and then used as a part of the generated
            SMV models name.  (_LTL_uniqueId_SPECF_N_)
  in_ltl_expr - the input LTL Formula in the form of node_ptr. The
            expression should be flattened, rewritten wrt input
            variables, negated, i.e. be ready for conversion without
            any additional transformations
  single_justice - is a boolean that when true, it builds a monitor
            such that there is a single fairness condition instead of
            possibly many.

  The function also performs memory-sharing on the input expression,
  since the returned module may be smaller if the memory of
  expression is shared. So DO NOT modify the output expressions.

  But the invoker can modify the declarations and module itself (but
  not their expressions). See generate_smv_module for more info.

  The return value is the created SMV module in the form of node_ptr.
  
*/
node_ptr ltl2smv_core(NuSMVEnv_ptr env,
                      unsigned int uniqueId,
                      node_ptr in_ltl_expr,
                      boolean single_justice,
                      const Ltl2SmvPrefixes* prefixes);

/*!
  \brief The routine converting an LTL formula to am SMV Module.

  The parameters are:
  uniqueId - is a positive integer number, which is converted to string
            representation and then used as a part of the generated
            SMV models name.  (_LTL_uniqueId_SPECF_N_)
  in_ltl_expr - the input LTL Formula in the form of node_ptr. The
            expression should be flattened, rewritten wrt input
            variables, negated, i.e. be ready for conversion without
            any additional transformations

  The function also performs memory-sharing on the input expression,
  since the returned module may be smaller if the memory of
  expression is shared. So DO NOT modify the output expressions.

  But the invoker can modify the declarations and module itself (but
  not their expressions). See generate_smv_module for more info.

  The return value is the created SMV module in the form of node_ptr.

  The computed module will contain less than one justice condition.

  
*/
node_ptr ltl2smv_single_justice(NuSMVEnv_ptr env,
                                       unsigned int uniqueId,
                                       node_ptr in_ltl_expr);

#endif /* __NUSMV_CORE_LTL_LTL2SMV_LTL2SMV_H__ */
