# 16 "src/approx_lexer.mll"
 

open Lexing

include Approx_tokens

let list_last l = List.hd (List.rev l)

let lines_starts = ref []

(* The table of keywords *)

let keywords = [
  "and", AND;
  "as", AS;
  "assert", ASSERT;
  "begin", BEGIN;
  "class", CLASS;
  "constraint", CONSTRAINT;
  "do", DO;
  "done", DONE;
  "downto", DOWNTO;
  "else", ELSE;
  "end", END;
  "exception", EXCEPTION;
  "external", EXTERNAL;
  "false", FALSE;
  "for", FOR;
  "fun", FUN;
  "function", FUNCTION;
  "functor", FUNCTOR;
  "if", IF;
  "in", IN;
  "include", INCLUDE;
  "inherit", INHERIT;
  "initializer", INITIALIZER;
  "lazy", LAZY;
  "let", LET;
  "match", MATCH;
  "method", METHOD;
  "module", MODULE;
  "mutable", MUTABLE;
  "new", NEW;
  "object", OBJECT;
  "of", OF;
  "open", OPEN;
  "or", OR;
  "private", PRIVATE;
  "rec", REC;
  "sig", SIG;
  "struct", STRUCT;
  "then", THEN;
  "to", TO;
  "true", TRUE;
  "try", TRY;
  "type", TYPE;
  "val", VAL;
  "virtual", VIRTUAL;
  "when", WHEN;
  "while", WHILE;
  "with", WITH;

  "mod", INFIXOP3("mod");
  "land", INFIXOP3("land");
  "lor", INFIXOP3("lor");
  "lxor", INFIXOP3("lxor");
  "lsl", INFIXOP4("lsl");
  "lsr", INFIXOP4("lsr");
  "asr", INFIXOP4("asr");
]

let keyword_table =
  let t = Hashtbl.create 149 in
  List.iter (fun (x,y) -> Hashtbl.add t x y) keywords;
  t

let lexer_extensions : (Lexing.lexbuf -> Approx_tokens.token) list ref = ref []

let enable_extension name =
  let t = IndentExtend.find name in
  List.iter
    (fun (x,y) -> Hashtbl.add keyword_table x y)
    t.IndentExtend.keywords;
  match t.IndentExtend.lexer with
  | None -> ()
  | Some f -> lexer_extensions := f :: !lexer_extensions

let disable_extensions () =
  Hashtbl.clear keyword_table;
  lexer_extensions := [];
  List.iter (fun (x,y) -> Hashtbl.add keyword_table x y) keywords

(* To buffer string literals *)

let initial_string_buffer = Bytes.create 256
let string_buff = ref initial_string_buffer
let string_index = ref 0

let reset_string_buffer () =
  string_buff := initial_string_buffer;
  string_index := 0

let store_string_char c =
  if !string_index >= Bytes.length (!string_buff) then begin
    let new_buff = Bytes.create (Bytes.length (!string_buff) * 2) in
    Bytes.blit (!string_buff) 0 new_buff 0 (Bytes.length (!string_buff));
    string_buff := new_buff
  end;
  Bytes.unsafe_set (!string_buff) (!string_index) c;
  incr string_index

let get_stored_string () =
  let s = Bytes.sub (!string_buff) 0 (!string_index) in
  string_buff := initial_string_buffer;
  Bytes.to_string s

(* To store the position of the beginning of a string and comment *)
let string_start_loc = ref (-1);;
let quotation_start_loc = ref (-1);;
let quotation_kind = ref (`Camlp4 "<:<": [ `Camlp4 of string | `Ppx of string ]);;
type in_comment = Comment
                | Code
                | Cinaps (* Like code, but started / ended with "(*$" / "$*)"
                            rather than "{[" / "]}" *)
                | Verbatim
                | CommentCont
let comment_stack : in_comment list ref =
  ref []
;;
let entering_inline_code_block = ref false;;
let rec close_comment () = match !comment_stack with
  | Comment :: r -> comment_stack := r; COMMENT
  | CommentCont :: r -> comment_stack := r; COMMENTCONT
  | (Code | Cinaps | Verbatim) :: r ->
      comment_stack := r; ignore (close_comment ()); COMMENTCONT
  | [] -> assert false
;;
let in_comment () = match !comment_stack with
  | (Comment | CommentCont | Verbatim) :: _ -> true
  | (Code | Cinaps) :: _ | [] -> false
;;
let in_verbatim () = List.mem Verbatim !comment_stack
;;
let rewind lexbuf n =
  lexbuf.lex_curr_pos <- lexbuf.lex_curr_pos - n;
  let curpos = lexbuf.lex_curr_p in
  lexbuf.lex_curr_p <- { curpos with pos_cnum = curpos.pos_cnum - n }
;;
let check_commentclose lexbuf f =
  let s = Lexing.lexeme lexbuf in
  let len = String.length s in
  if s.[len - 1] <> ')' then f s else
    let rollback =
      if len >= 2 && !comment_stack <> [] && s.[len - 2] = '*'
      then 2 (* this is a comment end, unparse it *)
      else 1 (* only unparse the closing paren *)
    in
    let op = String.sub s 0 (len - rollback) in
    rewind lexbuf rollback;
    f op
;;
let init () =
  lines_starts := [];
  (* disable_extensions(); *)
  reset_string_buffer();
  string_start_loc := -1;
  quotation_start_loc := -1;
  quotation_kind := `Camlp4 "<:<";
  comment_stack := [];
  entering_inline_code_block := false
;;


(* To translate escape sequences *)

let char_for_backslash = function
  | 'n' -> '\010'
  | 'r' -> '\013'
  | 'b' -> '\008'
  | 't' -> '\009'
  | c   -> c

let can_overflow f lexbuf =
  let s = Lexing.lexeme lexbuf in
  try InRange (f s) with Failure _ -> Overflow s

let char_for_decimal_code i s =
  let c = 100 * (Char.code(s.[i]) - 48) +
      10 * (Char.code(s.[i+1]) - 48) +
      (Char.code(s.[i+2]) - 48) in
  if (c < 0 || c > 255) then
    failwith "Bad escaped decimal char"
  else Char.chr c

let char_for_hexadecimal_code lexbuf i =
  let d1 = Char.code (Lexing.lexeme_char lexbuf i) in
  let val1 = if d1 >= 97 then d1 - 87
    else if d1 >= 65 then d1 - 55
    else d1 - 48
  in
  let d2 = Char.code (Lexing.lexeme_char lexbuf (i+1)) in
  let val2 = if d2 >= 97 then d2 - 87
    else if d2 >= 65 then d2 - 55
    else d2 - 48
  in
  Char.chr (val1 * 16 + val2)

(* To convert integer literals, allowing max_int + 1 (PR#4210) *)

let cvt_int_literal s =
  - int_of_string ("-" ^ s)
let cvt_int32_literal s =
  Int32.neg (Int32.of_string ("-" ^ String.sub s 0 (String.length s - 1)))
let cvt_int64_literal s =
  Int64.neg (Int64.of_string ("-" ^ String.sub s 0 (String.length s - 1)))
let cvt_nativeint_literal s =
  Nativeint.neg (Nativeint.of_string ("-" ^ String.sub s 0 (String.length s - 1)))

(* Remove underscores from float literals *)

let remove_underscores s =
  let s = Bytes.of_string s in
  let l = Bytes.length s in
  let rec remove src dst =
    if src >= l then
      if dst >= l then s else Bytes.sub s 0 dst
    else
      match Bytes.get s src with
        '_' -> remove (src + 1) dst
      |  c  -> Bytes.set s dst c; remove (src + 1) (dst + 1)
  in
  Bytes.to_string (remove 0 0)

(* Update the current location with file name and line number. *)

let update_loc lexbuf file line absolute chars =
  let pos = lexbuf.lex_curr_p in
  let new_file = match file with
    | None -> pos.pos_fname
    | Some s -> s
  in
  lexbuf.lex_curr_p <- {
    pos with
    pos_fname = new_file;
    pos_lnum = if absolute then line else pos.pos_lnum + line;
    pos_bol = pos.pos_cnum - chars;
  };
  lines_starts := (lexbuf.lex_curr_p.pos_lnum, lexbuf.lex_curr_p.pos_bol) :: !lines_starts;

;;

# 254 "src/approx_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\166\255\167\255\224\000\003\001\038\001\073\001\108\001\
    \190\255\143\001\180\001\091\000\217\001\090\000\071\000\046\000\
    \252\001\215\255\217\255\031\002\122\000\066\002\005\000\096\002\
    \063\003\092\000\240\255\173\002\082\003\155\003\118\004\081\005\
    \044\006\007\007\253\007\221\008\057\009\020\010\126\000\001\000\
    \255\255\228\010\189\011\141\012\102\013\210\009\054\014\251\255\
    \176\255\006\015\229\015\248\255\246\015\097\000\127\000\100\000\
    \063\016\026\017\239\255\238\255\234\255\230\003\132\000\101\000\
    \237\255\224\000\102\000\236\255\193\004\103\000\235\255\229\255\
    \245\017\245\018\148\019\169\255\210\019\210\020\113\021\168\255\
    \241\255\242\255\243\255\202\002\156\005\095\000\003\001\166\005\
    \119\006\064\003\105\003\183\002\253\003\109\000\233\255\151\021\
    \231\255\171\255\196\255\225\255\213\021\230\255\165\022\174\255\
    \132\023\167\023\038\024\197\024\003\025\226\255\121\007\029\004\
    \004\000\224\255\007\000\115\001\017\002\008\000\005\000\224\255\
    \226\025\172\255\005\026\040\026\075\026\212\255\208\255\209\255\
    \210\255\206\255\105\026\066\027\018\028\235\028\082\000\110\000\
    \199\255\200\255\201\255\187\255\084\000\184\255\193\255\202\029\
    \189\255\191\255\237\029\016\030\177\255\051\030\175\255\173\255\
    \170\255\052\004\251\255\252\255\006\000\253\255\081\030\088\000\
    \255\255\254\255\033\031\242\004\243\255\156\003\252\255\241\031\
    \199\001\203\001\254\255\111\000\114\000\255\255\072\000\239\001\
    \253\255\251\255\193\032\145\033\167\023\118\000\152\000\120\000\
    \250\255\249\255\248\255\091\007\216\004\121\000\247\255\248\004\
    \122\000\246\255\074\008\125\000\245\255\143\008\243\255\244\255\
    \009\000\245\255\241\004\252\255\040\000\126\000\126\000\255\255\
    \254\255\253\255\113\034\130\000\033\001\208\000\251\255\250\255\
    \249\255\095\010\148\006\219\000\248\255\219\007\220\000\247\255\
    \047\011\229\000\246\255\251\002\248\255\012\000\249\255\140\034\
    \255\255\250\255\008\012\229\007\253\255\028\001\038\001\109\008\
    \252\255\216\012\251\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\085\000\082\000\080\000\074\000\072\000\
    \255\255\063\000\060\000\053\000\052\000\048\000\044\000\042\000\
    \076\000\255\255\255\255\033\000\032\000\050\000\057\000\058\000\
    \039\000\037\000\255\255\010\000\010\000\009\000\008\000\008\000\
    \008\000\008\000\008\000\005\000\003\000\002\000\001\000\000\000\
    \255\255\255\255\008\000\255\255\008\000\079\000\255\255\255\255\
    \255\255\255\255\006\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\008\000\255\255\255\255\255\255\021\000\021\000\021\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \008\000\008\000\086\000\255\255\008\000\008\000\087\000\255\255\
    \255\255\255\255\255\255\255\255\011\000\255\255\255\255\255\255\
    \010\000\010\000\010\000\011\000\255\255\023\000\255\255\084\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\025\000\255\255\
    \081\000\051\000\081\000\029\000\255\255\255\255\255\255\255\255\
    \031\000\255\255\255\255\255\255\255\255\255\255\031\000\255\255\
    \034\000\255\255\083\000\077\000\041\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\048\000\255\255\048\000\069\000\067\000\
    \255\255\255\255\255\255\255\255\070\000\255\255\255\255\061\000\
    \255\255\255\255\078\000\073\000\255\255\075\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\002\000\255\255\004\000\004\000\
    \255\255\255\255\255\255\255\255\255\255\012\000\255\255\012\000\
    \011\000\011\000\255\255\012\000\012\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\002\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \010\000\255\255\012\000\255\255\012\000\012\000\012\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\006\000\255\255\007\000\
    \255\255\255\255\005\000\005\000\255\255\001\000\001\000\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\053\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\255\255\255\255\000\000\060\000\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\255\255\255\255\255\255\
    \000\000\255\255\255\255\000\000\255\255\255\255\000\000\000\000\
    \255\255\030\000\255\255\000\000\255\255\030\000\255\255\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \000\000\000\000\000\000\000\000\255\255\000\000\255\255\000\000\
    \255\255\255\255\108\000\255\255\255\255\000\000\114\000\255\255\
    \255\255\000\000\114\000\115\000\114\000\117\000\255\255\000\000\
    \255\255\000\000\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\255\255\000\000\000\000\255\255\
    \000\000\000\000\255\255\255\255\000\000\255\255\000\000\000\000\
    \000\000\154\000\000\000\000\000\255\255\000\000\255\255\255\255\
    \000\000\000\000\255\255\164\000\000\000\181\000\000\000\255\255\
    \255\255\255\255\000\000\255\255\255\255\000\000\255\255\255\255\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\255\255\255\255\255\255\000\000\255\255\
    \255\255\000\000\255\255\255\255\000\000\198\000\000\000\000\000\
    \255\255\000\000\211\000\000\000\255\255\255\255\255\255\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\000\000\000\000\
    \000\000\255\255\255\255\255\255\000\000\255\255\255\255\000\000\
    \255\255\255\255\000\000\228\000\000\000\255\255\000\000\233\000\
    \000\000\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \000\000\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\038\000\040\000\040\000\038\000\039\000\113\000\119\000\
    \157\000\113\000\119\000\201\000\112\000\118\000\230\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \038\000\007\000\026\000\020\000\005\000\003\000\019\000\034\000\
    \025\000\018\000\024\000\006\000\017\000\016\000\015\000\003\000\
    \028\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\014\000\013\000\021\000\012\000\009\000\035\000\
    \004\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\011\000\125\000\022\000\004\000\037\000\
    \029\000\031\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\032\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\033\000\030\000\
    \030\000\030\000\030\000\023\000\010\000\008\000\036\000\130\000\
    \135\000\128\000\071\000\111\000\127\000\126\000\093\000\038\000\
    \059\000\055\000\038\000\058\000\064\000\067\000\070\000\090\000\
    \090\000\094\000\140\000\139\000\141\000\129\000\160\000\137\000\
    \170\000\136\000\111\000\134\000\173\000\186\000\038\000\185\000\
    \190\000\193\000\183\000\176\000\196\000\209\000\058\000\208\000\
    \207\000\216\000\110\000\110\000\110\000\110\000\110\000\110\000\
    \110\000\110\000\110\000\110\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\176\000\185\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\138\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\215\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \002\000\003\000\220\000\223\000\003\000\003\000\003\000\255\255\
    \255\255\152\000\003\000\003\000\226\000\003\000\003\000\003\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\004\000\238\000\238\000\004\000\
    \004\000\004\000\000\000\213\000\151\000\004\000\004\000\238\000\
    \004\000\004\000\004\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\000\000\238\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\000\000\000\000\238\000\005\000\
    \215\000\000\000\005\000\005\000\005\000\000\000\000\000\150\000\
    \005\000\005\000\000\000\005\000\005\000\005\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\000\000\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\000\000\
    \000\000\000\000\122\000\000\000\000\000\122\000\122\000\122\000\
    \000\000\000\000\121\000\122\000\122\000\000\000\122\000\149\000\
    \122\000\000\000\000\000\000\000\000\000\113\000\000\000\004\000\
    \112\000\004\000\000\000\122\000\005\000\122\000\122\000\122\000\
    \122\000\122\000\000\000\000\000\000\000\146\000\000\000\000\000\
    \146\000\146\000\146\000\000\000\000\000\117\000\146\000\146\000\
    \000\000\146\000\146\000\146\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\005\000\000\000\005\000\000\000\146\000\122\000\
    \146\000\147\000\146\000\146\000\146\000\000\000\000\000\000\000\
    \104\000\000\000\000\000\104\000\104\000\104\000\000\000\000\000\
    \103\000\104\000\104\000\000\000\104\000\104\000\104\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\122\000\000\000\122\000\
    \000\000\104\000\146\000\104\000\104\000\104\000\104\000\104\000\
    \175\000\169\000\000\000\175\000\175\000\104\000\000\000\175\000\
    \104\000\104\000\104\000\000\000\000\000\103\000\104\000\104\000\
    \000\000\104\000\104\000\104\000\000\000\000\000\000\000\175\000\
    \146\000\000\000\146\000\175\000\145\000\104\000\104\000\000\000\
    \104\000\104\000\104\000\104\000\104\000\000\000\000\000\000\000\
    \175\000\000\000\104\000\175\000\000\000\104\000\104\000\104\000\
    \000\000\000\000\103\000\104\000\104\000\000\000\104\000\104\000\
    \104\000\000\000\000\000\104\000\144\000\104\000\000\000\175\000\
    \000\000\142\000\104\000\104\000\000\000\104\000\104\000\104\000\
    \104\000\104\000\116\000\113\000\000\000\122\000\112\000\000\000\
    \122\000\122\000\122\000\000\000\000\000\121\000\122\000\122\000\
    \000\000\122\000\123\000\122\000\000\000\000\000\000\000\000\000\
    \143\000\116\000\104\000\115\000\000\000\000\000\122\000\104\000\
    \122\000\122\000\124\000\122\000\122\000\000\000\000\000\000\000\
    \104\000\000\000\174\000\104\000\104\000\120\000\174\000\000\000\
    \103\000\104\000\104\000\000\000\104\000\104\000\104\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\104\000\000\000\104\000\
    \000\000\104\000\122\000\104\000\104\000\104\000\104\000\104\000\
    \000\000\000\000\000\000\104\000\000\000\000\000\104\000\104\000\
    \104\000\000\000\174\000\103\000\104\000\104\000\000\000\105\000\
    \104\000\104\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \122\000\000\000\122\000\000\000\106\000\104\000\104\000\104\000\
    \104\000\104\000\104\000\000\000\000\000\000\000\000\000\100\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\000\000\104\000\098\000\104\000\000\000\000\000\
    \104\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\101\000\000\000\000\000\104\000\100\000\
    \104\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\102\000\100\000\
    \100\000\100\000\100\000\084\000\099\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\000\000\083\000\000\000\000\000\092\000\000\000\092\000\
    \000\000\081\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\000\000\230\000\000\000\000\000\
    \229\000\000\000\000\000\000\000\027\000\000\000\000\000\000\000\
    \000\000\255\255\083\000\000\000\000\000\000\000\091\000\000\000\
    \000\000\082\000\000\000\080\000\000\000\232\000\000\000\000\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\000\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\231\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \003\000\000\000\000\000\003\000\003\000\003\000\000\000\000\000\
    \096\000\095\000\003\000\000\000\003\000\003\000\003\000\000\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \084\000\000\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\081\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\085\000\000\000\000\000\083\000\
    \000\000\090\000\090\000\000\000\000\000\003\000\081\000\089\000\
    \000\000\086\000\000\000\000\000\000\000\000\000\183\000\000\000\
    \000\000\182\000\087\000\000\000\082\000\000\000\080\000\000\000\
    \000\000\027\000\000\000\000\000\085\000\081\000\000\000\083\000\
    \000\000\000\000\000\000\003\000\000\000\003\000\082\000\000\000\
    \080\000\086\000\029\000\184\000\000\000\000\000\000\000\000\000\
    \090\000\000\000\087\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\082\000\000\000\080\000\
    \000\000\000\000\000\000\000\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\000\000\000\000\
    \180\000\000\000\029\000\232\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\111\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\111\000\157\000\000\000\
    \000\000\156\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\110\000\110\000\110\000\
    \110\000\110\000\110\000\110\000\110\000\110\000\110\000\000\000\
    \000\000\000\000\000\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\159\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\000\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\041\000\255\255\030\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \158\000\000\000\000\000\000\000\000\000\000\000\000\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\000\000\000\000\000\000\000\000\030\000\000\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\213\000\169\000\000\000\212\000\168\000\
    \000\000\000\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\000\000\000\000\166\000\000\000\000\000\000\000\
    \214\000\165\000\172\000\000\000\171\000\000\000\000\000\000\000\
    \000\000\000\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\000\000\000\000\155\000\000\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\210\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\167\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\041\000\000\000\
    \030\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\000\000\000\000\000\000\000\000\
    \030\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\076\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \000\000\083\000\000\000\000\000\000\000\000\000\000\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\000\000\000\000\000\000\
    \000\000\255\255\170\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\084\000\000\000\000\000\000\000\000\000\
    \000\000\083\000\000\000\000\000\000\000\000\000\000\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\000\000\000\000\000\000\
    \000\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \000\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \000\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\041\000\000\000\030\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\000\000\
    \000\000\000\000\000\000\030\000\000\000\030\000\030\000\030\000\
    \030\000\072\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\000\000\000\000\
    \000\000\000\000\000\000\081\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\088\000\000\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\000\000\000\000\
    \000\000\000\000\000\000\082\000\000\000\080\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\041\000\000\000\030\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\000\000\000\000\000\000\000\000\030\000\000\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\116\000\113\000\071\000\000\000\112\000\000\000\
    \000\000\000\000\000\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\194\000\000\000\000\000\000\000\
    \000\000\116\000\000\000\115\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\110\000\110\000\110\000\110\000\110\000\110\000\110\000\
    \110\000\110\000\110\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\000\000\000\000\000\000\000\000\000\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\000\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\000\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\055\000\
    \000\000\000\000\054\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\239\000\239\000\239\000\
    \239\000\239\000\239\000\239\000\239\000\239\000\239\000\000\000\
    \000\000\000\000\056\000\000\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \000\000\052\000\000\000\000\000\057\000\000\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \000\000\255\255\195\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\195\000\195\000\195\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\201\000\000\000\000\000\200\000\240\000\240\000\240\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\000\000\
    \000\000\000\000\000\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\203\000\000\000\000\000\000\000\000\000\202\000\206\000\
    \000\000\205\000\000\000\000\000\000\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\000\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\000\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\255\255\045\000\000\000\
    \000\000\045\000\045\000\045\000\049\000\204\000\000\000\045\000\
    \045\000\000\000\045\000\045\000\045\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\045\000\
    \000\000\045\000\045\000\045\000\050\000\045\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\045\000\049\000\000\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \000\000\045\000\045\000\045\000\000\000\045\000\045\000\045\000\
    \046\000\000\000\000\000\045\000\045\000\000\000\045\000\045\000\
    \045\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\045\000\000\000\045\000\045\000\045\000\
    \045\000\045\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\199\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\045\000\
    \046\000\000\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\000\000\045\000\000\000\045\000\
    \000\000\000\000\000\000\000\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\000\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\045\000\000\000\000\000\045\000\045\000\
    \045\000\000\000\000\000\048\000\045\000\045\000\000\000\045\000\
    \045\000\045\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\045\000\000\000\045\000\045\000\
    \045\000\045\000\045\000\000\000\000\000\000\000\000\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \045\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\041\000\000\000\030\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\045\000\000\000\
    \045\000\000\000\000\000\000\000\000\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\000\000\
    \000\000\000\000\000\000\030\000\000\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\224\000\
    \224\000\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \224\000\224\000\224\000\224\000\224\000\224\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \224\000\224\000\224\000\224\000\224\000\224\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\042\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\000\000\
    \000\000\000\000\000\000\042\000\000\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\225\000\
    \225\000\225\000\225\000\225\000\225\000\225\000\225\000\225\000\
    \225\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \225\000\225\000\225\000\225\000\225\000\225\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \225\000\225\000\225\000\225\000\225\000\225\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\000\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\000\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\043\000\000\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \000\000\000\000\000\000\000\000\042\000\000\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\241\000\241\000\241\000\241\000\241\000\241\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\241\000\241\000\241\000\241\000\241\000\241\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\000\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\044\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \000\000\000\000\000\000\000\000\044\000\000\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \242\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \242\000\242\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\242\000\242\000\242\000\242\000\242\000\242\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\242\000\242\000\242\000\242\000\242\000\242\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\000\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\000\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\043\000\000\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\000\000\000\000\000\000\000\000\044\000\000\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\000\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\046\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \047\000\000\000\000\000\000\000\000\000\000\000\000\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\000\000\000\000\000\000\000\000\046\000\000\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\000\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\049\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \051\000\000\000\000\000\000\000\000\000\000\000\000\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\000\000\000\000\000\000\000\000\049\000\000\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\000\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\000\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\045\000\000\000\
    \000\000\045\000\045\000\045\000\000\000\000\000\048\000\045\000\
    \045\000\000\000\045\000\045\000\045\000\000\000\063\000\000\000\
    \063\000\000\000\000\000\000\000\000\000\063\000\000\000\045\000\
    \000\000\045\000\045\000\045\000\045\000\045\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\045\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\063\000\000\000\000\000\000\000\000\000\000\000\
    \063\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\045\000\000\000\045\000\063\000\000\000\042\000\000\000\
    \063\000\000\000\063\000\000\000\000\000\000\000\061\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\000\000\000\000\000\000\000\000\042\000\000\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\000\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\000\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\041\000\
    \000\000\030\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \000\000\030\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\041\000\000\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \000\000\000\000\000\000\000\000\030\000\000\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\073\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\000\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\000\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\074\000\041\000\074\000\000\000\255\255\255\255\074\000\
    \074\000\255\255\074\000\255\255\074\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\074\000\074\000\074\000\255\255\074\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\074\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\074\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\074\000\000\000\000\000\
    \074\000\074\000\074\000\000\000\000\000\075\000\074\000\074\000\
    \000\000\074\000\074\000\074\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\074\000\000\000\
    \074\000\074\000\074\000\074\000\074\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\000\000\074\000\000\000\000\000\255\255\000\000\041\000\
    \000\000\030\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\000\000\000\000\000\000\000\000\
    \074\000\000\000\074\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \000\000\030\000\000\000\030\000\030\000\030\000\077\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\078\000\041\000\
    \078\000\000\000\255\255\255\255\078\000\078\000\255\255\078\000\
    \255\255\078\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\078\000\078\000\
    \078\000\255\255\078\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \078\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\078\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\078\000\000\000\000\000\078\000\078\000\078\000\
    \000\000\000\000\079\000\078\000\078\000\000\000\078\000\078\000\
    \078\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\078\000\000\000\078\000\078\000\078\000\
    \078\000\078\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \095\000\000\000\000\000\095\000\095\000\095\000\000\000\000\000\
    \097\000\095\000\095\000\000\000\095\000\095\000\095\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\078\000\
    \000\000\095\000\255\255\095\000\095\000\095\000\095\000\095\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\078\000\000\000\078\000\
    \000\000\000\000\000\000\000\000\000\000\095\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\100\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\000\000\
    \000\000\000\000\000\000\095\000\000\000\095\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \000\000\000\000\000\000\000\000\100\000\000\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \000\000\099\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\000\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \000\000\000\000\000\000\000\000\100\000\000\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \000\000\099\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\000\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\000\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\104\000\000\000\000\000\
    \104\000\104\000\104\000\000\000\000\000\103\000\104\000\104\000\
    \000\000\104\000\104\000\104\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\104\000\000\000\
    \104\000\104\000\104\000\104\000\104\000\000\000\000\000\189\000\
    \104\000\189\000\000\000\104\000\104\000\104\000\189\000\000\000\
    \103\000\104\000\104\000\000\000\104\000\104\000\104\000\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\104\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \104\000\000\000\104\000\189\000\000\000\104\000\000\000\000\000\
    \000\000\189\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\189\000\000\000\000\000\
    \000\000\189\000\000\000\189\000\000\000\000\000\000\000\187\000\
    \000\000\000\000\000\000\104\000\000\000\104\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\104\000\
    \255\255\255\255\104\000\104\000\104\000\000\000\255\255\103\000\
    \104\000\104\000\255\255\104\000\104\000\104\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \104\000\255\255\107\000\104\000\104\000\104\000\104\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\104\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\104\000\255\255\104\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\104\000\000\000\
    \000\000\104\000\104\000\104\000\000\000\000\000\103\000\104\000\
    \104\000\000\000\104\000\104\000\104\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\104\000\
    \000\000\104\000\104\000\104\000\104\000\104\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\104\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\108\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\000\000\000\000\109\000\
    \000\000\104\000\000\000\104\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\000\000\000\000\
    \000\000\000\000\108\000\000\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\000\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\000\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\104\000\000\000\000\000\104\000\104\000\
    \104\000\000\000\000\000\103\000\104\000\104\000\000\000\104\000\
    \104\000\104\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\104\000\000\000\104\000\104\000\
    \104\000\104\000\104\000\000\000\000\000\000\000\122\000\000\000\
    \000\000\122\000\122\000\122\000\000\000\000\000\121\000\122\000\
    \122\000\000\000\122\000\122\000\122\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\122\000\
    \104\000\122\000\122\000\122\000\122\000\122\000\000\000\000\000\
    \000\000\122\000\000\000\000\000\122\000\122\000\122\000\000\000\
    \000\000\121\000\122\000\122\000\000\000\122\000\122\000\122\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\104\000\000\000\
    \104\000\000\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\000\000\000\000\000\000\122\000\000\000\000\000\122\000\
    \122\000\122\000\000\000\000\000\121\000\122\000\122\000\000\000\
    \122\000\122\000\122\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\122\000\000\000\122\000\000\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\000\000\000\000\000\000\000\000\
    \131\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\000\000\122\000\000\000\122\000\000\000\
    \000\000\122\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\000\000\000\000\000\000\122\000\
    \131\000\122\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \000\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \000\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \132\000\000\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\000\000\000\000\000\000\
    \000\000\131\000\000\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\000\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\133\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\000\000\000\000\000\000\
    \000\000\133\000\000\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\000\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\000\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\132\000\000\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\000\000\000\000\
    \000\000\000\000\133\000\000\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\000\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\000\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\104\000\000\000\000\000\104\000\104\000\
    \104\000\000\000\000\000\103\000\104\000\104\000\000\000\104\000\
    \104\000\104\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\104\000\000\000\104\000\104\000\
    \104\000\104\000\104\000\000\000\000\000\000\000\146\000\000\000\
    \000\000\146\000\146\000\146\000\000\000\000\000\148\000\146\000\
    \146\000\000\000\146\000\146\000\146\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\146\000\
    \104\000\146\000\146\000\146\000\146\000\146\000\000\000\000\000\
    \000\000\146\000\000\000\000\000\146\000\146\000\146\000\000\000\
    \000\000\148\000\146\000\146\000\000\000\146\000\146\000\146\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\104\000\000\000\
    \104\000\000\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\000\000\000\000\000\000\122\000\000\000\000\000\122\000\
    \122\000\122\000\000\000\000\000\121\000\122\000\122\000\000\000\
    \122\000\122\000\122\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\146\000\000\000\146\000\000\000\122\000\146\000\122\000\
    \122\000\122\000\122\000\122\000\000\000\000\000\000\000\000\000\
    \162\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\000\000\146\000\000\000\146\000\000\000\
    \000\000\122\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\000\000\000\000\000\000\122\000\
    \162\000\122\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\000\000\000\000\161\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \000\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\000\000\000\000\000\000\000\000\
    \162\000\000\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\000\000\000\000\161\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \000\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \178\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\176\000\000\000\000\000\000\000\
    \178\000\000\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\179\000\
    \178\000\178\000\178\000\178\000\000\000\177\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \000\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\000\000\000\000\000\000\000\000\
    \178\000\000\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\000\000\177\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \000\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\000\000\000\000\000\000\000\000\
    \178\000\000\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\000\000\177\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \000\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \000\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\219\000\000\000\219\000\000\000\000\000\238\000\000\000\
    \219\000\237\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\000\000\236\000\000\000\236\000\000\000\
    \000\000\000\000\000\000\236\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\235\000\235\000\235\000\235\000\
    \235\000\235\000\235\000\235\000\235\000\235\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\219\000\000\000\000\000\
    \000\000\000\000\000\000\219\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\219\000\
    \000\000\000\000\000\000\219\000\000\000\219\000\000\000\000\000\
    \236\000\217\000\000\000\000\000\000\000\000\000\236\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\236\000\000\000\000\000\000\000\236\000\000\000\
    \236\000\000\000\000\000\000\000\234\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\039\000\000\000\000\000\112\000\118\000\
    \156\000\114\000\117\000\200\000\114\000\117\000\229\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\015\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\013\000\
    \011\000\014\000\022\000\020\000\014\000\014\000\025\000\038\000\
    \053\000\054\000\038\000\055\000\063\000\066\000\069\000\085\000\
    \085\000\093\000\134\000\135\000\140\000\013\000\159\000\011\000\
    \171\000\011\000\020\000\011\000\172\000\181\000\038\000\183\000\
    \189\000\192\000\182\000\174\000\195\000\204\000\054\000\205\000\
    \206\000\211\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\174\000\182\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\011\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\213\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\219\000\222\000\003\000\003\000\003\000\114\000\
    \117\000\003\000\003\000\003\000\225\000\003\000\003\000\003\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\003\000\255\255\003\000\003\000\003\000\003\000\
    \003\000\255\255\255\255\255\255\004\000\237\000\237\000\004\000\
    \004\000\004\000\255\255\212\000\004\000\004\000\004\000\238\000\
    \004\000\004\000\004\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\255\255\237\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\255\255\255\255\238\000\005\000\
    \212\000\255\255\005\000\005\000\005\000\255\255\255\255\005\000\
    \005\000\005\000\255\255\005\000\005\000\005\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\003\000\255\255\003\000\255\255\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\255\255\
    \255\255\255\255\006\000\255\255\255\255\006\000\006\000\006\000\
    \255\255\255\255\006\000\006\000\006\000\255\255\006\000\006\000\
    \006\000\255\255\255\255\255\255\255\255\115\000\255\255\004\000\
    \115\000\004\000\255\255\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\255\255\255\255\255\255\007\000\255\255\255\255\
    \007\000\007\000\007\000\255\255\255\255\115\000\007\000\007\000\
    \255\255\007\000\007\000\007\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\005\000\255\255\005\000\255\255\007\000\006\000\
    \007\000\007\000\007\000\007\000\007\000\255\255\255\255\255\255\
    \009\000\255\255\255\255\009\000\009\000\009\000\255\255\255\255\
    \009\000\009\000\009\000\255\255\009\000\009\000\009\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\006\000\255\255\006\000\
    \255\255\009\000\007\000\009\000\009\000\009\000\009\000\009\000\
    \168\000\168\000\255\255\168\000\169\000\010\000\255\255\169\000\
    \010\000\010\000\010\000\255\255\255\255\010\000\010\000\010\000\
    \255\255\010\000\010\000\010\000\255\255\255\255\255\255\168\000\
    \007\000\255\255\007\000\169\000\009\000\009\000\010\000\255\255\
    \010\000\010\000\010\000\010\000\010\000\255\255\255\255\255\255\
    \175\000\255\255\012\000\175\000\255\255\012\000\012\000\012\000\
    \255\255\255\255\012\000\012\000\012\000\255\255\012\000\012\000\
    \012\000\255\255\255\255\009\000\009\000\009\000\255\255\175\000\
    \255\255\010\000\010\000\012\000\255\255\012\000\012\000\012\000\
    \012\000\012\000\116\000\116\000\255\255\016\000\116\000\255\255\
    \016\000\016\000\016\000\255\255\255\255\016\000\016\000\016\000\
    \255\255\016\000\016\000\016\000\255\255\255\255\255\255\255\255\
    \010\000\116\000\010\000\116\000\255\255\255\255\016\000\012\000\
    \016\000\016\000\016\000\016\000\016\000\255\255\255\255\255\255\
    \019\000\255\255\168\000\019\000\019\000\019\000\169\000\255\255\
    \019\000\019\000\019\000\255\255\019\000\019\000\019\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\012\000\255\255\012\000\
    \255\255\019\000\016\000\019\000\019\000\019\000\019\000\019\000\
    \255\255\255\255\255\255\021\000\255\255\255\255\021\000\021\000\
    \021\000\255\255\175\000\021\000\021\000\021\000\255\255\021\000\
    \021\000\021\000\255\255\115\000\255\255\255\255\255\255\255\255\
    \016\000\255\255\016\000\255\255\021\000\019\000\021\000\021\000\
    \021\000\021\000\021\000\255\255\255\255\255\255\255\255\023\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\255\255\019\000\023\000\019\000\255\255\255\255\
    \021\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\255\255\255\255\021\000\023\000\
    \021\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\027\000\023\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\255\255\027\000\255\255\255\255\083\000\255\255\083\000\
    \255\255\027\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\255\255\227\000\255\255\255\255\
    \227\000\255\255\255\255\255\255\027\000\255\255\255\255\255\255\
    \255\255\116\000\027\000\255\255\255\255\255\255\091\000\255\255\
    \255\255\027\000\255\255\027\000\255\255\227\000\255\255\255\255\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\255\255\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\227\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \024\000\255\255\255\255\024\000\024\000\024\000\255\255\255\255\
    \024\000\024\000\024\000\255\255\024\000\024\000\024\000\255\255\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \255\255\024\000\255\255\024\000\024\000\024\000\024\000\024\000\
    \028\000\255\255\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\089\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\028\000\255\255\255\255\028\000\
    \255\255\090\000\090\000\255\255\255\255\024\000\028\000\089\000\
    \255\255\028\000\255\255\255\255\255\255\255\255\165\000\255\255\
    \255\255\165\000\028\000\255\255\089\000\255\255\089\000\255\255\
    \255\255\028\000\255\255\255\255\028\000\090\000\255\255\028\000\
    \255\255\255\255\255\255\024\000\255\255\024\000\028\000\255\255\
    \028\000\028\000\029\000\165\000\255\255\255\255\255\255\255\255\
    \090\000\255\255\028\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\090\000\255\255\090\000\
    \255\255\255\255\255\255\255\255\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\255\255\255\255\
    \165\000\255\255\029\000\227\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\111\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\111\000\153\000\255\255\
    \255\255\153\000\255\255\255\255\255\255\255\255\255\255\061\000\
    \061\000\061\000\061\000\061\000\061\000\111\000\111\000\111\000\
    \111\000\111\000\111\000\111\000\111\000\111\000\111\000\255\255\
    \255\255\255\255\255\255\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\153\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\255\255\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\030\000\165\000\030\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \153\000\255\255\255\255\255\255\255\255\255\255\255\255\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\255\255\255\255\255\255\255\255\030\000\255\255\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\202\000\163\000\255\255\202\000\163\000\
    \255\255\255\255\068\000\068\000\068\000\068\000\068\000\068\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\255\255\255\255\163\000\255\255\255\255\255\255\
    \202\000\163\000\163\000\255\255\163\000\255\255\255\255\255\255\
    \255\255\255\255\068\000\068\000\068\000\068\000\068\000\068\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\255\255\255\255\153\000\255\255\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\202\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\163\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\031\000\255\255\
    \031\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\255\255\255\255\255\255\255\255\
    \031\000\255\255\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\087\000\087\000\
    \087\000\087\000\087\000\087\000\087\000\087\000\087\000\087\000\
    \255\255\084\000\255\255\255\255\255\255\255\255\255\255\087\000\
    \087\000\087\000\087\000\087\000\087\000\255\255\255\255\255\255\
    \255\255\202\000\163\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\084\000\255\255\255\255\255\255\255\255\
    \255\255\084\000\255\255\255\255\255\255\255\255\255\255\087\000\
    \087\000\087\000\087\000\087\000\087\000\255\255\255\255\255\255\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\032\000\255\255\032\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\255\255\
    \255\255\255\255\255\255\032\000\255\255\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \088\000\088\000\088\000\088\000\088\000\088\000\255\255\255\255\
    \255\255\255\255\255\255\088\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\088\000\255\255\
    \088\000\088\000\088\000\088\000\088\000\088\000\255\255\255\255\
    \255\255\255\255\255\255\088\000\255\255\088\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\033\000\255\255\033\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\255\255\255\255\255\255\255\255\033\000\255\255\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\110\000\110\000\033\000\255\255\110\000\255\255\
    \255\255\255\255\255\255\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\255\255\255\255\255\255\
    \255\255\110\000\255\255\110\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\110\000\110\000\110\000\110\000\110\000\110\000\110\000\
    \110\000\110\000\110\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\187\000\187\000\187\000\187\000\
    \187\000\187\000\255\255\255\255\255\255\255\255\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\034\000\
    \255\255\255\255\034\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\235\000\235\000\235\000\
    \235\000\235\000\235\000\235\000\235\000\235\000\235\000\255\255\
    \255\255\255\255\034\000\255\255\034\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \255\255\034\000\255\255\255\255\034\000\255\255\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \255\255\110\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\194\000\194\000\194\000\194\000\194\000\
    \194\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\197\000\255\255\255\255\197\000\239\000\239\000\239\000\
    \239\000\239\000\239\000\239\000\239\000\239\000\239\000\255\255\
    \255\255\255\255\255\255\194\000\194\000\194\000\194\000\194\000\
    \194\000\197\000\255\255\255\255\255\255\255\255\197\000\197\000\
    \255\255\197\000\255\255\255\255\255\255\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\255\255\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\255\255\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\035\000\255\255\
    \255\255\035\000\035\000\035\000\035\000\197\000\255\255\035\000\
    \035\000\255\255\035\000\035\000\035\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\035\000\
    \255\255\035\000\035\000\035\000\035\000\035\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\035\000\035\000\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \255\255\035\000\036\000\035\000\255\255\036\000\036\000\036\000\
    \036\000\255\255\255\255\036\000\036\000\255\255\036\000\036\000\
    \036\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\036\000\255\255\036\000\036\000\036\000\
    \036\000\036\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\197\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\036\000\
    \036\000\255\255\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\255\255\036\000\255\255\036\000\
    \255\255\255\255\255\255\255\255\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\045\000\255\255\255\255\045\000\045\000\
    \045\000\255\255\255\255\045\000\045\000\045\000\255\255\045\000\
    \045\000\045\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\045\000\255\255\045\000\045\000\
    \045\000\045\000\045\000\255\255\255\255\255\255\255\255\255\255\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \045\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\037\000\255\255\037\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\045\000\255\255\
    \045\000\255\255\255\255\255\255\255\255\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\255\255\
    \255\255\255\255\255\255\037\000\255\255\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \217\000\217\000\217\000\217\000\217\000\217\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \217\000\217\000\217\000\217\000\217\000\217\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\255\255\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\041\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\255\255\
    \255\255\255\255\255\255\041\000\255\255\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\224\000\
    \224\000\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \224\000\224\000\224\000\224\000\224\000\224\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \224\000\224\000\224\000\224\000\224\000\224\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\255\255\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\255\255\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\042\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\042\000\255\255\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \255\255\255\255\255\255\255\255\042\000\255\255\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \234\000\234\000\234\000\234\000\234\000\234\000\234\000\234\000\
    \234\000\234\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\234\000\234\000\234\000\234\000\234\000\234\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\234\000\234\000\234\000\234\000\234\000\234\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\255\255\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\043\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \255\255\255\255\255\255\255\255\043\000\255\255\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\241\000\
    \241\000\241\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\241\000\241\000\241\000\241\000\241\000\241\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\241\000\241\000\241\000\241\000\241\000\241\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\255\255\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\255\255\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\044\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\044\000\255\255\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\255\255\255\255\255\255\255\255\044\000\255\255\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\255\255\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\046\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\255\255\255\255\255\255\255\255\255\255\255\255\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\255\255\255\255\255\255\255\255\046\000\255\255\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\255\255\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\049\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\255\255\255\255\255\255\255\255\255\255\255\255\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\255\255\255\255\255\255\255\255\049\000\255\255\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\255\255\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\255\255\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\050\000\255\255\
    \255\255\050\000\050\000\050\000\255\255\255\255\050\000\050\000\
    \050\000\255\255\050\000\050\000\050\000\255\255\052\000\255\255\
    \052\000\255\255\255\255\255\255\255\255\052\000\255\255\050\000\
    \255\255\050\000\050\000\050\000\050\000\050\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\050\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\052\000\255\255\255\255\255\255\255\255\255\255\
    \052\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\050\000\255\255\050\000\052\000\255\255\056\000\255\255\
    \052\000\255\255\052\000\255\255\255\255\255\255\052\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\255\255\255\255\255\255\255\255\056\000\255\255\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\052\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\255\255\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\255\255\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\057\000\
    \255\255\057\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\255\255\255\255\255\255\
    \255\255\057\000\255\255\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\255\255\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\255\255\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\072\000\255\255\072\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \255\255\255\255\255\255\255\255\072\000\255\255\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\255\255\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\255\255\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\255\255\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \073\000\073\000\073\000\073\000\255\255\073\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\074\000\255\255\255\255\
    \074\000\074\000\074\000\255\255\255\255\074\000\074\000\074\000\
    \255\255\074\000\074\000\074\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\073\000\255\255\074\000\255\255\
    \074\000\074\000\074\000\074\000\074\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\073\000\255\255\255\255\255\255\
    \255\255\255\255\074\000\255\255\255\255\073\000\255\255\076\000\
    \255\255\076\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\255\255\255\255\255\255\255\255\
    \074\000\255\255\074\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\255\255\255\255\255\255\
    \255\255\076\000\255\255\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\255\255\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\255\255\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\255\255\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\077\000\077\000\077\000\
    \077\000\255\255\077\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\078\000\255\255\255\255\078\000\078\000\078\000\
    \255\255\255\255\078\000\078\000\078\000\255\255\078\000\078\000\
    \078\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\077\000\255\255\078\000\255\255\078\000\078\000\078\000\
    \078\000\078\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \095\000\255\255\255\255\095\000\095\000\095\000\255\255\255\255\
    \095\000\095\000\095\000\255\255\095\000\095\000\095\000\255\255\
    \255\255\077\000\255\255\255\255\255\255\255\255\255\255\078\000\
    \255\255\095\000\077\000\095\000\095\000\095\000\095\000\095\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\078\000\255\255\078\000\
    \255\255\255\255\255\255\255\255\255\255\095\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\100\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\255\255\
    \255\255\255\255\255\255\095\000\255\255\095\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \255\255\255\255\255\255\255\255\100\000\255\255\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \255\255\100\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\255\255\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\102\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \255\255\255\255\255\255\255\255\102\000\255\255\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \255\255\102\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\255\255\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\255\255\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\104\000\255\255\255\255\
    \104\000\104\000\104\000\255\255\255\255\104\000\104\000\104\000\
    \255\255\104\000\104\000\104\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\104\000\255\255\
    \104\000\104\000\104\000\104\000\104\000\255\255\255\255\180\000\
    \105\000\180\000\255\255\105\000\105\000\105\000\180\000\255\255\
    \105\000\105\000\105\000\255\255\105\000\105\000\105\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\105\000\104\000\105\000\105\000\105\000\105\000\105\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \104\000\255\255\104\000\180\000\255\255\105\000\255\255\255\255\
    \255\255\180\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\180\000\255\255\255\255\
    \255\255\180\000\255\255\180\000\255\255\255\255\255\255\180\000\
    \255\255\255\255\255\255\105\000\255\255\105\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\255\255\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\106\000\106\000\106\000\106\000\255\255\106\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\107\000\255\255\
    \255\255\107\000\107\000\107\000\255\255\255\255\107\000\107\000\
    \107\000\255\255\107\000\107\000\107\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\106\000\255\255\107\000\
    \255\255\107\000\107\000\107\000\107\000\107\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\106\000\255\255\255\255\
    \255\255\255\255\255\255\107\000\255\255\255\255\106\000\255\255\
    \255\255\255\255\108\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\255\255\255\255\108\000\
    \255\255\107\000\255\255\107\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\255\255\255\255\
    \255\255\255\255\108\000\255\255\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\255\255\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\255\255\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\120\000\255\255\255\255\120\000\120\000\
    \120\000\255\255\255\255\120\000\120\000\120\000\255\255\120\000\
    \120\000\120\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\120\000\255\255\120\000\120\000\
    \120\000\120\000\120\000\255\255\255\255\255\255\122\000\255\255\
    \255\255\122\000\122\000\122\000\255\255\255\255\122\000\122\000\
    \122\000\255\255\122\000\122\000\122\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\122\000\
    \120\000\122\000\122\000\122\000\122\000\122\000\255\255\255\255\
    \255\255\123\000\255\255\255\255\123\000\123\000\123\000\255\255\
    \255\255\123\000\123\000\123\000\255\255\123\000\123\000\123\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\120\000\255\255\
    \120\000\255\255\123\000\122\000\123\000\123\000\123\000\123\000\
    \123\000\255\255\255\255\255\255\124\000\255\255\255\255\124\000\
    \124\000\124\000\255\255\255\255\124\000\124\000\124\000\255\255\
    \124\000\124\000\124\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\122\000\255\255\122\000\255\255\124\000\123\000\124\000\
    \124\000\124\000\124\000\124\000\255\255\255\255\255\255\255\255\
    \130\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\255\255\123\000\255\255\123\000\255\255\
    \255\255\124\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\255\255\255\255\255\255\124\000\
    \130\000\124\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \255\255\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \255\255\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\131\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \131\000\255\255\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\255\255\255\255\255\255\
    \255\255\131\000\255\255\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\255\255\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\132\000\131\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\255\255\255\255\255\255\
    \255\255\132\000\255\255\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\255\255\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\255\255\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\133\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\133\000\255\255\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\255\255\255\255\
    \255\255\255\255\133\000\255\255\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\255\255\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\255\255\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\143\000\255\255\255\255\143\000\143\000\
    \143\000\255\255\255\255\143\000\143\000\143\000\255\255\143\000\
    \143\000\143\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\143\000\255\255\143\000\143\000\
    \143\000\143\000\143\000\255\255\255\255\255\255\146\000\255\255\
    \255\255\146\000\146\000\146\000\255\255\255\255\146\000\146\000\
    \146\000\255\255\146\000\146\000\146\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\146\000\
    \143\000\146\000\146\000\146\000\146\000\146\000\255\255\255\255\
    \255\255\147\000\255\255\255\255\147\000\147\000\147\000\255\255\
    \255\255\147\000\147\000\147\000\255\255\147\000\147\000\147\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\143\000\255\255\
    \143\000\255\255\147\000\146\000\147\000\147\000\147\000\147\000\
    \147\000\255\255\255\255\255\255\149\000\255\255\255\255\149\000\
    \149\000\149\000\255\255\255\255\149\000\149\000\149\000\255\255\
    \149\000\149\000\149\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\146\000\255\255\146\000\255\255\149\000\147\000\149\000\
    \149\000\149\000\149\000\149\000\255\255\255\255\255\255\255\255\
    \158\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\255\255\147\000\255\255\147\000\255\255\
    \255\255\149\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\255\255\255\255\255\255\149\000\
    \158\000\149\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\255\255\255\255\158\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \255\255\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \162\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\255\255\255\255\255\255\255\255\
    \162\000\255\255\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\255\255\255\255\162\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \255\255\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \167\000\162\000\162\000\162\000\162\000\162\000\162\000\162\000\
    \162\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\255\255\255\255\255\255\
    \167\000\255\255\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\255\255\167\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \255\255\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \178\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\255\255\255\255\255\255\255\255\
    \178\000\255\255\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\255\255\178\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \255\255\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \179\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\255\255\255\255\255\255\255\255\
    \179\000\255\255\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\255\255\179\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \255\255\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \255\255\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\210\000\255\255\210\000\255\255\255\255\231\000\255\255\
    \210\000\231\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\255\255\231\000\255\255\231\000\255\255\
    \255\255\255\255\255\255\231\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\231\000\231\000\231\000\231\000\
    \231\000\231\000\231\000\231\000\231\000\231\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\210\000\255\255\255\255\
    \255\255\255\255\255\255\210\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\210\000\
    \255\255\255\255\255\255\210\000\255\255\210\000\255\255\255\255\
    \231\000\210\000\255\255\255\255\255\255\255\255\231\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\231\000\255\255\255\255\255\255\231\000\255\255\
    \231\000\255\255\255\255\255\255\231\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\231\000";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\036\000\
    \000\000\012\000\000\000\000\000\002\000\000\000\000\000\027\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\000\
    \000\000\000\000\000\000\000\000\000\000\002\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \012\000\000\000\000\000\000\000\000\000\000\000\027\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\039\000\039\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\019\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\036\000\036\000\000\000\036\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\001\000\022\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\001\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\001\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\020\000\115\000\231\000\237\000\115\000\231\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \020\000\255\255\115\000\000\000\116\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\110\000\111\000\255\255\255\255\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\110\000\110\000\110\000\110\000\110\000\110\000\
    \110\000\110\000\110\000\110\000\111\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\111\000\111\000\111\000\111\000\
    \111\000\111\000\111\000\111\000\111\000\111\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \115\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_code =
   "\255\004\255\255\005\255\255\007\255\006\255\255\003\255\000\004\
    \001\005\255\007\255\255\006\255\007\255\255\000\004\001\005\003\
    \006\002\007\255\001\255\255\000\001\255";
}

let rec parse_token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 8 (-1); __ocaml_lex_parse_token_rec lexbuf 0
and __ocaml_lex_parse_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 295 "src/approx_lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        EOL
      )
# 2817 "src/approx_lexer.ml"

  | 1 ->
# 299 "src/approx_lexer.mll"
      ( SPACES )
# 2822 "src/approx_lexer.ml"

  | 2 ->
# 301 "src/approx_lexer.mll"
    ( UNDERSCORE )
# 2827 "src/approx_lexer.ml"

  | 3 ->
# 303 "src/approx_lexer.mll"
    ( TILDE )
# 2832 "src/approx_lexer.ml"

  | 4 ->
# 305 "src/approx_lexer.mll"
      ( let s = Lexing.lexeme lexbuf in
        let name = String.sub s 1 (String.length s - 2) in
        (*
           if Hashtbl.mem keyword_table name then
           raise (Error(Keyword_as_label name, Location.curr lexbuf));
        *)
        LABEL name )
# 2843 "src/approx_lexer.ml"

  | 5 ->
# 312 "src/approx_lexer.mll"
         ( QUESTION )
# 2848 "src/approx_lexer.ml"

  | 6 ->
# 313 "src/approx_lexer.mll"
         ( QUESTIONQUESTION )
# 2853 "src/approx_lexer.ml"

  | 7 ->
# 315 "src/approx_lexer.mll"
      ( let s = Lexing.lexeme lexbuf in
        let name = String.sub s 1 (String.length s - 2) in
        (*
           if Hashtbl.mem keyword_table name then
           raise (Error(Keyword_as_label name, Location.curr lexbuf));
        *)
        OPTLABEL name )
# 2864 "src/approx_lexer.ml"

  | 8 ->
# 323 "src/approx_lexer.mll"
    ( let s = Lexing.lexeme lexbuf in
      try
        let i = String.index_from s 1 '%' in
        let kw = String.sub s 0 i in
        try Hashtbl.find keyword_table kw
        with Not_found -> rewind lexbuf (String.length s - i); LIDENT s
      with Not_found ->
        try Hashtbl.find keyword_table s
        with Not_found -> LIDENT s )
# 2877 "src/approx_lexer.ml"

  | 9 ->
# 333 "src/approx_lexer.mll"
    ( UIDENT(Lexing.lexeme lexbuf) )
# 2882 "src/approx_lexer.ml"

  | 10 ->
# 335 "src/approx_lexer.mll"
    ( INT (can_overflow cvt_int_literal lexbuf) )
# 2887 "src/approx_lexer.ml"

  | 11 ->
# 337 "src/approx_lexer.mll"
    ( FLOAT (remove_underscores(Lexing.lexeme lexbuf)) )
# 2892 "src/approx_lexer.ml"

  | 12 ->
# 339 "src/approx_lexer.mll"
    ( INT32 (can_overflow cvt_int32_literal lexbuf) )
# 2897 "src/approx_lexer.ml"

  | 13 ->
# 341 "src/approx_lexer.mll"
    ( INT64 (can_overflow cvt_int64_literal lexbuf) )
# 2902 "src/approx_lexer.ml"

  | 14 ->
# 343 "src/approx_lexer.mll"
    ( NATIVEINT (can_overflow cvt_nativeint_literal lexbuf) )
# 2907 "src/approx_lexer.ml"

  | 15 ->
# 345 "src/approx_lexer.mll"
    ( reset_string_buffer();
      let string_start = lexbuf.lex_start_p in
      string_start_loc := Lexing.lexeme_start lexbuf;
      let token = string lexbuf in
      lexbuf.lex_start_p <- string_start;
      token )
# 2917 "src/approx_lexer.ml"

  | 16 ->
# 352 "src/approx_lexer.mll"
    ( update_loc lexbuf None 1 false 1;
      CHAR (InRange (Lexing.lexeme_char lexbuf 1)) )
# 2923 "src/approx_lexer.ml"

  | 17 ->
# 355 "src/approx_lexer.mll"
    ( CHAR( InRange (Lexing.lexeme_char lexbuf 1)) )
# 2928 "src/approx_lexer.ml"

  | 18 ->
# 357 "src/approx_lexer.mll"
    ( CHAR( InRange (char_for_backslash (Lexing.lexeme_char lexbuf 2))) )
# 2933 "src/approx_lexer.ml"

  | 19 ->
# 359 "src/approx_lexer.mll"
    ( CHAR(can_overflow (char_for_decimal_code 2) lexbuf) )
# 2938 "src/approx_lexer.ml"

  | 20 ->
# 361 "src/approx_lexer.mll"
    ( CHAR( InRange (char_for_hexadecimal_code lexbuf 3)) )
# 2943 "src/approx_lexer.ml"

  | 21 ->
# 363 "src/approx_lexer.mll"
    ( let l = Lexing.lexeme lexbuf in
      CHAR ( Overflow l )
    )
# 2950 "src/approx_lexer.ml"

  | 22 ->
# 367 "src/approx_lexer.mll"
    ( entering_inline_code_block := true;
      comment_stack := Comment :: !comment_stack;
      rewind lexbuf 1;
      COMMENT )
# 2958 "src/approx_lexer.ml"

  | 23 ->
# 372 "src/approx_lexer.mll"
    (
      let comment_start = lexbuf.lex_start_p in
      comment_stack := Comment :: !comment_stack;
      let token = comment lexbuf in
      lexbuf.lex_start_p <- comment_start;
      token
    )
# 2969 "src/approx_lexer.ml"

  | 24 ->
# 380 "src/approx_lexer.mll"
    (
      match !comment_stack with
      | _ :: _ ->
          close_comment ()
      | [] ->
          rewind lexbuf 1;
          STAR
    )
# 2981 "src/approx_lexer.ml"

  | 25 ->
# 389 "src/approx_lexer.mll"
      ( if !entering_inline_code_block then begin
          entering_inline_code_block := false;
          match !comment_stack with
          | Code :: _ -> OCAMLDOC_CODE
          | Verbatim :: _ ->
              let verb_start = lexbuf.lex_start_p in
              let token = verbatim lexbuf in
              lexbuf.lex_start_p <- verb_start;
              token
          | _ -> assert false
        end else begin
          rewind lexbuf 1;
          LBRACE
        end
      )
# 3000 "src/approx_lexer.ml"

  | 26 ->
# 405 "src/approx_lexer.mll"
      (
        match !comment_stack with
        | (Code|Verbatim)::r ->
            comment_stack := r;
            let comment_start = lexbuf.lex_start_p in
            let token = comment lexbuf in
            lexbuf.lex_start_p <- comment_start;
            token
        | _ ->
            rewind lexbuf 1;
            match Bytes.get lexbuf.lex_buffer (lexbuf.lex_curr_pos - 1) with
            | ']' -> RBRACKET
            | 'v' -> LIDENT "v"
            | _ -> assert false
      )
# 3019 "src/approx_lexer.ml"

  | 27 ->
# 421 "src/approx_lexer.mll"
      (
        match !comment_stack with
        | Code::r ->
            comment_stack := r;
            let comment_start = lexbuf.lex_start_p in
            let token = comment lexbuf in
            lexbuf.lex_start_p <- comment_start;
            token
        | _ -> rewind lexbuf 1; RBRACKET
      )
# 3033 "src/approx_lexer.ml"

  | 28 ->
# 432 "src/approx_lexer.mll"
      (
        match !comment_stack with
        | Verbatim::r ->
            comment_stack := r;
            let comment_start = lexbuf.lex_start_p in
            let token = comment lexbuf in
            lexbuf.lex_start_p <- comment_start;
            token
        | _ -> rewind lexbuf 1; LIDENT "v"
      )
# 3047 "src/approx_lexer.ml"

  | 29 ->
# 443 "src/approx_lexer.mll"
      (
        let start = lexbuf.lex_start_p in
        quotation_start_loc := Lexing.lexeme_start lexbuf;
        let s = Lexing.lexeme lexbuf in
        let tag = String.sub s 2 (String.length s - 3) in
        quotation_kind := `Camlp4 tag;
        let token = quotation lexbuf in
        lexbuf.lex_start_p <- start;
        token
      )
# 3061 "src/approx_lexer.ml"

  | 30 ->
# 454 "src/approx_lexer.mll"
      (
        let start = lexbuf.lex_start_p in
        quotation_start_loc := Lexing.lexeme_start lexbuf;
        let s = Lexing.lexeme lexbuf in
        let delim = String.sub s 1 (String.length s - 2) in
        quotation_kind := `Ppx delim;
        let token = quotation lexbuf in
        lexbuf.lex_start_p <- start;
        token
      )
# 3075 "src/approx_lexer.ml"

  | 31 ->
let
# 464 "src/approx_lexer.mll"
                                   _num
# 3081 "src/approx_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 465 "src/approx_lexer.mll"
                                       _name
# 3086 "src/approx_lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2) in
# 467 "src/approx_lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        LINE_DIRECTIVE
      )
# 3092 "src/approx_lexer.ml"

  | 32 ->
# 470 "src/approx_lexer.mll"
         ( SHARP )
# 3097 "src/approx_lexer.ml"

  | 33 ->
# 471 "src/approx_lexer.mll"
         ( AMPERSAND )
# 3102 "src/approx_lexer.ml"

  | 34 ->
# 472 "src/approx_lexer.mll"
         ( AMPERAMPER )
# 3107 "src/approx_lexer.ml"

  | 35 ->
# 473 "src/approx_lexer.mll"
         ( BACKQUOTE )
# 3112 "src/approx_lexer.ml"

  | 36 ->
# 474 "src/approx_lexer.mll"
         ( QUOTE )
# 3117 "src/approx_lexer.ml"

  | 37 ->
# 475 "src/approx_lexer.mll"
         ( LPAREN )
# 3122 "src/approx_lexer.ml"

  | 38 ->
# 476 "src/approx_lexer.mll"
         ( RPAREN )
# 3127 "src/approx_lexer.ml"

  | 39 ->
# 477 "src/approx_lexer.mll"
         ( STAR )
# 3132 "src/approx_lexer.ml"

  | 40 ->
# 478 "src/approx_lexer.mll"
         ( COMMA )
# 3137 "src/approx_lexer.ml"

  | 41 ->
# 479 "src/approx_lexer.mll"
         ( MINUSGREATER )
# 3142 "src/approx_lexer.ml"

  | 42 ->
# 480 "src/approx_lexer.mll"
         ( DOT )
# 3147 "src/approx_lexer.ml"

  | 43 ->
# 481 "src/approx_lexer.mll"
         ( DOTDOT )
# 3152 "src/approx_lexer.ml"

  | 44 ->
# 482 "src/approx_lexer.mll"
         ( COLON )
# 3157 "src/approx_lexer.ml"

  | 45 ->
# 483 "src/approx_lexer.mll"
         ( COLONCOLON )
# 3162 "src/approx_lexer.ml"

  | 46 ->
# 484 "src/approx_lexer.mll"
         ( COLONEQUAL )
# 3167 "src/approx_lexer.ml"

  | 47 ->
# 485 "src/approx_lexer.mll"
         ( COLONGREATER )
# 3172 "src/approx_lexer.ml"

  | 48 ->
# 486 "src/approx_lexer.mll"
                                                  ( SEMI )
# 3177 "src/approx_lexer.ml"

  | 49 ->
# 487 "src/approx_lexer.mll"
         ( SEMISEMI )
# 3182 "src/approx_lexer.ml"

  | 50 ->
# 488 "src/approx_lexer.mll"
         ( LESS )
# 3187 "src/approx_lexer.ml"

  | 51 ->
# 489 "src/approx_lexer.mll"
         ( LESSMINUS )
# 3192 "src/approx_lexer.ml"

  | 52 ->
# 490 "src/approx_lexer.mll"
         ( EQUAL )
# 3197 "src/approx_lexer.ml"

  | 53 ->
# 491 "src/approx_lexer.mll"
         ( LBRACKET )
# 3202 "src/approx_lexer.ml"

  | 54 ->
# 492 "src/approx_lexer.mll"
         ( LBRACKETBAR )
# 3207 "src/approx_lexer.ml"

  | 55 ->
# 493 "src/approx_lexer.mll"
         ( LBRACKETLESS )
# 3212 "src/approx_lexer.ml"

  | 56 ->
# 494 "src/approx_lexer.mll"
         ( LBRACKETGREATER )
# 3217 "src/approx_lexer.ml"

  | 57 ->
# 495 "src/approx_lexer.mll"
         ( RBRACKET )
# 3222 "src/approx_lexer.ml"

  | 58 ->
# 496 "src/approx_lexer.mll"
         ( LBRACE )
# 3227 "src/approx_lexer.ml"

  | 59 ->
# 497 "src/approx_lexer.mll"
         ( LBRACELESS )
# 3232 "src/approx_lexer.ml"

  | 60 ->
# 498 "src/approx_lexer.mll"
         ( BAR )
# 3237 "src/approx_lexer.ml"

  | 61 ->
# 499 "src/approx_lexer.mll"
         ( BARBAR )
# 3242 "src/approx_lexer.ml"

  | 62 ->
# 500 "src/approx_lexer.mll"
         ( BARRBRACKET )
# 3247 "src/approx_lexer.ml"

  | 63 ->
# 501 "src/approx_lexer.mll"
         ( GREATER )
# 3252 "src/approx_lexer.ml"

  | 64 ->
# 502 "src/approx_lexer.mll"
         ( GREATERRBRACKET )
# 3257 "src/approx_lexer.ml"

  | 65 ->
# 503 "src/approx_lexer.mll"
         ( RBRACE )
# 3262 "src/approx_lexer.ml"

  | 66 ->
# 504 "src/approx_lexer.mll"
         ( GREATERRBRACE )
# 3267 "src/approx_lexer.ml"

  | 67 ->
# 505 "src/approx_lexer.mll"
         ( LBRACKETPERCENT )
# 3272 "src/approx_lexer.ml"

  | 68 ->
# 506 "src/approx_lexer.mll"
          ( LBRACKETPERCENTPERCENT )
# 3277 "src/approx_lexer.ml"

  | 69 ->
# 507 "src/approx_lexer.mll"
         ( LBRACKETAT )
# 3282 "src/approx_lexer.ml"

  | 70 ->
# 508 "src/approx_lexer.mll"
          ( LBRACKETATAT )
# 3287 "src/approx_lexer.ml"

  | 71 ->
# 509 "src/approx_lexer.mll"
           ( LBRACKETATATAT )
# 3292 "src/approx_lexer.ml"

  | 72 ->
# 511 "src/approx_lexer.mll"
         ( BANG )
# 3297 "src/approx_lexer.ml"

  | 73 ->
# 513 "src/approx_lexer.mll"
         ( INFIXOP0 "!=" )
# 3302 "src/approx_lexer.ml"

  | 74 ->
# 514 "src/approx_lexer.mll"
         ( PLUS )
# 3307 "src/approx_lexer.ml"

  | 75 ->
# 515 "src/approx_lexer.mll"
         ( PLUSDOT )
# 3312 "src/approx_lexer.ml"

  | 76 ->
# 516 "src/approx_lexer.mll"
         ( MINUS )
# 3317 "src/approx_lexer.ml"

  | 77 ->
# 517 "src/approx_lexer.mll"
         ( MINUSDOT )
# 3322 "src/approx_lexer.ml"

  | 78 ->
# 520 "src/approx_lexer.mll"
    ( check_commentclose lexbuf (fun s -> PREFIXOP s) )
# 3327 "src/approx_lexer.ml"

  | 79 ->
# 522 "src/approx_lexer.mll"
    ( check_commentclose lexbuf (fun s -> PREFIXOP s) )
# 3332 "src/approx_lexer.ml"

  | 80 ->
# 524 "src/approx_lexer.mll"
    ( if !entering_inline_code_block then begin
        entering_inline_code_block := false;
        comment_stack := Cinaps :: !comment_stack;
        rewind lexbuf
          (Lexing.lexeme_end lexbuf - Lexing.lexeme_start lexbuf - 1);
        OCAMLDOC_CODE
      end
      else check_commentclose lexbuf (fun s -> INFIXOP0 s)
    )
# 3345 "src/approx_lexer.ml"

  | 81 ->
# 534 "src/approx_lexer.mll"
    ( check_commentclose lexbuf (fun s -> INFIXOP0 s) )
# 3350 "src/approx_lexer.ml"

  | 82 ->
# 536 "src/approx_lexer.mll"
    ( check_commentclose lexbuf (fun s -> INFIXOP1 s) )
# 3355 "src/approx_lexer.ml"

  | 83 ->
# 538 "src/approx_lexer.mll"
    ( check_commentclose lexbuf (fun s -> INFIXOP2 s) )
# 3360 "src/approx_lexer.ml"

  | 84 ->
# 540 "src/approx_lexer.mll"
    ( check_commentclose lexbuf (fun s -> INFIXOP4 s) )
# 3365 "src/approx_lexer.ml"

  | 85 ->
# 542 "src/approx_lexer.mll"
    ( check_commentclose lexbuf (fun s -> INFIXOP3 s) )
# 3370 "src/approx_lexer.ml"

  | 86 ->
# 545 "src/approx_lexer.mll"
    ( check_commentclose lexbuf (fun _ -> LET) )
# 3375 "src/approx_lexer.ml"

  | 87 ->
# 547 "src/approx_lexer.mll"
    ( check_commentclose lexbuf (fun _ -> AND) )
# 3380 "src/approx_lexer.ml"

  | 88 ->
# 549 "src/approx_lexer.mll"
        ( EOF )
# 3385 "src/approx_lexer.ml"

  | 89 ->
# 551 "src/approx_lexer.mll"
    ( ILLEGAL_CHAR (Lexing.lexeme_char lexbuf 0)      )
# 3390 "src/approx_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_parse_token_rec lexbuf __ocaml_lex_state

and quotation lexbuf =
   __ocaml_lex_quotation_rec lexbuf 153
and __ocaml_lex_quotation_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 555 "src/approx_lexer.mll"
      ( match !quotation_kind with
        | `Camlp4 tag -> QUOTATION ("<:"^tag^"<")
        | `Ppx _ -> quotation lexbuf )
# 3404 "src/approx_lexer.ml"

  | 1 ->
# 559 "src/approx_lexer.mll"
      ( match !quotation_kind with
        | `Ppx delim ->
            let s = Lexing.lexeme lexbuf in
            let ndelim = String.sub s 1 (String.length s - 2) in
            if ndelim = delim then QUOTATION ("{"^delim^"|")
            else quotation lexbuf
        | `Camlp4 _ -> quotation lexbuf )
# 3415 "src/approx_lexer.ml"

  | 2 ->
# 567 "src/approx_lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        quotation lexbuf
      )
# 3422 "src/approx_lexer.ml"

  | 3 ->
# 571 "src/approx_lexer.mll"
      ( match !quotation_kind with
        | `Camlp4 tag -> QUOTATION ("<:"^tag^"<")
        | `Ppx delim -> QUOTATION ("{"^delim^"|") )
# 3429 "src/approx_lexer.ml"

  | 4 ->
# 574 "src/approx_lexer.mll"
      ( quotation lexbuf )
# 3434 "src/approx_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_quotation_rec lexbuf __ocaml_lex_state

and comment lexbuf =
   __ocaml_lex_comment_rec lexbuf 163
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 578 "src/approx_lexer.mll"
      ( comment_stack := Comment :: !comment_stack;
        comment lexbuf
      )
# 3448 "src/approx_lexer.ml"

  | 1 ->
# 582 "src/approx_lexer.mll"
      ( let tok = close_comment () in
        if in_verbatim () then verbatim lexbuf
        else match !comment_stack with
        | (Comment | CommentCont) :: _ -> comment lexbuf
        | _ -> tok
      )
# 3458 "src/approx_lexer.ml"

  | 2 ->
# 589 "src/approx_lexer.mll"
      ( if in_verbatim() then comment lexbuf else
          let tok = match !comment_stack with
            | CommentCont::_ -> COMMENTCONT
            | Comment::r ->
                comment_stack := CommentCont::r;
                COMMENT
            | _s -> assert false
          in
          let block =
            match Bytes.get lexbuf.lex_buffer (lexbuf.lex_curr_pos - 1) with
            | '[' -> Code
            | 'v' -> Verbatim
            | _ -> assert false
          in
          comment_stack := block :: !comment_stack;
          entering_inline_code_block := true;
          (* unparse the token, to be parsed again as code *)
          lexbuf.Lexing.lex_curr_p <- lexbuf.Lexing.lex_start_p;
          lexbuf.Lexing.lex_curr_pos <- lexbuf.Lexing.lex_start_pos;
          tok
      )
# 3483 "src/approx_lexer.ml"

  | 3 ->
# 611 "src/approx_lexer.mll"
    ( reset_string_buffer();
      string_start_loc := Lexing.lexeme_start lexbuf;
      ignore (string lexbuf);
      reset_string_buffer ();
      comment lexbuf
    )
# 3493 "src/approx_lexer.ml"

  | 4 ->
# 618 "src/approx_lexer.mll"
    ( quotation_start_loc := Lexing.lexeme_start lexbuf;
      let s = Lexing.lexeme lexbuf in
      let delim = String.sub s 1 (String.length s - 2) in
      quotation_kind := `Ppx delim;
      ignore (quotation lexbuf);
      comment lexbuf
    )
# 3504 "src/approx_lexer.ml"

  | 5 ->
# 627 "src/approx_lexer.mll"
    ( comment lexbuf )
# 3509 "src/approx_lexer.ml"

  | 6 ->
# 629 "src/approx_lexer.mll"
    ( update_loc lexbuf None 1 false 1;
      comment lexbuf
    )
# 3516 "src/approx_lexer.ml"

  | 7 ->
# 633 "src/approx_lexer.mll"
    ( comment lexbuf )
# 3521 "src/approx_lexer.ml"

  | 8 ->
# 635 "src/approx_lexer.mll"
    ( comment lexbuf )
# 3526 "src/approx_lexer.ml"

  | 9 ->
# 637 "src/approx_lexer.mll"
    ( comment lexbuf )
# 3531 "src/approx_lexer.ml"

  | 10 ->
# 639 "src/approx_lexer.mll"
    ( comment lexbuf )
# 3536 "src/approx_lexer.ml"

  | 11 ->
# 641 "src/approx_lexer.mll"
    ( update_loc lexbuf None 1 false 0;
      comment lexbuf
    )
# 3543 "src/approx_lexer.ml"

  | 12 ->
# 645 "src/approx_lexer.mll"
    ( comment lexbuf )
# 3548 "src/approx_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and verbatim lexbuf =
   __ocaml_lex_verbatim_rec lexbuf 197
and __ocaml_lex_verbatim_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 651 "src/approx_lexer.mll"
      ( comment_stack := Comment :: !comment_stack;
        comment lexbuf
      )
# 3562 "src/approx_lexer.ml"

  | 1 ->
# 655 "src/approx_lexer.mll"
      ( (* leave the verbatim block and unparse the token *)
        comment_stack :=
          (match !comment_stack with
           | Verbatim :: s -> s
           | _ -> assert false);
        lexbuf.Lexing.lex_curr_p <- lexbuf.Lexing.lex_start_p;
        lexbuf.Lexing.lex_curr_pos <- lexbuf.Lexing.lex_start_pos;
        (* let the surrounding comments close themselves *)
        match !comment_stack with
        | Comment :: _ -> comment lexbuf
        | CommentCont :: r -> comment_stack := Comment :: r; comment lexbuf
        | _ -> OCAMLDOC_VERB
      )
# 3579 "src/approx_lexer.ml"

  | 2 ->
# 669 "src/approx_lexer.mll"
      ( (* Unparse the token but leave the comment stack.
           The token rule will reparse, detect it,
           pop the verbatim and return to the comment rule. *)
        rewind lexbuf 2;
        OCAMLDOC_VERB )
# 3588 "src/approx_lexer.ml"

  | 3 ->
# 675 "src/approx_lexer.mll"
    ( reset_string_buffer();
      string_start_loc := Lexing.lexeme_start lexbuf;
      ignore (string lexbuf);
      reset_string_buffer ();
      verbatim lexbuf
    )
# 3598 "src/approx_lexer.ml"

  | 4 ->
# 682 "src/approx_lexer.mll"
    ( verbatim lexbuf )
# 3603 "src/approx_lexer.ml"

  | 5 ->
# 684 "src/approx_lexer.mll"
    ( update_loc lexbuf None 1 false 1;
      verbatim lexbuf
    )
# 3610 "src/approx_lexer.ml"

  | 6 ->
# 688 "src/approx_lexer.mll"
    ( verbatim lexbuf )
# 3615 "src/approx_lexer.ml"

  | 7 ->
# 690 "src/approx_lexer.mll"
    ( verbatim lexbuf )
# 3620 "src/approx_lexer.ml"

  | 8 ->
# 692 "src/approx_lexer.mll"
    ( verbatim lexbuf )
# 3625 "src/approx_lexer.ml"

  | 9 ->
# 694 "src/approx_lexer.mll"
    ( verbatim lexbuf )
# 3630 "src/approx_lexer.ml"

  | 10 ->
# 696 "src/approx_lexer.mll"
    ( update_loc lexbuf None 1 false 0;
      verbatim lexbuf
    )
# 3637 "src/approx_lexer.ml"

  | 11 ->
# 700 "src/approx_lexer.mll"
    ( OCAMLDOC_VERB )
# 3642 "src/approx_lexer.ml"

  | 12 ->
# 702 "src/approx_lexer.mll"
    ( verbatim lexbuf )
# 3647 "src/approx_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_verbatim_rec lexbuf __ocaml_lex_state

and string lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 2 (-1); __ocaml_lex_string_rec lexbuf 227
and __ocaml_lex_string_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 706 "src/approx_lexer.mll"
      ( STRING (get_stored_string ()) )
# 3659 "src/approx_lexer.ml"

  | 1 ->
let
# 707 "src/approx_lexer.mll"
                                  space
# 3665 "src/approx_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 708 "src/approx_lexer.mll"
      ( update_loc lexbuf None 1 false (String.length space);
        string lexbuf
      )
# 3671 "src/approx_lexer.ml"

  | 2 ->
# 712 "src/approx_lexer.mll"
    ( store_string_char(char_for_backslash(Lexing.lexeme_char lexbuf 1));
      string lexbuf )
# 3677 "src/approx_lexer.ml"

  | 3 ->
# 715 "src/approx_lexer.mll"
    ( (match can_overflow (char_for_decimal_code 1) lexbuf with
      | Overflow _ ->
          let s = Lexing.lexeme lexbuf in
          for i = 0 to String.length s - 1 do store_string_char s.[i] done
      | InRange c -> store_string_char c);
      string lexbuf )
# 3687 "src/approx_lexer.ml"

  | 4 ->
# 722 "src/approx_lexer.mll"
    ( store_string_char(char_for_hexadecimal_code lexbuf 2);
      string lexbuf )
# 3693 "src/approx_lexer.ml"

  | 5 ->
# 725 "src/approx_lexer.mll"
    ( if in_comment ()
      then string lexbuf
      else begin
        (*  Should be an error, but we are very lax.
            raise (Error (Illegal_escape (Lexing.lexeme lexbuf),
              Location.curr lexbuf))
        *)
        store_string_char (Lexing.lexeme_char lexbuf 0);
        store_string_char (Lexing.lexeme_char lexbuf 1);
        string lexbuf
      end
    )
# 3709 "src/approx_lexer.ml"

  | 6 ->
# 738 "src/approx_lexer.mll"
    (
      update_loc lexbuf None 1 false 0;
      let s = Lexing.lexeme lexbuf in
      for i = 0 to String.length s - 1 do
        store_string_char s.[i];
      done;
      string lexbuf
    )
# 3721 "src/approx_lexer.ml"

  | 7 ->
# 747 "src/approx_lexer.mll"
    ( store_string_char(Lexing.lexeme_char lexbuf 0);
      string lexbuf )
# 3727 "src/approx_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_string_rec lexbuf __ocaml_lex_state

;;

# 750 "src/approx_lexer.mll"
 

let token =
  let rec tok lexbuf = function
    | [] -> parse_token lexbuf
    | x::xs -> begin
        try x lexbuf with
        | _ -> tok lexbuf xs
      end
  in fun lexbuf -> tok lexbuf !lexer_extensions

let rec token_locs lexbuf =
  match token lexbuf with
    COMMENT -> token_locs lexbuf
  | token ->
      token, ( lexbuf.lex_start_p, lexbuf.lex_curr_p)

let rec token_pos lexbuf =
  match token lexbuf with
    COMMENT -> token_pos lexbuf
  | token ->
      token, ( lexbuf.lex_start_p.pos_cnum, lexbuf.lex_curr_p.pos_cnum)


let token_locs_and_comments lexbuf =
  let token = token lexbuf in
  token,  ( lexbuf.lex_start_p, lexbuf.lex_curr_p)

let get_token = token

let token_with_comments = get_token

let rec token lexbuf =
  match get_token lexbuf with
    COMMENT -> token lexbuf
  | tok -> tok

let tokens_of_file filename =
  let ic = open_in filename in
  try
    init ();
    let lexbuf = Lexing.from_channel ic in
    let rec iter tokens =
      let token = token_pos lexbuf in
      match token with
        (EOF, _) -> List.rev tokens
      | _ -> iter (token :: tokens)
    in
    let tokens = iter [] in
    close_in ic;
    tokens
  with e -> close_in ic; raise e

let tokens_with_loc_of_string s =
  init ();
  let lexbuf = Lexing.from_string s in
  let rec iter tokens =
    let token = token_pos lexbuf in
    match token with
      (EOF, _) -> List.rev tokens
    | _ -> iter (token :: tokens)
  in
  let tokens = iter [] in
  tokens

let tokens_of_string s =
  init ();
  let lexbuf = Lexing.from_string s in
  let rec iter tokens =
    let token = token lexbuf in
    match token with
      (EOF) -> List.rev tokens
    | _ -> iter (token :: tokens)
  in
  let tokens = iter [] in
  tokens

let lines () = List.rev ( !lines_starts )


# 3815 "src/approx_lexer.ml"
