/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.lang.document;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemAlreadyExistsException;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.ProviderNotFoundException;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import net.sourceforge.pmd.annotation.Experimental;
import net.sourceforge.pmd.annotation.InternalApi;
import net.sourceforge.pmd.internal.util.AssertionUtil;
import net.sourceforge.pmd.lang.Language;
import net.sourceforge.pmd.lang.LanguageVersion;
import net.sourceforge.pmd.lang.LanguageVersionDiscoverer;
import net.sourceforge.pmd.lang.document.NioTextFile;
import net.sourceforge.pmd.lang.document.StringTextFile;
import net.sourceforge.pmd.lang.document.TextFile;
import net.sourceforge.pmd.util.IOUtil;
import net.sourceforge.pmd.util.log.MessageReporter;

public final class FileCollector
implements AutoCloseable {
    private final Set<TextFile> allFilesToProcess = new LinkedHashSet<TextFile>();
    private final List<Closeable> resourcesToClose = new ArrayList<Closeable>();
    private Charset charset = StandardCharsets.UTF_8;
    private final LanguageVersionDiscoverer discoverer;
    private final MessageReporter reporter;
    private final String outerFsDisplayName;
    @Deprecated
    private final List<String> legacyRelativizeRoots = new ArrayList<String>();
    private final List<Path> relativizeRootPaths = new ArrayList<Path>();
    private boolean closed;

    private FileCollector(LanguageVersionDiscoverer discoverer, MessageReporter reporter, String outerFsDisplayName) {
        this.discoverer = discoverer;
        this.reporter = reporter;
        this.outerFsDisplayName = outerFsDisplayName;
    }

    @InternalApi
    public static FileCollector newCollector(LanguageVersionDiscoverer discoverer, MessageReporter reporter) {
        return new FileCollector(discoverer, reporter, null);
    }

    @InternalApi
    public List<TextFile> getCollectedFiles() {
        if (this.closed) {
            throw new IllegalStateException("Collector was closed!");
        }
        ArrayList<TextFile> allFilesToProcess = new ArrayList<TextFile>(this.allFilesToProcess);
        Collections.sort(allFilesToProcess, new Comparator<TextFile>(){

            @Override
            public int compare(TextFile o1, TextFile o2) {
                return o1.getPathId().compareTo(o2.getPathId());
            }
        });
        return Collections.unmodifiableList(allFilesToProcess);
    }

    @InternalApi
    public MessageReporter getReporter() {
        return this.reporter;
    }

    @Override
    public void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
        IOException exception = IOUtil.closeAll(this.resourcesToClose);
        if (exception != null) {
            this.reporter.errorEx("Error while closing resources", exception);
        }
    }

    public boolean addFile(Path file) {
        if (!Files.isRegularFile(file, new LinkOption[0])) {
            this.reporter.error("Not a regular file {0}", file);
            return false;
        }
        LanguageVersion languageVersion = this.discoverLanguage(file.toString());
        if (languageVersion != null) {
            return this.addFileImpl(new NioTextFile(file, this.charset, languageVersion, this.getDisplayName(file)));
        }
        return false;
    }

    public boolean addFile(Path file, Language language) {
        AssertionUtil.requireParamNotNull("language", language);
        if (!Files.isRegularFile(file, new LinkOption[0])) {
            this.reporter.error("Not a regular file {0}", file);
            return false;
        }
        NioTextFile nioTextFile = new NioTextFile(file, this.charset, this.discoverer.getDefaultLanguageVersion(language), this.getDisplayName(file));
        return this.addFileImpl(nioTextFile);
    }

    public boolean addFile(TextFile textFile) {
        AssertionUtil.requireParamNotNull("textFile", textFile);
        if (this.checkContextualVersion(textFile)) {
            return this.addFileImpl(textFile);
        }
        return false;
    }

    public boolean addSourceFile(String sourceContents, String pathId) {
        AssertionUtil.requireParamNotNull("sourceContents", sourceContents);
        AssertionUtil.requireParamNotNull("pathId", pathId);
        LanguageVersion version = this.discoverLanguage(pathId);
        if (version != null) {
            return this.addFileImpl(new StringTextFile(sourceContents, pathId, pathId, version));
        }
        return false;
    }

    private boolean addFileImpl(TextFile textFile) {
        this.reporter.trace("Adding file {0} (lang: {1}) ", textFile.getPathId(), textFile.getLanguageVersion().getTerseName());
        if (this.allFilesToProcess.add(textFile)) {
            return true;
        }
        this.reporter.trace("File was already collected, skipping", new Object[0]);
        return false;
    }

    private LanguageVersion discoverLanguage(String file) {
        if (this.discoverer.getForcedVersion() != null) {
            return this.discoverer.getForcedVersion();
        }
        List<Language> languages = this.discoverer.getLanguagesForFile(file);
        if (languages.isEmpty()) {
            this.reporter.trace("File {0} matches no known language, ignoring", file);
            return null;
        }
        Language lang = languages.get(0);
        if (languages.size() > 1) {
            this.reporter.trace("File {0} matches multiple languages ({1}), selecting {2}", file, languages, lang);
        }
        return this.discoverer.getDefaultLanguageVersion(lang);
    }

    private boolean checkContextualVersion(TextFile textFile) {
        Language language;
        LanguageVersion contextVersion;
        LanguageVersion fileVersion = textFile.getLanguageVersion();
        if (!fileVersion.equals(contextVersion = this.discoverer.getDefaultLanguageVersion(language = fileVersion.getLanguage()))) {
            this.reporter.error("Cannot add file {0}: version ''{2}'' does not match ''{1}''", textFile.getPathId(), contextVersion, fileVersion);
            return false;
        }
        return true;
    }

    private String getDisplayName(Path file) {
        String localDisplayName = this.getLocalDisplayName(file);
        if (this.outerFsDisplayName != null) {
            return this.outerFsDisplayName + "!" + localDisplayName;
        }
        return localDisplayName;
    }

    private String getLocalDisplayName(Path file) {
        if (!this.relativizeRootPaths.isEmpty()) {
            return FileCollector.getDisplayName(file, this.relativizeRootPaths);
        }
        return FileCollector.getDisplayNameLegacy(file, this.legacyRelativizeRoots);
    }

    static String getDisplayNameLegacy(Path file, List<String> relativizeRoots) {
        String fileName = file.toString();
        for (String root : relativizeRoots) {
            if (!file.startsWith(root)) continue;
            if (fileName.startsWith(File.separator, root.length())) {
                return fileName.substring(root.length() + 1);
            }
            return fileName.substring(root.length());
        }
        return fileName;
    }

    private static String getDisplayName(Path file, List<Path> relativizeRoots) {
        Path best = file;
        for (Path root : relativizeRoots) {
            Path candidate;
            if (FileCollector.isFileSystemRoot(root)) {
                best = file.toAbsolutePath();
                continue;
            }
            if (!root.getFileSystem().equals(file.getFileSystem())) {
                root = file.getFileSystem().getPath(root.toString(), new String[0]);
            }
            if (root.isAbsolute() != file.isAbsolute()) {
                root = root.toAbsolutePath();
                file = file.toAbsolutePath();
            }
            if ((candidate = root.relativize(file)).getNameCount() >= best.getNameCount()) continue;
            best = candidate;
        }
        return best.toString();
    }

    private static boolean isFileSystemRoot(Path root) {
        return root.isAbsolute() && root.getNameCount() == 0;
    }

    public boolean addDirectory(Path dir) throws IOException {
        if (!Files.isDirectory(dir, new LinkOption[0])) {
            this.reporter.error("Not a directory {0}", dir);
            return false;
        }
        Files.walkFileTree(dir, EnumSet.of(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (attrs.isRegularFile()) {
                    FileCollector.this.addFile(file);
                }
                return super.visitFile(file, attrs);
            }
        });
        return true;
    }

    public boolean addFileOrDirectory(Path file) throws IOException {
        if (Files.isDirectory(file, new LinkOption[0])) {
            return this.addDirectory(file);
        }
        if (Files.isRegularFile(file, new LinkOption[0])) {
            return this.addFile(file);
        }
        this.reporter.error("Not a file or directory {0}", file);
        return false;
    }

    @Deprecated
    public FileSystem addZipFile(Path zipFile) {
        if (!Files.isRegularFile(zipFile, new LinkOption[0])) {
            throw new IllegalArgumentException("Not a regular file: " + zipFile);
        }
        URI zipUri = URI.create("jar:" + zipFile.toUri());
        try {
            FileSystem fs = FileSystems.newFileSystem(zipUri, Collections.emptyMap());
            this.resourcesToClose.add(fs);
            return fs;
        }
        catch (IOException | FileSystemAlreadyExistsException | ProviderNotFoundException e) {
            this.reporter.errorEx("Cannot open zip file " + zipFile, e);
            return null;
        }
    }

    @Experimental
    public boolean addZipFileWithContent(Path zipFile) throws IOException {
        FileSystem fs;
        if (!Files.isRegularFile(zipFile, new LinkOption[0])) {
            throw new IllegalArgumentException("Not a regular file: " + zipFile);
        }
        URI zipUri = URI.create("jar:" + zipFile.toUri());
        boolean isNewFileSystem = false;
        try {
            fs = FileSystems.getFileSystem(zipUri);
        }
        catch (FileSystemNotFoundException ignored) {
            try {
                fs = FileSystems.newFileSystem(zipUri, Collections.emptyMap());
                isNewFileSystem = true;
            }
            catch (IOException | ProviderNotFoundException e) {
                this.reporter.errorEx("Cannot open zip file " + zipFile, e);
                return false;
            }
        }
        try (FileCollector zipCollector = this.newZipCollector(zipFile);){
            for (Path zipRoot : fs.getRootDirectories()) {
                zipCollector.addFileOrDirectory(zipRoot);
            }
            this.absorb(zipCollector);
            if (isNewFileSystem) {
                this.resourcesToClose.add(fs);
            }
        }
        catch (IOException ioe) {
            this.reporter.errorEx("Error reading zip file " + zipFile + ", will be skipped", ioe);
            fs.close();
            return false;
        }
        return true;
    }

    @Experimental
    private FileCollector newZipCollector(Path zipFilePath) {
        String zipDisplayName = this.getDisplayName(zipFilePath);
        return new FileCollector(this.discoverer, this.reporter, zipDisplayName);
    }

    public void setCharset(Charset charset) {
        this.charset = Objects.requireNonNull(charset);
    }

    @Deprecated
    public void relativizeWith(String prefix) {
        this.legacyRelativizeRoots.add(Objects.requireNonNull(prefix));
    }

    public void relativizeWith(Path path) {
        this.relativizeRootPaths.add(Objects.requireNonNull(path));
        Collections.sort(this.relativizeRootPaths, new Comparator<Path>(){

            @Override
            public int compare(Path o1, Path o2) {
                int lengthCmp = Integer.compare(o1.getNameCount(), o2.getNameCount());
                return lengthCmp == 0 ? o1.compareTo(o2) : lengthCmp;
            }
        });
    }

    public void exclude(FileCollector excludeCollector) {
        HashSet<TextFile> toExclude = new HashSet<TextFile>(excludeCollector.allFilesToProcess);
        Iterator<TextFile> iterator = this.allFilesToProcess.iterator();
        while (iterator.hasNext()) {
            TextFile file = iterator.next();
            if (!toExclude.contains(file)) continue;
            this.reporter.trace("Excluding file {0}", file.getPathId());
            iterator.remove();
        }
    }

    public void absorb(FileCollector otherCollector) {
        this.allFilesToProcess.addAll(otherCollector.allFilesToProcess);
        this.resourcesToClose.addAll(otherCollector.resourcesToClose);
        otherCollector.allFilesToProcess.clear();
        otherCollector.resourcesToClose.clear();
    }

    public void filterLanguages(Set<Language> languages) {
        Iterator<TextFile> iterator = this.allFilesToProcess.iterator();
        while (iterator.hasNext()) {
            TextFile file = iterator.next();
            Language lang = file.getLanguageVersion().getLanguage();
            if (languages.contains(lang)) continue;
            this.reporter.trace("Filtering out {0}, no rules for language {1}", file.getPathId(), lang);
            iterator.remove();
        }
    }

    public String toString() {
        return "FileCollector{filesToProcess=" + this.allFilesToProcess + '}';
    }
}

