/*
 * Decompiled with CFR 0.152.
 */
package name.abuchen.portfolio.snapshot;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import name.abuchen.portfolio.Messages;
import name.abuchen.portfolio.model.Account;
import name.abuchen.portfolio.model.AccountTransaction;
import name.abuchen.portfolio.model.Client;
import name.abuchen.portfolio.model.Portfolio;
import name.abuchen.portfolio.model.PortfolioTransaction;
import name.abuchen.portfolio.model.Security;
import name.abuchen.portfolio.model.Transaction;
import name.abuchen.portfolio.model.TransactionPair;
import name.abuchen.portfolio.money.CurrencyConverter;
import name.abuchen.portfolio.money.Money;
import name.abuchen.portfolio.money.MoneyCollectors;
import name.abuchen.portfolio.money.MutableMoney;
import name.abuchen.portfolio.money.Values;
import name.abuchen.portfolio.snapshot.AccountSnapshot;
import name.abuchen.portfolio.snapshot.ClientIRRYield;
import name.abuchen.portfolio.snapshot.ClientSnapshot;
import name.abuchen.portfolio.snapshot.security.CapitalGainsRecord;
import name.abuchen.portfolio.snapshot.security.SecurityPerformanceIndicator;
import name.abuchen.portfolio.snapshot.security.SecurityPerformanceRecord;
import name.abuchen.portfolio.snapshot.security.SecurityPerformanceSnapshot;
import name.abuchen.portfolio.snapshot.trail.Trail;
import name.abuchen.portfolio.snapshot.trail.TrailProvider;
import name.abuchen.portfolio.snapshot.trail.TrailRecord;
import name.abuchen.portfolio.util.Interval;

public class ClientPerformanceSnapshot {
    private final Client client;
    private final CurrencyConverter converter;
    private final Interval period;
    private ClientSnapshot snapshotStart;
    private ClientSnapshot snapshotEnd;
    private final EnumMap<CategoryType, Category> categories = new EnumMap(CategoryType.class);
    private final List<TransactionPair<?>> earnings = new ArrayList();
    private final List<TransactionPair<?>> fees = new ArrayList();
    private final List<TransactionPair<?>> taxes = new ArrayList();
    private double irr;

    public ClientPerformanceSnapshot(Client client, CurrencyConverter converter, LocalDate startDate, LocalDate endDate) {
        this(client, converter, Interval.of(startDate, endDate));
    }

    public ClientPerformanceSnapshot(Client client, CurrencyConverter converter, Interval period) {
        this.client = client;
        this.converter = converter;
        this.period = period;
        this.snapshotStart = ClientSnapshot.create(client, converter, period.getStart());
        this.snapshotEnd = ClientSnapshot.create(client, converter, period.getEnd());
        this.calculate();
    }

    public Client getClient() {
        return this.client;
    }

    public ClientSnapshot getStartClientSnapshot() {
        return this.snapshotStart;
    }

    public ClientSnapshot getEndClientSnapshot() {
        return this.snapshotEnd;
    }

    public List<Category> getCategories() {
        return new ArrayList<Category>(this.categories.values());
    }

    public Category getCategoryByType(CategoryType type) {
        return this.categories.get((Object)type);
    }

    public Money getValue(CategoryType categoryType) {
        return this.categories.get((Object)categoryType).getValuation();
    }

    public List<TransactionPair<?>> getEarnings() {
        return this.earnings;
    }

    public List<TransactionPair<?>> getFees() {
        return this.fees;
    }

    public List<TransactionPair<?>> getTaxes() {
        return this.taxes;
    }

    public double getPerformanceIRR() {
        return this.irr;
    }

    public Money getAbsoluteDelta() {
        MutableMoney delta = MutableMoney.of(this.converter.getTermCurrency());
        for (Map.Entry<CategoryType, Category> entry : this.categories.entrySet()) {
            switch (entry.getKey()) {
                case CAPITAL_GAINS: 
                case REALIZED_CAPITAL_GAINS: 
                case EARNINGS: 
                case CURRENCY_GAINS: {
                    delta.add(entry.getValue().getValuation());
                    break;
                }
                case FEES: 
                case TAXES: {
                    delta.subtract(entry.getValue().getValuation());
                    break;
                }
            }
        }
        return delta.toMoney();
    }

    private void calculate() {
        this.categories.put(CategoryType.INITIAL_VALUE, new Category(String.format(Messages.ColumnInitialValue, Values.Date.format(this.snapshotStart.getTime())), "", this.snapshotStart.getMonetaryAssets()));
        Money zero = Money.of(this.converter.getTermCurrency(), 0L);
        this.categories.put(CategoryType.CAPITAL_GAINS, new Category(Messages.ColumnCapitalGains, "+", zero));
        this.categories.put(CategoryType.REALIZED_CAPITAL_GAINS, new Category(Messages.LabelRealizedCapitalGains, "+", zero));
        this.categories.put(CategoryType.EARNINGS, new Category(Messages.ColumnEarnings, "+", zero));
        this.categories.put(CategoryType.FEES, new Category(Messages.ColumnPaidFees, "-", zero));
        this.categories.put(CategoryType.TAXES, new Category(Messages.ColumnPaidTaxes, "-", zero));
        this.categories.put(CategoryType.CURRENCY_GAINS, new Category(Messages.ColumnCurrencyGains, "+", zero));
        this.categories.put(CategoryType.TRANSFERS, new Category(Messages.ColumnTransfers, "+", zero));
        this.categories.put(CategoryType.FINAL_VALUE, new Category(String.format(Messages.ColumnFinalValue, Values.Date.format(this.snapshotEnd.getTime())), "=", this.snapshotEnd.getMonetaryAssets()));
        this.irr = ClientIRRYield.create(this.client, this.snapshotStart, this.snapshotEnd).getIrr();
        this.addCapitalGains();
        this.addEarnings();
        this.addCurrencyGains();
    }

    private void addCapitalGains() {
        SecurityPerformanceSnapshot securityPerformance = SecurityPerformanceSnapshot.create(this.client, this.converter, this.period, this.snapshotStart, this.snapshotEnd, SecurityPerformanceIndicator.CapitalGains.class);
        Category realizedCapitalGains = this.categories.get((Object)CategoryType.REALIZED_CAPITAL_GAINS);
        this.addCapitalGains(realizedCapitalGains, securityPerformance, record -> record.getRealizedCapitalGains());
        Category capitalGains = this.categories.get((Object)CategoryType.CAPITAL_GAINS);
        this.addCapitalGains(capitalGains, securityPerformance, record -> record.getUnrealizedCapitalGains());
    }

    private void addCapitalGains(Category category, SecurityPerformanceSnapshot securityPerformance, Function<SecurityPerformanceRecord, CapitalGainsRecord> mapper) {
        category.positions = securityPerformance.getRecords().stream().sorted((p1, p2) -> p1.getSecurityName().compareToIgnoreCase(p2.getSecurityName())).map(mapper).filter(gains -> !gains.getCapitalGains().isZero() || !gains.getForexCaptialGains().isZero()).map(gains -> new Position(gains.getSecurity(), gains.getCapitalGains(), gains.getCapitalGainsTrail(), gains.getForexCaptialGains(), gains.getForexCapitalGainsTrail())).collect(Collectors.toList());
        category.valuation = category.positions.stream().map(Position::getValue).collect(MoneyCollectors.sum(this.converter.getTermCurrency()));
    }

    private void addEarnings() {
        String termCurrency = this.converter.getTermCurrency();
        MutableMoney mEarnings = MutableMoney.of(termCurrency);
        MutableMoney mFees = MutableMoney.of(termCurrency);
        MutableMoney mTaxes = MutableMoney.of(termCurrency);
        MutableMoney mDeposits = MutableMoney.of(termCurrency);
        MutableMoney mRemovals = MutableMoney.of(termCurrency);
        HashMap<Security, MutableMoney> earningsBySecurity = new HashMap<Security, MutableMoney>();
        HashMap<Security, MutableMoney> feesBySecurity = new HashMap<Security, MutableMoney>();
        HashMap<Security, MutableMoney> taxesBySecurity = new HashMap<Security, MutableMoney>();
        for (Account account : this.client.getAccounts()) {
            for (AccountTransaction accountTransaction : account.getTransactions()) {
                if (!this.period.contains(accountTransaction.getDateTime())) continue;
                Money value = accountTransaction.getMonetaryAmount().with(this.converter.at(accountTransaction.getDateTime()));
                switch (accountTransaction.getType()) {
                    case INTEREST: 
                    case DIVIDENDS: {
                        this.addEarningTransaction(account, accountTransaction, mEarnings, earningsBySecurity, mFees, mTaxes, feesBySecurity, taxesBySecurity);
                        break;
                    }
                    case INTEREST_CHARGE: {
                        mEarnings.subtract(value);
                        this.earnings.add(new TransactionPair<AccountTransaction>(account, accountTransaction));
                        earningsBySecurity.computeIfAbsent(null, s -> MutableMoney.of(termCurrency)).subtract(value);
                        break;
                    }
                    case DEPOSIT: {
                        mDeposits.add(value);
                        break;
                    }
                    case REMOVAL: {
                        mRemovals.add(value);
                        break;
                    }
                    case FEES: {
                        mFees.add(value);
                        this.fees.add(new TransactionPair<AccountTransaction>(account, accountTransaction));
                        feesBySecurity.computeIfAbsent(accountTransaction.getSecurity(), s -> MutableMoney.of(termCurrency)).add(value);
                        break;
                    }
                    case FEES_REFUND: {
                        mFees.subtract(value);
                        this.fees.add(new TransactionPair<AccountTransaction>(account, accountTransaction));
                        feesBySecurity.computeIfAbsent(accountTransaction.getSecurity(), s -> MutableMoney.of(termCurrency)).subtract(value);
                        break;
                    }
                    case TAXES: {
                        mTaxes.add(value);
                        this.taxes.add(new TransactionPair<AccountTransaction>(account, accountTransaction));
                        taxesBySecurity.computeIfAbsent(accountTransaction.getSecurity(), s -> MutableMoney.of(termCurrency)).add(value);
                        break;
                    }
                    case TAX_REFUND: {
                        mTaxes.subtract(value);
                        this.taxes.add(new TransactionPair<AccountTransaction>(account, accountTransaction));
                        taxesBySecurity.computeIfAbsent(accountTransaction.getSecurity(), s -> MutableMoney.of(termCurrency)).subtract(value);
                        break;
                    }
                    case BUY: 
                    case SELL: 
                    case TRANSFER_IN: 
                    case TRANSFER_OUT: {
                        break;
                    }
                    default: {
                        throw new UnsupportedOperationException();
                    }
                }
            }
        }
        for (Portfolio portfolio : this.client.getPortfolios()) {
            for (PortfolioTransaction portfolioTransaction : portfolio.getTransactions()) {
                if (!this.period.contains(portfolioTransaction.getDateTime())) continue;
                Money unit = portfolioTransaction.getUnitSum(Transaction.Unit.Type.FEE, this.converter);
                if (!unit.isZero()) {
                    mFees.add(unit);
                    this.fees.add(new TransactionPair<PortfolioTransaction>(portfolio, portfolioTransaction));
                    feesBySecurity.computeIfAbsent(portfolioTransaction.getSecurity(), s -> MutableMoney.of(termCurrency)).add(unit);
                }
                if (!(unit = portfolioTransaction.getUnitSum(Transaction.Unit.Type.TAX, this.converter)).isZero()) {
                    mTaxes.add(unit);
                    this.taxes.add(new TransactionPair<PortfolioTransaction>(portfolio, portfolioTransaction));
                    taxesBySecurity.computeIfAbsent(portfolioTransaction.getSecurity(), s -> MutableMoney.of(termCurrency)).add(unit);
                }
                switch (portfolioTransaction.getType()) {
                    case DELIVERY_INBOUND: {
                        mDeposits.add(portfolioTransaction.getMonetaryAmount().with(this.converter.at(portfolioTransaction.getDateTime())));
                        break;
                    }
                    case DELIVERY_OUTBOUND: {
                        mRemovals.add(portfolioTransaction.getMonetaryAmount().with(this.converter.at(portfolioTransaction.getDateTime())));
                        break;
                    }
                    case BUY: 
                    case SELL: 
                    case TRANSFER_IN: 
                    case TRANSFER_OUT: {
                        break;
                    }
                    default: {
                        throw new UnsupportedOperationException();
                    }
                }
            }
        }
        BiFunction<Map, String, List> asPositions = (map, otherLabel) -> map.entrySet().stream().filter(entry -> !((MutableMoney)entry.getValue()).isZero()).map(entry -> entry.getKey() == null ? new Position((String)otherLabel, ((MutableMoney)entry.getValue()).toMoney(), null) : new Position((Security)entry.getKey(), ((MutableMoney)entry.getValue()).toMoney(), null)).sorted((p1, p2) -> {
            if (p1.getSecurity() == null) {
                return p2.getSecurity() == null ? 0 : 1;
            }
            if (p2.getSecurity() == null) {
                return -1;
            }
            return p1.getLabel().compareToIgnoreCase(p2.getLabel());
        }).collect(Collectors.toList());
        Category earningsCategory = this.categories.get((Object)CategoryType.EARNINGS);
        earningsCategory.valuation = mEarnings.toMoney();
        earningsCategory.positions = asPositions.apply(earningsBySecurity, Messages.LabelInterest);
        this.categories.get((Object)CategoryType.FEES).valuation = mFees.toMoney();
        this.categories.get((Object)CategoryType.FEES).positions = asPositions.apply(feesBySecurity, Messages.LabelOtherCategory);
        this.categories.get((Object)CategoryType.TAXES).valuation = mTaxes.toMoney();
        this.categories.get((Object)CategoryType.TAXES).positions = asPositions.apply(taxesBySecurity, Messages.LabelOtherCategory);
        this.categories.get((Object)CategoryType.TRANSFERS).valuation = mDeposits.toMoney().subtract(mRemovals.toMoney());
        this.categories.get((Object)CategoryType.TRANSFERS).positions.add(new Position(Messages.LabelDeposits, mDeposits.toMoney(), null));
        this.categories.get((Object)CategoryType.TRANSFERS).positions.add(new Position(Messages.LabelRemovals, mRemovals.toMoney(), null));
    }

    private void addEarningTransaction(Account account, AccountTransaction transaction, MutableMoney mEarnings, Map<Security, MutableMoney> earningsBySecurity, MutableMoney mFees, MutableMoney mTaxes, Map<Security, MutableMoney> feesBySecurity, Map<Security, MutableMoney> taxesBySecurity) {
        Money tax;
        Money earned = transaction.getGrossValue().with(this.converter.at(transaction.getDateTime()));
        mEarnings.add(earned);
        this.earnings.add(new TransactionPair<AccountTransaction>(account, transaction));
        earningsBySecurity.computeIfAbsent(transaction.getSecurity(), k -> MutableMoney.of(this.converter.getTermCurrency())).add(earned);
        Money fee = transaction.getUnitSum(Transaction.Unit.Type.FEE, this.converter).with(this.converter.at(transaction.getDateTime()));
        if (!fee.isZero()) {
            mFees.add(fee);
            this.fees.add(new TransactionPair<AccountTransaction>(account, transaction));
            feesBySecurity.computeIfAbsent(transaction.getSecurity(), s -> MutableMoney.of(this.converter.getTermCurrency())).add(fee);
        }
        if (!(tax = transaction.getUnitSum(Transaction.Unit.Type.TAX, this.converter).with(this.converter.at(transaction.getDateTime()))).isZero()) {
            mTaxes.add(tax);
            this.taxes.add(new TransactionPair<AccountTransaction>(account, transaction));
            taxesBySecurity.computeIfAbsent(transaction.getSecurity(), s -> MutableMoney.of(this.converter.getTermCurrency())).add(tax);
        }
    }

    private void addCurrencyGains() {
        HashMap<String, MutableMoney> currency2money = new HashMap<String, MutableMoney>();
        for (AccountSnapshot snapshot : this.snapshotStart.getAccounts()) {
            if (this.converter.getTermCurrency().equals(snapshot.getAccount().getCurrencyCode())) continue;
            MutableMoney value = currency2money.computeIfAbsent(snapshot.getAccount().getCurrencyCode(), c -> MutableMoney.of(this.converter.getTermCurrency()));
            value.subtract(snapshot.getFunds());
            for (AccountTransaction t : snapshot.getAccount().getTransactions()) {
                if (!this.period.contains(t.getDateTime())) continue;
                switch (t.getType()) {
                    case DEPOSIT: 
                    case INTEREST: 
                    case DIVIDENDS: 
                    case FEES_REFUND: 
                    case TAX_REFUND: 
                    case SELL: {
                        value.subtract(t.getMonetaryAmount().with(this.converter.at(t.getDateTime())));
                        break;
                    }
                    case REMOVAL: 
                    case INTEREST_CHARGE: 
                    case FEES: 
                    case TAXES: 
                    case BUY: {
                        value.add(t.getMonetaryAmount().with(this.converter.at(t.getDateTime())));
                        break;
                    }
                    case TRANSFER_IN: {
                        value.subtract(this.determineTransferAmount(t));
                        break;
                    }
                    case TRANSFER_OUT: {
                        value.add(this.determineTransferAmount(t));
                        break;
                    }
                    default: {
                        throw new UnsupportedOperationException();
                    }
                }
            }
        }
        for (AccountSnapshot snapshot : this.snapshotEnd.getAccounts()) {
            if (this.converter.getTermCurrency().equals(snapshot.getAccount().getCurrencyCode())) continue;
            currency2money.computeIfAbsent(snapshot.getAccount().getCurrencyCode(), c -> MutableMoney.of(this.converter.getTermCurrency())).add(snapshot.getFunds());
        }
        Category currencyGains = this.categories.get((Object)CategoryType.CURRENCY_GAINS);
        currency2money.forEach((currency, money) -> {
            currencyGains.valuation = currencyGains.valuation.add(money.toMoney());
            currencyGains.positions.add(new Position((String)currency, money.toMoney(), null));
        });
        Collections.sort(currencyGains.positions, (p1, p2) -> p1.getLabel().compareTo(p2.getLabel()));
    }

    private Money determineTransferAmount(AccountTransaction t) {
        if (this.converter.getTermCurrency().equals(t.getCurrencyCode())) {
            return t.getMonetaryAmount();
        }
        Transaction other = t.getCrossEntry().getCrossTransaction(t);
        if (this.converter.getTermCurrency().equals(other.getCurrencyCode())) {
            return other.getMonetaryAmount();
        }
        MutableMoney m = MutableMoney.of(this.converter.getTermCurrency());
        m.add(t.getMonetaryAmount().with(this.converter.at(t.getDateTime())));
        m.add(other.getMonetaryAmount().with(this.converter.at(t.getDateTime())));
        return m.divide(2.0).toMoney();
    }

    public static class Category {
        private List<Position> positions = new ArrayList<Position>();
        private String label;
        private String sign;
        private Money valuation;

        public Category(String label, String sign, Money valuation) {
            this.label = label;
            this.sign = sign;
            this.valuation = valuation;
        }

        public Money getValuation() {
            return this.valuation;
        }

        public String getLabel() {
            return this.label;
        }

        public String getSign() {
            return this.sign;
        }

        public List<Position> getPositions() {
            return this.positions;
        }
    }

    public static enum CategoryType {
        INITIAL_VALUE,
        CAPITAL_GAINS,
        REALIZED_CAPITAL_GAINS,
        EARNINGS,
        FEES,
        TAXES,
        CURRENCY_GAINS,
        TRANSFERS,
        FINAL_VALUE;

    }

    public static class Position
    implements TrailProvider {
        public static final String TRAIL_VALUE = "value";
        public static final String TRAIL_FOREX_GAIN = "forexGain";
        private final String label;
        private final Security security;
        private final Money value;
        private final TrailRecord valueTrail;
        private final Money forexGain;
        private final TrailRecord forexGainTrail;

        private Position(Security security, Money value, TrailRecord trail) {
            this(security.getName(), security, value, trail, null, null);
        }

        private Position(Security security, Money value, TrailRecord trail, Money forexGain, TrailRecord forexGainTrail) {
            this(security.getName(), security, value, trail, forexGain, forexGainTrail);
        }

        private Position(String label, Money value, TrailRecord trail) {
            this(label, null, value, trail, null, null);
        }

        private Position(String label, Security security, Money value, TrailRecord valueTrail, Money forexGain, TrailRecord forexGainTrail) {
            this.label = label;
            this.security = security;
            this.value = value;
            this.valueTrail = valueTrail;
            this.forexGain = forexGain;
            this.forexGainTrail = forexGainTrail;
        }

        public Money getValue() {
            return this.value;
        }

        public String getLabel() {
            return this.label;
        }

        public Security getSecurity() {
            return this.security;
        }

        public Money getForexGain() {
            return this.forexGain;
        }

        @Override
        public Optional<Trail> explain(String key) {
            switch (key) {
                case "value": {
                    return Trail.of(this.label, this.valueTrail);
                }
                case "forexGain": {
                    return Trail.of(this.label, this.forexGainTrail);
                }
            }
            return Optional.empty();
        }
    }
}

