/*
 * Decompiled with CFR 0.152.
 */
package name.abuchen.portfolio.datatransfer;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import name.abuchen.portfolio.Messages;
import name.abuchen.portfolio.datatransfer.Extractor;
import name.abuchen.portfolio.datatransfer.SecurityCache;
import name.abuchen.portfolio.model.AccountTransaction;
import name.abuchen.portfolio.model.BuySellEntry;
import name.abuchen.portfolio.model.Client;
import name.abuchen.portfolio.model.PortfolioTransaction;
import name.abuchen.portfolio.model.Security;
import name.abuchen.portfolio.model.Transaction;
import name.abuchen.portfolio.money.CurrencyUnit;
import name.abuchen.portfolio.money.Money;
import name.abuchen.portfolio.money.Values;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class IBFlexStatementExtractor
implements Extractor {
    private final Client client;
    private List<Security> allSecurities;
    private Map<String, String> exchanges;

    public IBFlexStatementExtractor(Client client) {
        this.client = client;
        this.allSecurities = new ArrayList<Security>(client.getSecurities());
        this.exchanges = new HashMap<String, String>();
        this.exchanges.put("EBS", "SW");
        this.exchanges.put("LSE", "L");
        this.exchanges.put("SWX", "SW");
        this.exchanges.put("TSE", "TO");
        this.exchanges.put("VENTURE", "V");
        this.exchanges.put("IBIS", "DE");
        this.exchanges.put("TGATE", "DE");
        this.exchanges.put("SWB", "SG");
        this.exchanges.put("FWB", "F");
    }

    public Client getClient() {
        return this.client;
    }

    private LocalDateTime convertDate(String date) throws DateTimeParseException {
        if (date.length() > 8) {
            return LocalDate.parse(date).atStartOfDay();
        }
        return LocalDate.parse(date, DateTimeFormatter.ofPattern("yyyyMMdd")).atStartOfDay();
    }

    private LocalDateTime convertDate(String date, String time) throws DateTimeParseException {
        return LocalDateTime.parse(String.format("%s %s", date, time), DateTimeFormatter.ofPattern("yyyyMMdd HHmmss")).withSecond(0).withNano(0);
    }

    IBFlexStatementExtractorResult importActivityStatement(InputStream f) {
        IBFlexStatementExtractorResult result = new IBFlexStatementExtractorResult();
        try {
            DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
            dbFactory.setFeature("http://javax.xml.XMLConstants/feature/secure-processing", true);
            DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
            Document doc = dBuilder.parse(f);
            doc.getDocumentElement().normalize();
            result.parseDocument(doc);
        }
        catch (IOException | ParserConfigurationException | SAXException e) {
            result.addError(e);
        }
        return result;
    }

    private String asCurrencyUnit(String currency) {
        if (currency == null) {
            return "EUR";
        }
        CurrencyUnit unit = CurrencyUnit.getInstance(currency.trim());
        return unit == null ? "EUR" : unit.getCurrencyCode();
    }

    @Override
    public String getLabel() {
        return Messages.IBXML_Label;
    }

    @Override
    public List<Extractor.Item> extract(SecurityCache securityCache, Extractor.InputFile inputFile, List<Exception> errors) {
        try {
            Throwable throwable = null;
            Object var5_7 = null;
            try (FileInputStream in = new FileInputStream(inputFile.getFile());){
                IBFlexStatementExtractorResult result = this.importActivityStatement(in);
                errors.addAll(result.getErrors());
                return result.getResults();
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            errors.add(e);
            return Collections.emptyList();
        }
    }

    private class IBFlexStatementExtractorResult {
        private Document document;
        private List<Exception> errors = new ArrayList<Exception>();
        private List<Extractor.Item> results = new ArrayList<Extractor.Item>();
        private String ibAccountCurrency = null;
        private Function<Element, Extractor.Item> importAccountInformation = element -> {
            CurrencyUnit currencyUnit;
            String currency = IBFlexStatementExtractor.this.asCurrencyUnit(element.getAttribute("currency"));
            if (currency != null && !currency.isEmpty() && (currencyUnit = CurrencyUnit.getInstance(currency)) != null) {
                this.ibAccountCurrency = currency;
            }
            return null;
        };
        private Function<Element, Extractor.Item> buildAccountTransaction = element -> {
            AccountTransaction transaction = new AccountTransaction();
            if (element.hasAttribute("reportDate")) {
                if (element.getAttribute("reportDate").length() == 15) {
                    transaction.setDateTime(IBFlexStatementExtractor.this.convertDate(element.getAttribute("reportDate").substring(0, 8)));
                } else {
                    transaction.setDateTime(IBFlexStatementExtractor.this.convertDate(element.getAttribute("reportDate")));
                }
            } else if (element.getAttribute("dateTime").length() == 15) {
                transaction.setDateTime(IBFlexStatementExtractor.this.convertDate(element.getAttribute("dateTime").substring(0, 8)));
            } else {
                transaction.setDateTime(IBFlexStatementExtractor.this.convertDate(element.getAttribute("dateTime")));
            }
            Double amount = Double.parseDouble(element.getAttribute("amount"));
            String currency = IBFlexStatementExtractor.this.asCurrencyUnit(element.getAttribute("currency"));
            String type = element.getAttribute("type");
            if (type.equals("Deposits") || type.equals("Deposits & Withdrawals") || type.equals("Deposits/Withdrawals")) {
                if (amount >= 0.0) {
                    transaction.setType(AccountTransaction.Type.DEPOSIT);
                } else {
                    transaction.setType(AccountTransaction.Type.REMOVAL);
                }
            } else if (type.equals("Dividends") || type.equals("Payment In Lieu Of Dividends")) {
                transaction.setType(AccountTransaction.Type.DIVIDENDS);
                if (element.getAttribute("symbol").length() > 0) {
                    transaction.setSecurity(this.getOrCreateSecurity((Element)element, true));
                }
                this.calculateShares(transaction, (Element)element);
            } else if (type.equals("Withholding Tax")) {
                if (element.getAttribute("symbol").length() > 0) {
                    transaction.setSecurity(this.getOrCreateSecurity((Element)element, true));
                }
                if (amount <= 0.0) {
                    transaction.setType(AccountTransaction.Type.TAXES);
                } else {
                    transaction.setType(AccountTransaction.Type.TAX_REFUND);
                }
            } else if (type.equals("Broker Interest Received")) {
                transaction.setType(AccountTransaction.Type.INTEREST);
            } else if (type.equals("Broker Interest Paid")) {
                transaction.setType(AccountTransaction.Type.INTEREST_CHARGE);
            } else if (type.equals("Other Fees")) {
                if (amount <= 0.0) {
                    transaction.setType(AccountTransaction.Type.FEES);
                } else {
                    transaction.setType(AccountTransaction.Type.FEES_REFUND);
                }
            } else {
                throw new IllegalArgumentException();
            }
            amount = Math.abs(amount);
            this.setAmount((Element)element, transaction, amount, currency);
            transaction.setNote(element.getAttribute("description"));
            return new Extractor.TransactionItem(transaction);
        };
        private Function<Element, Extractor.Item> buildPortfolioTransaction = element -> {
            String assetCategory = element.getAttribute("assetCategory");
            if (!Arrays.asList("STK", "OPT").contains(assetCategory)) {
                return null;
            }
            BuySellEntry transaction = new BuySellEntry();
            if (element.getAttribute("buySell").equals("BUY")) {
                transaction.setType(PortfolioTransaction.Type.BUY);
            } else if (element.getAttribute("buySell").equals("SELL")) {
                transaction.setType(PortfolioTransaction.Type.SELL);
            } else {
                throw new IllegalArgumentException();
            }
            if (element.hasAttribute("dateTime")) {
                transaction.setDate(IBFlexStatementExtractor.this.convertDate(element.getAttribute("dateTime").substring(0, 8), element.getAttribute("dateTime").substring(9, 15)));
            } else if (element.hasAttribute("tradeTime")) {
                transaction.setDate(IBFlexStatementExtractor.this.convertDate(element.getAttribute("tradeDate"), element.getAttribute("tradeTime")));
            } else {
                transaction.setDate(IBFlexStatementExtractor.this.convertDate(element.getAttribute("tradeDate"), "000000"));
            }
            String currency = IBFlexStatementExtractor.this.asCurrencyUnit(element.getAttribute("currency"));
            Double amount = Math.abs(Double.parseDouble(element.getAttribute("netCash")));
            this.setAmount((Element)element, transaction.getPortfolioTransaction(), amount, currency);
            this.setAmount((Element)element, transaction.getAccountTransaction(), amount, currency, false);
            Double qty = Math.abs(Double.parseDouble(element.getAttribute("quantity")));
            Double multiplier = Double.parseDouble(Optional.ofNullable(element.getAttribute("multiplier")).orElse("1"));
            transaction.setShares(Math.round(qty * (double)Values.Share.factor() * multiplier));
            double fees = Math.abs(Double.parseDouble(element.getAttribute("ibCommission")));
            String feesCurrency = IBFlexStatementExtractor.this.asCurrencyUnit(element.getAttribute("ibCommissionCurrency"));
            Transaction.Unit feeUnit = this.createUnit((Element)element, Transaction.Unit.Type.FEE, fees, feesCurrency);
            transaction.getPortfolioTransaction().addUnit(feeUnit);
            double taxes = Math.abs(Double.parseDouble(element.getAttribute("taxes")));
            Transaction.Unit taxUnit = this.createUnit((Element)element, Transaction.Unit.Type.TAX, taxes, currency);
            transaction.getPortfolioTransaction().addUnit(taxUnit);
            transaction.setSecurity(this.getOrCreateSecurity((Element)element, true));
            transaction.setNote(element.getAttribute("description"));
            return new Extractor.BuySellEntryItem(transaction);
        };
        private Function<Element, Extractor.Item> buildCorporateTransaction = eElement -> {
            Money proceeds = Money.of(IBFlexStatementExtractor.this.asCurrencyUnit(eElement.getAttribute("currency")), Values.Amount.factorize(Double.parseDouble(eElement.getAttribute("proceeds"))));
            if (!proceeds.isZero()) {
                BuySellEntry transaction = new BuySellEntry();
                if (Double.parseDouble(eElement.getAttribute("quantity")) >= 0.0) {
                    transaction.setType(PortfolioTransaction.Type.BUY);
                } else {
                    transaction.setType(PortfolioTransaction.Type.SELL);
                }
                transaction.setDate(IBFlexStatementExtractor.this.convertDate(eElement.getAttribute("reportDate")));
                double qty = Math.abs(Double.parseDouble(eElement.getAttribute("quantity")));
                transaction.setShares(Values.Share.factorize(qty));
                transaction.setSecurity(this.getOrCreateSecurity((Element)eElement, true));
                transaction.setNote(eElement.getAttribute("description"));
                transaction.setMonetaryAmount(proceeds);
                return new Extractor.BuySellEntryItem(transaction);
            }
            PortfolioTransaction transaction = new PortfolioTransaction();
            if (Double.parseDouble(eElement.getAttribute("quantity")) >= 0.0) {
                transaction.setType(PortfolioTransaction.Type.DELIVERY_INBOUND);
            } else {
                transaction.setType(PortfolioTransaction.Type.DELIVERY_OUTBOUND);
            }
            transaction.setDateTime(IBFlexStatementExtractor.this.convertDate(eElement.getAttribute("reportDate")));
            Double qty = Math.abs(Double.parseDouble(eElement.getAttribute("quantity")));
            transaction.setShares(Math.round(qty * (double)Values.Share.factor()));
            transaction.setSecurity(this.getOrCreateSecurity((Element)eElement, true));
            transaction.setNote(eElement.getAttribute("description"));
            transaction.setMonetaryAmount(proceeds);
            return new Extractor.TransactionItem(transaction);
        };

        private IBFlexStatementExtractorResult() {
        }

        private Transaction.Unit createUnit(Element element, Transaction.Unit.Type unitType, Double amount, String currency) {
            Transaction.Unit unit;
            if (this.ibAccountCurrency == null || this.ibAccountCurrency.equals(currency)) {
                unit = new Transaction.Unit(unitType, Money.of(currency, Values.Amount.factorize(amount)));
            } else {
                String fxRateToBaseString = element.getAttribute("fxRateToBase");
                BigDecimal fxRateToBase = fxRateToBaseString != null && !fxRateToBaseString.isEmpty() ? new BigDecimal(fxRateToBaseString).setScale(4, RoundingMode.HALF_DOWN) : new BigDecimal("1.0000");
                BigDecimal inverseRate = BigDecimal.ONE.divide(fxRateToBase, 10, RoundingMode.HALF_DOWN);
                BigDecimal baseCurrencyMoney = BigDecimal.valueOf(amount).setScale(2, RoundingMode.HALF_DOWN).divide(inverseRate, RoundingMode.HALF_DOWN);
                unit = new Transaction.Unit(unitType, Money.of(this.ibAccountCurrency, Math.round(baseCurrencyMoney.doubleValue() * (double)Values.Amount.factor())), Money.of(currency, Values.Amount.factorize(amount)), fxRateToBase);
            }
            return unit;
        }

        private void setAmount(Element element, Transaction transaction, Double amount, String currency) {
            this.setAmount(element, transaction, amount, currency, true);
        }

        private void setAmount(Element element, Transaction transaction, Double amount, String currency, boolean addUnit) {
            if (this.ibAccountCurrency != null && !this.ibAccountCurrency.equals(currency)) {
                String fxRateToBaseString = element.getAttribute("fxRateToBase");
                BigDecimal fxRateToBase = fxRateToBaseString != null && !fxRateToBaseString.isEmpty() ? BigDecimal.valueOf(Double.parseDouble(fxRateToBaseString)) : new BigDecimal(1);
                BigDecimal inverseRate = BigDecimal.ONE.divide(fxRateToBase, 10, RoundingMode.HALF_DOWN);
                BigDecimal baseCurrencyMoney = BigDecimal.valueOf(amount * (double)Values.Amount.factor()).divide(inverseRate, RoundingMode.HALF_DOWN);
                transaction.setAmount(Math.round(baseCurrencyMoney.doubleValue()));
                transaction.setCurrencyCode(this.ibAccountCurrency);
                if (addUnit) {
                    Transaction.Unit grossValue = new Transaction.Unit(Transaction.Unit.Type.GROSS_VALUE, transaction.getMonetaryAmount(), Money.of(currency, Math.round(amount * (double)Values.Amount.factor())), fxRateToBase);
                    transaction.addUnit(grossValue);
                }
            } else {
                transaction.setAmount(Math.round(amount * (double)Values.Amount.factor()));
                transaction.setCurrencyCode(currency);
            }
        }

        private void importModelObjects(String type, Function<Element, Extractor.Item> handleNodeFunction) {
            NodeList nList = this.document.getElementsByTagName(type);
            int temp = 0;
            while (temp < nList.getLength()) {
                Node nNode = nList.item(temp);
                if (nNode.getNodeType() == 1) {
                    try {
                        Extractor.Item item = handleNodeFunction.apply((Element)nNode);
                        if (item != null) {
                            this.results.add(item);
                        }
                    }
                    catch (Exception e) {
                        this.errors.add(e);
                    }
                }
                ++temp;
            }
        }

        public void parseDocument(Document doc) {
            this.document = doc;
            if (this.document == null) {
                return;
            }
            this.importModelObjects("AccountInformation", this.importAccountInformation);
            this.importModelObjects("Trade", this.buildPortfolioTransaction);
            this.importModelObjects("CashTransaction", this.buildAccountTransaction);
            this.importModelObjects("CorporateAction", this.buildCorporateTransaction);
        }

        public void addError(Exception e) {
            this.errors.add(e);
        }

        private Security getOrCreateSecurity(Element element, boolean doCreate) {
            Optional<String> tickerSymbol = Optional.ofNullable(element.getAttribute("symbol"));
            String assetCategory = element.getAttribute("assetCategory");
            String exchange = element.getAttribute("exchange");
            String quoteFeed = "MANUAL";
            String currency = IBFlexStatementExtractor.this.asCurrencyUnit(element.getAttribute("currency"));
            String isin = element.getAttribute("isin");
            String cusip = element.getAttribute("cusip");
            Optional<String> computedTickerSymbol = tickerSymbol.map(t -> t.replaceAll(" ", "-"));
            if (isin.length() == 0 && cusip.length() > 0) {
                isin = cusip;
            }
            String conID = element.getAttribute("conid");
            String description = element.getAttribute("description");
            if ("OPT".equals(assetCategory) && (computedTickerSymbol = tickerSymbol.map(t -> t.replaceAll("\\s+", ""))).filter(p -> p.matches(".*\\d{6}[CP]\\d{8}")).isPresent()) {
                quoteFeed = "YAHOO";
            }
            if ("STK".equals(assetCategory)) {
                computedTickerSymbol = tickerSymbol;
                if (!"USD".equals(currency)) {
                    computedTickerSymbol = computedTickerSymbol.map(t -> t.replaceAll("[a-z]*$", ""));
                    if ("EUR".equals(currency)) {
                        computedTickerSymbol = computedTickerSymbol.map(t -> t.replaceAll("EUR$", ""));
                    }
                    if (tickerSymbol.isPresent() && IBFlexStatementExtractor.this.exchanges.containsKey(exchange)) {
                        computedTickerSymbol = computedTickerSymbol.map(t -> String.valueOf(t) + '.' + (String)IBFlexStatementExtractor.this.exchanges.get(exchange));
                    }
                }
                quoteFeed = "ALPHAVANTAGE";
            }
            Security s2 = null;
            for (Security s : IBFlexStatementExtractor.this.allSecurities) {
                if (conID != null && conID.length() > 0 && conID.equals(s.getWkn())) {
                    return s;
                }
                if (isin.length() > 0 && isin.equals(s.getIsin())) {
                    if (currency.equals(s.getCurrencyCode())) {
                        return s;
                    }
                    s2 = s;
                }
                if (!computedTickerSymbol.isPresent() || !computedTickerSymbol.get().equals(s.getTickerSymbol())) continue;
                return s;
            }
            if (s2 != null) {
                return s2;
            }
            if (!doCreate) {
                return null;
            }
            Security security = new Security(description, isin, computedTickerSymbol.orElse(null), quoteFeed);
            security.setWkn(conID);
            security.setCurrencyCode(currency);
            security.setNote(description);
            IBFlexStatementExtractor.this.allSecurities.add(security);
            Extractor.SecurityItem item = new Extractor.SecurityItem(security);
            this.results.add(item);
            return security;
        }

        private void calculateShares(Transaction transaction, Element element) {
            long numShares = 0L;
            String desc = element.getAttribute("description");
            double amount = Double.parseDouble(element.getAttribute("amount"));
            Pattern dividendPattern = Pattern.compile(".*DIVIDEND.* ([0-9]*\\.[0-9]*) .*");
            Matcher tagmatch = dividendPattern.matcher(desc);
            if (tagmatch.find()) {
                double dividend = Double.parseDouble(tagmatch.group(1));
                numShares = Math.round(amount / dividend) * (long)Values.Share.factor();
            }
            transaction.setShares(numShares);
        }

        public List<Exception> getErrors() {
            return this.errors;
        }

        public List<Extractor.Item> getResults() {
            return this.results;
        }
    }
}

