__version__   = "$Revision: 1.2 $"[11:-2]
__copyright__ = """Copyright (c) 2003 Not Another Corporation Incorporated
                   www.notanothercorporation.com"""
__license__   = """Licensed under the GNU LGPL

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""
__doc__ = """
** %(name)s **
File version %(version)s::
 
    %(copyright)s

    %(license)s

Part of the C{mongoose} package, provides some sample reporters.

TODO: Move base C{Reporter} class to here?

TODO: Add unit tests.

I{* $Id: reporters.py,v 1.2 2003/09/24 17:03:31 philiplindsay Exp $ *}
""" % {'name':__name__, 'version':__version__, 'copyright':__copyright__,
       'license':__license__}

import sys

import mongoose


def _wrap(text, width = 80):
    """
    A word-wrap function that preserves existing line breaks and most
    spaces in the text. Expects that existing line breaks are
    posix newlines ().

    From: U{http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/148061}
      - Added default width value.
    """
    return reduce(lambda line, word, width=width: '%s%s%s' %
                  (line,
                   ' \n'[(len(line[line.rfind('\n')+1:])
                         + len(word.split('\n',1)[0]
                              ) >= width)],
                   word),
                  text.split(' ')
                 )


class ConsoleReporter(mongoose.Reporter):
    """
    A reporter which outputs to a console's standard error (C{stderr}).

    Sub-classers should override C{getMessageString()}.
    """

    MESSAGE_TEXT = "\nThis program has encountered a problem that means it can not continue.\n\nPlease contact your technical support provider, system administrator or the vendor of this software and ask them for assistance.\n\nMongoose Incident Identifier: %s\n"
    def getMessageString(self, uhx, sessionInfo):
        """
        Returns a basic message string.

        Sub-classers can either completely override this method or use
        its output in the construction of their message string.

        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        return self.MESSAGE_TEXT % (uhx.incidentId)

        
    def process(self, uhx, sessionInfo):
        """
        Displays a message string on standard error (C{stderr}).

        TODO: Wrap to the actual width of the tty.

        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        print >> sys.stderr, _wrap(self.getMessageString(uhx, sessionInfo))



_LINE_FORMAT = "%s\t%s\t%s\t%s\n"
def _formatLogEntry_Short(uhx, sessionInfo):
    """
    Returns a string formatted in the "short" log entry style.

    @param uhx: An C{UnhandledException} instance.
    @param sessionInfo: A C{SessionInfo} instance.
    """
    return _LINE_FORMAT % (uhx.when, uhx.incidentId, uhx.type, uhx.value)



_ENTRY_FORMAT = "-" * 80 + "\n" + _LINE_FORMAT + "\n%s\n%s\n" + "-" * 80 + "\n"
def _formatLogEntry_Long(uhx, sessionInfo):
    """
    Returns a string formatted in the "long" log entry style.

    @param uhx: An C{UnhandledException} instance.
    @param sessionInfo: A C{SessionInfo} instance.
    """
    return _ENTRY_FORMAT % (uhx.when, uhx.incidentId, uhx.type, uhx.value,
                            uhx.formattedException, sessionInfo)



class FileLogReporter(mongoose.Reporter):
    """
    Reporter that appends a log entry to a file.

    Sub-classes can override the C{getMessageString_*()} methods or
    add new formats by adding a method with a name of the form
    C{getMessageString_<format>()}.

    NOTE: At present file locking is not used.
    TODO: Use file locking.
      - See U{http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/65203}
        for possible starting point. NOTE: Needs to work on Win95/98.
    """
    FORMAT_LONG = "Long"
    FORMAT_SHORT = "Short"

    HANDLER_PREFIX = "getMessageString_"

    format = ""
    logFilename = ""

    _DEBUG = False


    def __init__(self, logFilename, format = FORMAT_LONG, _DEBUG = False):
        """
        Initialise the reporter.
        
        TODO: Check C{logFilename} is valid (not necessarily exists)?
        TODO: Check C{format} is legal?

        @param logFilename: The full path name of the file.
        @param format: The format of the log entry. For the base
                       C{FileLogReporter} this can be either:
                       C{FileLogReporter.FORMAT_LONG} or
                       C{FileLogReporter.FORMAT_SHORT}. (Support is optional.)
        """
        self._DEBUG = _DEBUG
        
        self.format = format
        self.logFilename = logFilename
        

    def getMessageString_Short(self, uhx, sessionInfo):
        """
        Returns a message string in the "short" one-line style.

        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        return _formatLogEntry_Short(uhx, sessionInfo)
    

    def getMessageString_Long(self, uhx, sessionInfo):
        """
        Returns a message string in the "long" multi-line style.

        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        return _formatLogEntry_Long(uhx, sessionInfo)


    def _getMessageHandler(self, format):
        """
        Returns a method which will format a message in the specified style.

        At present the methods have a signature of::

          getMessageString_<format name>(self, uhx, sessionInfo)

        If the requested handler can not be found the default of
        C{getMessageString_Short()} is used.

        @param format: A string specifying the desired format.
        """
        return getattr(self, self.HANDLER_PREFIX + format,
                       self.getMessageString_Short)


    def getMessageString(self, uhx, sessionInfo):
        """
        Returns a log message string, ready for output to a file, in the format
        specified at the creation of the C{FileLogReporter} instance.
        
        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        return self._getMessageHandler(self.format)(uhx, sessionInfo)


    def process(self, uhx, sessionInfo):
        """
        Appends a log entry to the log file specified at the creation
        of the C{FileLogReporter} instance.
        
        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.        
        """
        hLogFile = None
        
        try:
            hLogFile = file(self.logFilename, "a")

            hLogFile.write(self.getMessageString(uhx, sessionInfo))
            
        except:
            if self._DEBUG:
                raise

        try:
            if hLogFile:
                hLogFile.close()
        except:
            if self._DEBUG:
                raise



class EmailReporter(mongoose.Reporter):
    """
    Reporter that emails an incident report to one or more recipients.

    NOTE: This reporter does I{not} request permission from the user before
    emailing the report.
    """

    smtpServer = ""
    fromAddress = ""
    toAddresses = []
    subjectLine = ""
    messagePrologue = ""

    _DEBUG = False

    
    def __init__(self, smtpServer, fromAddress, toAddresses, subjectLine,
                 messagePrologue = "", _DEBUG = False):
        """
        Initialise the reporter.

        @param smtpServer: Hostname of the SMTP server. If the
                           hostname ends with a colon (":") followed by a
                           number, that suffix will be stripped off and the
                           number interpreted as the port number to use.
        @param fromAddress: The single address that the email will appear to be
                            sent from.
        @param toAddresses: A list of one or more recipent addresses 
        @param subjectLine: The subject of the email.
        @param messagePrologue: Optional text to be insert before the
                                automatically generated message body.
        """

        self.smtpServer = smtpServer
        self.fromAddress = fromAddress
        self.toAddresses = toAddresses
        self.subjectLine = subjectLine
        self.messagePrologue = messagePrologue

        self._DEBUG = _DEBUG
    
    
    def getMessageString(self, uhx, sessionInfo):
        """
        Returns an incident report message string, ready for insertion into
        the body of an email.

        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        return _formatLogEntry_Long(uhx, sessionInfo)


    def process(self, uhx, sessionInfo):
        """
        Creates a RFC2822 compliant (hopefully!) mail message, connects
        to a smtp server and sends the message.
        
        TODO: Either insert or append incident id to subject line?
        TODO: Find a better way to close connection.

        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.        
        """
        # Prepare the message...
        messageHeader = "From: %s\r\n"\
                        "To: %s\r\n"\
                        "Subject: %s\r\n"\
                        "X-Mongoose-Id: %s\r\n\r\n" % \
                        (self.fromAddress,
                         ", ".join(self.toAddresses),
                         self.subjectLine,
                         uhx.incidentId)


        message = messageHeader + self.messagePrologue + \
                  self.getMessageString(uhx, sessionInfo)

        # Deliver the message...
        server = None

        try:
            server = smtplib.SMTP(self.smtpServer)
            # server.set_debuglevel(1)
            server.sendmail(self.fromAddress, self.toAddresses, message)
        except:
            if self._DEBUG:
                raise

        if server:
            # If the connection is open we want to close it...
            # (Using .quit() on a closed connection causes an error.)
            if getattr(server, "sock", False): # The 'sock' attribute is
                                               # undocumented...
                server.quit()
            

class GuiReporter(mongoose.Reporter):
    """
    A reporter which displays a wxWindows dialog to alert the user.

    Sub-classers should override C{getMessageString()} and, possibly,
    C{getLogEntry()}.
    
    """

    caption = ""

    MESSAGE_TEXT = "This program has encountered a problem that means it is unable to continue. \n\nPlease contact your technical support provider, system administrator or the vendor of this software and ask them for assistance. They may ask you to read to them some of the information shown below."


    def __init__(self, caption = "An error has occurred..."):
        """
        Initialise the reporter.

        @param caption: The dialog box caption.
        """
        self.caption = caption
        

    def getMessageString(self, uhx, sessionInfo):
        """
        Returns a basic message string.

        Sub-classers can either completely override this method or use
        its output in the construction of their message string.

        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        return self.MESSAGE_TEXT


    _LOG_FORMAT = "Mongoose Incident Identifier: %s\n\n"\
                  "Time: %s\n\n"\
                  "Type: %s\n\n"\
                  "Value: %s\n\n"\
                  "Session information\n"\
                  "------------------------------------\n"\
                  "%s\n"
    def getLogEntry(self, uhx, sessionInfo):
        """
        Return a pre-formatted log entry, suitable for display
        in the scroll box of the dialog.
        
        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        return self._LOG_FORMAT % (uhx.incidentId, uhx.when, uhx.type,
                                   uhx.value, sessionInfo)

        
    def process(self, uhx, sessionInfo):
        """
        Displays a wxWindows dialog to alert the user.

        @param uhx: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        from wxPython.wx import wxPySimpleApp, wxGetApp, wxDialog, \
             wxStaticText, wxTextCtrl, wxBoxSizer, wxButton, \
             wxALL, wxTE_READONLY, wxTE_MULTILINE, \
             wxDEFAULT_DIALOG_STYLE, wxRESIZE_BORDER, wxVERTICAL, \
             wxEXPAND, wxLEFT, wxRIGHT, wxBOTTOM, \
             wxALIGN_CENTRE_HORIZONTAL, wxALIGN_BOTTOM, wxALIGN_LEFT, \
             EVT_BUTTON


        class MongooseDialog(wxDialog):
            """
            Customised dialog box window.
            """
            def __init__(self, caption, staticMessage, scrollMessage):
                """
                Create the customised dialog box.
                """
                wxDialog.__init__(self, None, -1, caption,
                                  style = wxDEFAULT_DIALOG_STYLE |
                                  wxRESIZE_BORDER)


                sizer2 = wxBoxSizer(wxVERTICAL)

                # Create the control to display the static message... 
                staticTextCtrl = wxStaticText(self, -1, staticMessage)
                sizer2.Add(staticTextCtrl, 0,
                           flag = wxALIGN_LEFT | wxALL, border = 10)

                # Create the control to display the scrollable log... 
                scrollTextCtrl = wxTextCtrl(self, -1,
                                            scrollMessage,
                                            size = (100,100),
                                            style = wxTE_READONLY |
                                            wxTE_MULTILINE)
                scrollTextCtrl.SetSizeHints(100, 100)
                sizer2.Add(scrollTextCtrl, 1,
                           flag = wxEXPAND | wxLEFT | wxRIGHT | wxBOTTOM,
                           border = 10)

                # "OK" button.
                button = wxButton(self, -1, "OK")
                sizer2.Add(button, 0,
                           flag = wxALIGN_CENTRE_HORIZONTAL |
                           wxALIGN_BOTTOM | wxALL,
                           border = 10)

                EVT_BUTTON(self, button.GetId(), self.OnClick)
                # button.SetDefault() # Probably better if people have to
                                      # "manually" close this dialog.

                self.SetSizeHints(100, 100)

                self.SetSizer(sizer2)
                self.SetAutoLayout(True)
                sizer2.Fit(self)
                sizer2.SetSizeHints(self) # Sets minimum size.


            def OnClick(self, e):
                """
                Closes the dialog box.
                """
                self.Close()


        # Check if the wxWindows framework has already been initialised.
        if wxGetApp() is None:
            # TODO: Can we initialise less? Should we quit if no app exists?
            #       Can we be sure the app is functional? Can we totally
            #       destroy the old one instead?
            wxPySimpleApp()

        # Create & display the dialog.
        dialog = MongooseDialog(self.caption,
                                _wrap(self.getMessageString(uhx, sessionInfo),
                                      width = 60),
                                self.getLogEntry(uhx, sessionInfo))
        dialog.ShowModal()

        

if __name__ == "__main__":
    # Demonstrate the reporters...
    import os
    import time
    
    exctype = value = tb = None
    
    try:
        raise Exception("Dummy Exception")
    except Exception, e:
        pass

    exctype, value, tb = sys.exc_info()

    incidentTime = time.asctime()
    
    dummyUHX = mongoose.UnhandledException(exctype, value, tb, incidentTime)

    dummySessionInfo = mongoose.SessionInfo({'info one': 'Stuff',
                                             'info two': 'A value'})

    BANNER = "-------------------------------------------------------\n" \
             "  %s\n" \
             "-------------------------------------------------------\n"

    # -----------------------------------------------------------------
    print BANNER % ("ConsoleReporter")
    
    ConsoleReporter().processAndUnload(dummyUHX, dummySessionInfo)

    # -----------------------------------------------------------------
    print BANNER % ("FileLogReporter")

    TEMP_FILENAME = "temp.log"

    print "Writing log to files: %s, %s\n" % (TEMP_FILENAME, "l"+TEMP_FILENAME)

    FileLogReporter(TEMP_FILENAME, format = FileLogReporter.FORMAT_SHORT,
                    _DEBUG = True).processAndUnload(dummyUHX, dummySessionInfo)

    FileLogReporter("l" + TEMP_FILENAME, format = FileLogReporter.FORMAT_LONG,
                    _DEBUG = True).processAndUnload(dummyUHX, dummySessionInfo)


    # -----------------------------------------------------------------
    print BANNER % ("EmailReporter")
    try:
        import smtplib
    except:
        print "Skipping, can't import 'smtplib'..."
    else:
        if len(sys.argv) == 3:
            SMTP_SERVER = sys.argv[1]
            TO_ADDRESSES = [sys.argv[2]]
            FROM_ADDRESS = "mongoose@example.com"
            SUBJECT_LINE = "Mongoose Incident Report"

            MESSAGE_PROLOGUE = "(This is an optional message prologue.)\n"

            EmailReporter("las", FROM_ADDRESS, TO_ADDRESSES, SUBJECT_LINE,
                          messagePrologue = MESSAGE_PROLOGUE,
                          _DEBUG = True).processAndUnload(dummyUHX,
                                                          dummySessionInfo)
        else:
            print "Skipping, no smtp server or 'To' address supplied...\n\n"\
                  "Usage: python %s [<smtp server> <'To' address>]\n" % \
                  (sys.argv[0])
            

    # -----------------------------------------------------------------
    print BANNER % ("GuiReporter")
    try:
        import wxPython
    except:
        print "Skipping, can't import 'wxPython'...\n"
    else:
        GuiReporter().processAndUnload(dummyUHX, dummySessionInfo)
