
"""
Tests for gravity-style spatial interaction models

Test data is the Austria migration dataset used in Dennet's (2012) practical primer
on spatial interaction modeling. The data was made avialable through the
following dropbox link: http://dl.dropbox.com/u/8649795/AT_Austria.csv.
The data has been pre-filtered so that there are no intra-zonal flows,

Dennett, A. (2012). Estimating flows between geographical locations:get me
started in spatial interaction modelling (Working Paper No. 184). UCL: Citeseer.

"""

import unittest
import math
import numpy as np
from ..gravity import BaseGravity, Gravity, Production, Attraction, Doubly


class TestGravity(unittest.TestCase):
    """Tests for gravity-type models"""

    def setUp(self):
        self.f = np.array([1131, 1887, 69, 738, 98, 31, 43, 19, 1633,
                           14055, 416, 1276, 1850, 388, 303, 159, 2301, 20164,
                           1080, 1831, 1943, 742, 674, 407, 85, 379, 1597,
                           1608, 328, 317, 469, 114, 762, 1110, 2973, 1252,
                           1081, 622, 425, 262, 196, 2027, 3498, 346, 1332,
                           2144, 821, 274, 49, 378, 1349, 310, 851, 2117,
                           630, 106, 87, 424, 978, 490, 670, 577, 546,
                           569, 33, 128, 643, 154, 328, 199, 112, 587])

        self.o = np.array(['AT11',
                           'AT11',
                           'AT11',
                           'AT11',
                           'AT11',
                           'AT11',
                           'AT11',
                           'AT11',
                           'AT12',
                           'AT12',
                           'AT12',
                           'AT12',
                           'AT12',
                           'AT12',
                           'AT12',
                           'AT12',
                           'AT13',
                           'AT13',
                           'AT13',
                           'AT13',
                           'AT13',
                           'AT13',
                           'AT13',
                           'AT13',
                           'AT21',
                           'AT21',
                           'AT21',
                           'AT21',
                           'AT21',
                           'AT21',
                           'AT21',
                           'AT21',
                           'AT22',
                           'AT22',
                           'AT22',
                           'AT22',
                           'AT22',
                           'AT22',
                           'AT22',
                           'AT22',
                           'AT31',
                           'AT31',
                           'AT31',
                           'AT31',
                           'AT31',
                           'AT31',
                           'AT31',
                           'AT31',
                           'AT32',
                           'AT32',
                           'AT32',
                           'AT32',
                           'AT32',
                           'AT32',
                           'AT32',
                           'AT32',
                           'AT33',
                           'AT33',
                           'AT33',
                           'AT33',
                           'AT33',
                           'AT33',
                           'AT33',
                           'AT33',
                           'AT34',
                           'AT34',
                           'AT34',
                           'AT34',
                           'AT34',
                           'AT34',
                           'AT34',
                           'AT34'])

        self.d = np.array(['AT12',
                           'AT13',
                           'AT21',
                           'AT22',
                           'AT31',
                           'AT32',
                           'AT33',
                           'AT34',
                           'AT11',
                           'AT13',
                           'AT21',
                           'AT22',
                           'AT31',
                           'AT32',
                           'AT33',
                           'AT34',
                           'AT11',
                           'AT12',
                           'AT21',
                           'AT22',
                           'AT31',
                           'AT32',
                           'AT33',
                           'AT34',
                           'AT11',
                           'AT12',
                           'AT13',
                           'AT22',
                           'AT31',
                           'AT32',
                           'AT33',
                           'AT34',
                           'AT11',
                           'AT12',
                           'AT13',
                           'AT21',
                           'AT31',
                           'AT32',
                           'AT33',
                           'AT34',
                           'AT11',
                           'AT12',
                           'AT13',
                           'AT21',
                           'AT22',
                           'AT32',
                           'AT33',
                           'AT34',
                           'AT11',
                           'AT12',
                           'AT13',
                           'AT21',
                           'AT22',
                           'AT31',
                           'AT33',
                           'AT34',
                           'AT11',
                           'AT12',
                           'AT13',
                           'AT21',
                           'AT22',
                           'AT31',
                           'AT32',
                           'AT34',
                           'AT11',
                           'AT12',
                           'AT13',
                           'AT21',
                           'AT22',
                           'AT31',
                           'AT32',
                           'AT33'])

        self.dij = np.array([103.001845,
                             84.204666,
                             220.811933,
                             132.00748,
                             214.511814,
                             246.933305,
                             390.85611,
                             505.089539,
                             103.001845,
                             45.796272,
                             216.994739,
                             129.878172,
                             140.706671,
                             201.232355,
                             343.50075,
                             453.515594,
                             84.204666,
                             45.796272,
                             249.932874,
                             158.630661,
                             186.420738,
                             244.108305,
                             387.61776,
                             498.407152,
                             220.811933,
                             216.994739,
                             249.932874,
                             92.407958,
                             151.777157,
                             92.894408,
                             194.851669,
                             306.105825,
                             132.00748,
                             129.878172,
                             158.630661,
                             92.407958,
                             124.563096,
                             122.433524,
                             261.893783,
                             376.34667,
                             214.511814,
                             140.706671,
                             186.420738,
                             151.777157,
                             124.563096,
                             81.753652,
                             208.456383,
                             314.793199,
                             246.933305,
                             201.232355,
                             244.108305,
                             92.894408,
                             122.433524,
                             81.753652,
                             145.076472,
                             258.591197,
                             390.85611,
                             343.50075,
                             387.61776,
                             194.851669,
                             261.893783,
                             208.456383,
                             145.076472,
                             114.46325,
                             505.089539,
                             453.515594,
                             498.407152,
                             306.105825,
                             376.34667,
                             314.793199,
                             258.591197,
                             114.46325])

        self.o_var = np.array([4320,
                               4320,
                               4320,
                               4320,
                               4320,
                               4320,
                               4320,
                               4320,
                               21478,
                               21478,
                               21478,
                               21478,
                               21478,
                               21478,
                               21478,
                               21478,
                               30500,
                               30500,
                               30500,
                               30500,
                               30500,
                               30500,
                               30500,
                               30500,
                               5012,
                               5012,
                               5012,
                               5012,
                               5012,
                               5012,
                               5012,
                               5012,
                               8811,
                               8811,
                               8811,
                               8811,
                               8811,
                               8811,
                               8811,
                               8811,
                               11349,
                               11349,
                               11349,
                               11349,
                               11349,
                               11349,
                               11349,
                               11349,
                               6021,
                               6021,
                               6021,
                               6021,
                               6021,
                               6021,
                               6021,
                               6021,
                               4477,
                               4477,
                               4477,
                               4477,
                               4477,
                               4477,
                               4477,
                               4477,
                               2275,
                               2275,
                               2275,
                               2275,
                               2275,
                               2275,
                               2275,
                               2275])

        self.d_var = np.array([27169,
                               28710,
                               4354,
                               9069,
                               8577,
                               4963,
                               3923,
                               2026,
                               5452,
                               28710,
                               4354,
                               9069,
                               8577,
                               4963,
                               3923,
                               2026,
                               5452,
                               27169,
                               4354,
                               9069,
                               8577,
                               4963,
                               3923,
                               2026,
                               5452,
                               27169,
                               28710,
                               9069,
                               8577,
                               4963,
                               3923,
                               2026,
                               5452,
                               27169,
                               28710,
                               4354,
                               8577,
                               4963,
                               3923,
                               2026,
                               5452,
                               27169,
                               28710,
                               4354,
                               9069,
                               4963,
                               3923,
                               2026,
                               5452,
                               27169,
                               28710,
                               4354,
                               9069,
                               8577,
                               3923,
                               2026,
                               5452,
                               27169,
                               28710,
                               4354,
                               9069,
                               8577,
                               4963,
                               2026,
                               5452,
                               27169,
                               28710,
                               4354,
                               9069,
                               8577,
                               4963,
                               3923])

    def test_BaseGravity_exp(self):
        f = np.array(self.f).reshape((-1, 1))
        dij = np.array(self.dij).reshape((-1, 1))
        model = BaseGravity(f, dij, 'exp', constant=False)
        np.testing.assert_allclose(model.params, [0.01641585], atol=.0001)
        self.assertAlmostEqual(model.AIC, 957622.28429746185, delta=.0001)
        np.testing.assert_allclose(model.cov_params, [[1.92096665e-10]])
        self.assertAlmostEqual(model.deviance, 1087408.9707170483, delta=.0001)
        self.assertAlmostEqual(model.llf, -478810.14214873099, delta=.0001)
        self.assertAlmostEqual(model.llnull, -88037.0499629, delta=.0001)
        np.testing.assert_allclose(model.pvalues, [0.])
        np.testing.assert_allclose(model.std_err, [1.38598941e-05])
        np.testing.assert_allclose(model.tvalues, [1184.41355888])
        np.testing.assert_allclose(model.yhat,
                                   [5.42415692e+00, 3.98401807e+00, 3.75177744e+01,
                                    8.73217546e+00, 3.38315236e+01, 5.76055685e+01,
                                    6.11695077e+02, 3.98970414e+03, 5.42415692e+00,
                                    2.12078133e+00, 3.52389616e+01, 8.43222048e+00,
                                    1.00726025e+01, 2.72049640e+01, 2.81140796e+02,
                                    1.71101560e+03, 3.98401807e+00, 2.12078133e+00,
                                    6.05130899e+01, 1.35184658e+01, 2.13329799e+01,
                                    5.49951210e+01, 5.80026424e+02, 3.57519614e+03,
                                    3.75177744e+01, 3.52389616e+01, 6.05130899e+01,
                                    4.55832329e+00, 1.20799918e+01, 4.59486946e+00,
                                    2.44995584e+01, 1.52168163e+02, 8.73217546e+00,
                                    8.43222048e+00, 1.35184658e+01, 4.55832329e+00,
                                    7.72767984e+00, 7.46219749e+00, 7.36414576e+01,
                                    4.82050643e+02, 3.38315236e+01, 1.00726025e+01,
                                    2.13329799e+01, 1.20799918e+01, 7.72767984e+00,
                                    3.82690126e+00, 3.06302472e+01, 1.75492594e+02,
                                    5.76055685e+01, 2.72049640e+01, 5.49951210e+01,
                                    4.59486946e+00, 7.46219749e+00, 3.82690126e+00,
                                    1.08216970e+01, 6.97553001e+01, 6.11695077e+02,
                                    2.81140796e+02, 5.80026424e+02, 2.44995584e+01,
                                    7.36414576e+01, 3.06302472e+01, 1.08216970e+01,
                                    6.54702760e+00, 3.98970414e+03, 1.71101560e+03,
                                    3.57519614e+03, 1.52168163e+02, 4.82050643e+02,
                                    1.75492594e+02, 6.97553001e+01, 6.54702760e+00])

    def test_BaseGravity_pow(self):
        f = np.array(self.f).reshape((-1, 1))
        dij = np.array(self.dij).reshape((-1, 1))
        model = BaseGravity(f, dij, 'pow', constant=False)
        np.testing.assert_allclose(model.params, [1.27223738], atol=.0001)
        self.assertAlmostEqual(model.AIC, 377298.04716333596, delta=.0001)
        np.testing.assert_allclose(model.cov_params, [[4.31955426e-07]])
        self.assertAlmostEqual(model.deviance, 409811.34329065739, delta=.0001)
        self.assertAlmostEqual(model.llf, -188648.02358166798, delta=.0001)
        self.assertAlmostEqual(model.llnull, -88037.0499629, delta=.0001)
        np.testing.assert_allclose(model.pvalues, [0.])
        np.testing.assert_allclose(model.std_err, [0.00065723], atol=.000001)
        np.testing.assert_allclose(model.tvalues, [1935.74740017])
        np.testing.assert_allclose(model.yhat,
                                   [363.76143383,
                                    281.50403714,
                                    959.7388893,
                                    498.77506053,
                                    925.03759732,
                                    1106.44361848,
                                    1984.54428735,
                                    2749.95948574,
                                    363.76143383,
                                    129.70901679,
                                    938.68096943,
                                    488.56203387,
                                    540.96136464,
                                    852.80642651,
                                    1683.84456031,
                                    2397.81642174,
                                    281.50403714,
                                    129.70901679,
                                    1123.57104159,
                                    630.10766251,
                                    773.76239688,
                                    1090.36467516,
                                    1963.64917204,
                                    2703.75625368,
                                    959.7388893,
                                    938.68096943,
                                    1123.57104159,
                                    316.84652033,
                                    595.67905738,
                                    318.9700416,
                                    818.55371165,
                                    1454.18199247,
                                    498.77506053,
                                    488.56203387,
                                    630.10766251,
                                    316.84652033,
                                    463.26843623,
                                    453.2156204,
                                    1192.42000515,
                                    1891.29566175,
                                    925.03759732,
                                    540.96136464,
                                    773.76239688,
                                    595.67905738,
                                    463.26843623,
                                    271.12096396,
                                    891.94447199,
                                    1506.88882976,
                                    1106.44361848,
                                    852.80642651,
                                    1090.36467516,
                                    318.9700416,
                                    453.2156204,
                                    271.12096396,
                                    562.42482847,
                                    1173.32244253,
                                    1984.54428735,
                                    1683.84456031,
                                    1963.64917204,
                                    818.55371165,
                                    1192.42000515,
                                    891.94447199,
                                    562.42482847,
                                    416.01781589,
                                    2749.95948574,
                                    2397.81642174,
                                    2703.75625368,
                                    1454.18199247,
                                    1891.29566175,
                                    1506.88882976,
                                    1173.32244253,
                                    416.01781589])

    def test_QuasiPoisson(self):
        f = np.array(self.f).reshape((-1, 1))
        dij = np.array(self.dij).reshape((-1, 1))
        model = BaseGravity(f, dij, 'exp', constant=False, Quasi=True)
        np.testing.assert_allclose(model.params, [0.01641585], atol=.0001)
        self.assertTrue(math.isnan(model.AIC))
        np.testing.assert_allclose(model.cov_params, [[0.00079749]],
                                   atol=1.0e-8)
        self.assertAlmostEqual(model.deviance, 1087408.9707170483, delta=.0001)
        self.assertTrue(np.isnan(model.llf))
        self.assertTrue(np.isnan(model.llnull))
        np.testing.assert_allclose(model.pvalues, [0.56103881])
        np.testing.assert_allclose(model.std_err, [0.02823993], atol=1.0e-8)
        np.testing.assert_allclose(model.tvalues, [0.58129922])
        np.testing.assert_allclose(model.yhat,
                                   [5.42415692e+00, 3.98401807e+00, 3.75177744e+01,
                                    8.73217546e+00, 3.38315236e+01, 5.76055685e+01,
                                    6.11695077e+02, 3.98970414e+03, 5.42415692e+00,
                                    2.12078133e+00, 3.52389616e+01, 8.43222048e+00,
                                    1.00726025e+01, 2.72049640e+01, 2.81140796e+02,
                                    1.71101560e+03, 3.98401807e+00, 2.12078133e+00,
                                    6.05130899e+01, 1.35184658e+01, 2.13329799e+01,
                                    5.49951210e+01, 5.80026424e+02, 3.57519614e+03,
                                    3.75177744e+01, 3.52389616e+01, 6.05130899e+01,
                                    4.55832329e+00, 1.20799918e+01, 4.59486946e+00,
                                    2.44995584e+01, 1.52168163e+02, 8.73217546e+00,
                                    8.43222048e+00, 1.35184658e+01, 4.55832329e+00,
                                    7.72767984e+00, 7.46219749e+00, 7.36414576e+01,
                                    4.82050643e+02, 3.38315236e+01, 1.00726025e+01,
                                    2.13329799e+01, 1.20799918e+01, 7.72767984e+00,
                                    3.82690126e+00, 3.06302472e+01, 1.75492594e+02,
                                    5.76055685e+01, 2.72049640e+01, 5.49951210e+01,
                                    4.59486946e+00, 7.46219749e+00, 3.82690126e+00,
                                    1.08216970e+01, 6.97553001e+01, 6.11695077e+02,
                                    2.81140796e+02, 5.80026424e+02, 2.44995584e+01,
                                    7.36414576e+01, 3.06302472e+01, 1.08216970e+01,
                                    6.54702760e+00, 3.98970414e+03, 1.71101560e+03,
                                    3.57519614e+03, 1.52168163e+02, 4.82050643e+02,
                                    1.75492594e+02, 6.97553001e+01, 6.54702760e+00])

    def test_Gravity(self):
        model = Gravity(self.f, self.o_var, self.d_var,
                        self.dij, 'exp', constant=True)
        np.testing.assert_allclose(
            model.params, [-7.95447436e+00, 8.63867812e-01, 8.80474585e-01, -6.20544765e-03])
        self.assertAlmostEqual(model.AIC, 20395.085388908723, delta=.0001)
        np.testing.assert_allclose(model.cov_params,
                                   [[5.70906352e-03, -3.00814799e-04, -2.62650384e-04,
                                     -2.40317578e-06],
                                    [-3.00814799e-04, 2.67121974e-05, 3.21466745e-06,
                                     1.16544737e-07],
                                       [-2.62650384e-04, 3.21466745e-06, 2.28600781e-05,
                                        9.94368232e-08],
                                       [-2.40317578e-06, 1.16544737e-07, 9.94368232e-08,
                                        2.68830005e-09]])
        self.assertAlmostEqual(model.deviance, 19806.408696637576, delta=.0001)
        self.assertAlmostEqual(model.llf, -10193.542694454361, delta=.0001)
        self.assertAlmostEqual(model.llnull, -88037.0499629, delta=.0001)
        np.testing.assert_allclose(model.pvalues, [0., 0., 0., 0.])
        np.testing.assert_allclose(
            model.std_err, [
                7.55583451e-02, 5.16838440e-03, 4.78122141e-03, 5.18488192e-05])
        np.testing.assert_allclose(
            model.tvalues, [-105.27592086, 167.14465196, 184.15264854, -119.68349034])
        np.testing.assert_allclose(model.yhat,
                                   [2053.49248374,
                                    2422.40705883,
                                    197.17666947,
                                    652.77645945,
                                    372.46664089,
                                    188.15630595,
                                    62.62225447,
                                    17.22633782,
                                    1995.44179687,
                                    12287.11927555,
                                    806.92929317,
                                    2643.59913196,
                                    2353.33783354,
                                    998.56216427,
                                    335.77135891,
                                    94.81498069,
                                    3035.77484367,
                                    15846.25211871,
                                    890.511914,
                                    2994.19536934,
                                    2399.15053753,
                                    1036.08892279,
                                    345.71715146,
                                    97.15537629,
                                    273.27020389,
                                    1150.87005074,
                                    984.81363732,
                                    948.91636667,
                                    625.0285152,
                                    556.41059801,
                                    240.2714148,
                                    67.32796418,
                                    771.93257863,
                                    3217.0998412,
                                    2825.35215036,
                                    809.66631035,
                                    1204.76218438,
                                    754.13231343,
                                    258.03819482,
                                    70.88540396,
                                    575.70226041,
                                    3743.25042014,
                                    2959.00444172,
                                    697.06556556,
                                    1574.69697708,
                                    1207.94322877,
                                    447.3674688,
                                    129.24387416,
                                    272.27577768,
                                    1487.02882957,
                                    1196.36810195,
                                    580.9635273,
                                    922.83252627,
                                    1130.90519845,
                                    383.40648414,
                                    105.94015788,
                                    86.29277039,
                                    476.14958977,
                                    380.14055538,
                                    238.89720288,
                                    300.687118,
                                    398.84078404,
                                    365.10261002,
                                    200.59513613,
                                    23.66650989,
                                    134.05168303,
                                    106.50884151,
                                    66.7421182,
                                    82.35371404,
                                    114.87900692,
                                    100.58000293,
                                    199.99352826])
        self.assertAlmostEquals(model.D2, 0.88713874099960177)
        self.assertAlmostEquals(model.adj_D2, 0.88215956780840776)
        self.assertAlmostEquals(model.SSI, 0.72706171189789603)
        self.assertAlmostEquals(model.pseudoR2, 0.88421303645743465)
        self.assertAlmostEquals(model.adj_pseudoR2, 0.88416760104130376)
        self.assertAlmostEquals(model.SRMSE, 0.62063116008447083)

    def test_local_Gravity(self):
        model = Gravity(self.f, self.o_var, self.d_var, self.dij, 'exp')
        local = model.local(loc_index=self.o, locs=np.unique(self.o))
        self.assertEqual(list(local.keys()).sort(), ['stde0',
                                                     'stde1',
                                                     'stde2',
                                                     'pvalue2',
                                                     'SRMSE',
                                                     'pvalue0',
                                                     'deviance',
                                                     'adj_pseudoR2',
                                                     'pvalue1',
                                                     'tvalue0',
                                                     'tvalue2',
                                                     'adj_D2',
                                                     'tvalue1',
                                                     'SSI',
                                                     'AIC',
                                                     'param1',
                                                     'param0',
                                                     'D2',
                                                     'pseudoR2',
                                                     'param2'].sort())

    def test_Production(self):
        model = Production(self.f, self.o, self.d_var,
                           self.dij, 'exp', constant=True)
        np.testing.assert_allclose(model.params,
                                   [-1.11700938, 1.68662317, 2.15188689, 0.60300297,
                                    0.88380784, 1.20926104, 0.68938983, 1.15472804,
                                    1.02479968, 0.89278717, -0.00727113], atol=.0001)
        self.assertAlmostEqual(model.AIC, 15882.651018068489, delta=.0001)
        np.testing.assert_allclose(model.cov_params,
                                   [[2.58467540e-03, -3.29423877e-04, -3.27686611e-04,
                                     -3.08689103e-04, -2.97140418e-04, -2.89494010e-04,
                                     -3.24014540e-04, -3.00776842e-04, -2.84393168e-04,
                                     -2.24000219e-04, -9.64855587e-07],
                                    [-3.29423877e-04, 3.03025458e-04, 2.53009591e-04,
                                     2.47263232e-04, 2.49058621e-04, 2.47981815e-04,
                                     2.48504221e-04, 2.42350062e-04, 2.38815483e-04,
                                     7.03380199e-06, 9.21233182e-08],
                                       [-3.27686611e-04, 2.53009591e-04, 2.87125687e-04,
                                        2.47623385e-04, 2.49193103e-04, 2.48208882e-04,
                                        2.48776786e-04, 2.43211814e-04, 2.40006717e-04,
                                        6.93978830e-06, 8.51141937e-08],
                                       [-3.08689103e-04, 2.47263232e-04, 2.47623385e-04,
                                        4.64543893e-04, 2.54358195e-04, 2.56271647e-04,
                                        2.58881194e-04, 2.72166616e-04, 2.79448200e-04,
                                        7.48669925e-06, -1.28025978e-07],
                                       [-2.97140418e-04, 2.49058621e-04, 2.49193103e-04,
                                        2.54358195e-04, 3.69553926e-04, 2.52457119e-04,
                                        2.53895776e-04, 2.59259137e-04, 2.62162787e-04,
                                        5.35894223e-06, -4.43827259e-08],
                                       [-2.89494010e-04, 2.47981815e-04, 2.48208882e-04,
                                        2.56271647e-04, 2.52457119e-04, 3.47667496e-04,
                                        2.55361893e-04, 2.63782581e-04, 2.68393880e-04,
                                        5.00405857e-06, -8.03379392e-08],
                                       [-3.24014540e-04, 2.48504221e-04, 2.48776786e-04,
                                        2.58881194e-04, 2.53895776e-04, 2.55361893e-04,
                                        4.30583201e-04, 2.68391703e-04, 2.74115589e-04,
                                        8.62466197e-06, -9.29399372e-08],
                                       [-3.00776842e-04, 2.42350062e-04, 2.43211814e-04,
                                        2.72166616e-04, 2.59259137e-04, 2.63782581e-04,
                                        2.68391703e-04, 5.29121755e-04, 3.15535312e-04,
                                        8.88670616e-06, -3.18385859e-07],
                                       [-2.84393168e-04, 2.38815483e-04, 2.40006717e-04,
                                        2.79448200e-04, 2.62162787e-04, 2.68393880e-04,
                                        2.74115589e-04, 3.15535312e-04, 7.96308690e-04,
                                        8.69726183e-06, -4.44906514e-07],
                                       [-2.24000219e-04, 7.03380199e-06, 6.93978830e-06,
                                        7.48669925e-06, 5.35894223e-06, 5.00405857e-06,
                                        8.62466197e-06, 8.88670616e-06, 8.69726183e-06,
                                        2.17985878e-05, 6.51339971e-08],
                                       [-9.64855587e-07, 9.21233182e-08, 8.51141937e-08,
                                        -1.28025978e-07, -4.43827259e-08, -8.03379392e-08,
                                        -9.29399372e-08, -3.18385859e-07, -4.44906514e-07,
                                        6.51339971e-08, 2.77308674e-09]])
        self.assertAlmostEqual(model.deviance, 15279.974241770311, delta=.0001)
        self.assertAlmostEqual(model.llf, -7930.3255090342445, delta=.0001)
        self.assertAlmostEqual(model.llnull, -88037.0499629, delta=.0001)
        np.testing.assert_allclose(model.pvalues,
                                   [5.43122293e-107,
                                    0.00000000e+000,
                                    0.00000000e+000,
                                    3.06800447e-172,
                                    0.00000000e+000,
                                    0.00000000e+000,
                                    5.04395549e-242,
                                    0.00000000e+000,
                                    9.03955976e-289,
                                    0.00000000e+000,
                                    0.00000000e+000])
        np.testing.assert_allclose(model.std_err,
                                   [5.08397030e-02,
                                    1.74076264e-02,
                                    1.69447835e-02,
                                    2.15532803e-02,
                                    1.92237854e-02,
                                    1.86458439e-02,
                                    2.07504988e-02,
                                    2.30026467e-02,
                                    2.82189420e-02,
                                    4.66889578e-03,
                                    5.26601057e-05])
        np.testing.assert_allclose(model.tvalues,
                                   [-21.97120187, 96.88989939, 126.99406254, 27.97731759,
                                    45.97470357, 64.85418671, 33.22280753, 50.19979035,
                                    36.31602055, 191.22019711, -138.07670549])
        np.testing.assert_allclose(model.yhat,
                                   [1.40705950e+03, 1.69457663e+03, 1.16508879e+02,
                                    4.27850723e+02, 2.23425179e+02, 1.08301078e+02,
                                    3.08300817e+01, 7.44793331e+00, 1.81162644e+03,
                                    1.21014912e+04, 6.46999802e+02, 2.34696906e+03,
                                    2.06388796e+03, 8.15528209e+02, 2.34966095e+02,
                                    5.85312512e+01, 3.30741049e+03, 1.83446566e+04,
                                    8.10873546e+02, 3.03231168e+03, 2.35717102e+03,
                                    9.50837295e+02, 2.71489717e+02, 6.72496632e+01,
                                    2.60277189e+02, 1.12260001e+03, 9.28118288e+02,
                                    1.04284804e+03, 6.44343295e+02, 6.06652130e+02,
                                    2.34315477e+02, 5.78455649e+01, 6.57379261e+02,
                                    2.80075361e+03, 2.38710037e+03, 7.17245241e+02,
                                    1.03993511e+03, 6.48056270e+02, 1.90566474e+02,
                                    4.59636590e+01, 4.99603238e+02, 3.58445439e+03,
                                    2.70058180e+03, 6.44960859e+02, 1.51347637e+03,
                                    1.20618713e+03, 3.89165529e+02, 9.95706858e+01,
                                    2.34675109e+02, 1.37251483e+03, 1.05563448e+03,
                                    5.88432822e+02, 9.13951678e+02, 1.16884200e+03,
                                    3.66858927e+02, 8.90901579e+01, 1.31244011e+02,
                                    7.76879800e+02, 5.92149430e+02, 4.46507449e+02,
                                    5.27992298e+02, 7.40876898e+02, 7.20725128e+02,
                                    4.04624989e+02, 5.02255240e+01, 3.06563409e+02,
                                    2.32354948e+02, 1.74615053e+02, 2.01734215e+02,
                                    3.00280455e+02, 2.77258060e+02, 6.40968342e+02])
        self.assertAlmostEquals(model.D2, 0.912931356874)
        self.assertAlmostEquals(model.adj_D2, 0.89865780882)
        self.assertAlmostEquals(model.SSI, 0.740619203383)
        self.assertAlmostEquals(model.pseudoR2, 0.909920590111)
        self.assertAlmostEquals(model.adj_pseudoR2, 0.909795642717)
        self.assertAlmostEquals(model.SRMSE, 0.46622685091043831)

    def test_local_Production(self):
        model = Production(self.f, self.o, self.d_var, self.dij, 'exp')
        local = model.local(locs=np.unique(self.o))
        self.assertEqual(list(local.keys()).sort(), ['stde0',
                                                     'stde1',
                                                     'stde2',
                                                     'pvalue2',
                                                     'SRMSE',
                                                     'pvalue0',
                                                     'deviance',
                                                     'adj_pseudoR2',
                                                     'pvalue1',
                                                     'tvalue0',
                                                     'tvalue2',
                                                     'adj_D2',
                                                     'tvalue1',
                                                     'SSI',
                                                     'AIC',
                                                     'param1',
                                                     'param0',
                                                     'D2',
                                                     'pseudoR2',
                                                     'param2'].sort())

    def test_Attraction(self):
        model = Production(self.f, self.d, self.o_var,
                           self.dij, 'exp', constant=True)
        np.testing.assert_allclose(model.params,
                                   [-0.88439723,
                                    1.62180605,
                                    1.92772078,
                                    0.12462001,
                                    0.62378812,
                                    0.69646073,
                                    0.20909411,
                                    0.6856777,
                                    0.48539625,
                                    0.89235874,
                                    -0.00693755],
                                   atol=.001)
        self.assertAlmostEqual(model.AIC, 16275.899321893821, delta=.0001)
        np.testing.assert_allclose(model.cov_params,
                                   [[3.01436996e-03, -2.61742292e-04, -3.18191276e-04,
                                     -2.61736294e-04, -2.53401872e-04, -2.53545012e-04,
                                     -2.81169571e-04, -2.43409544e-04, -2.12802803e-04,
                                     -2.71488782e-04, -1.17108280e-06],
                                    [-2.61742292e-04, 2.36371652e-04, 1.97978106e-04,
                                     1.92565769e-04, 1.94367290e-04, 1.93561823e-04,
                                     1.93929484e-04, 1.87837851e-04, 1.84018218e-04,
                                     5.78923328e-06, 8.62912701e-08],
                                       [-3.18191276e-04, 1.97978106e-04, 2.37130911e-04,
                                        1.95813824e-04, 1.96321084e-04, 1.95974290e-04,
                                        1.97059881e-04, 1.93136341e-04, 1.90444087e-04,
                                        1.16187824e-05, 7.68842070e-08],
                                       [-2.61736294e-04, 1.92565769e-04, 1.95813824e-04,
                                        4.45977428e-04, 1.98639315e-04, 2.00358776e-04,
                                        2.01640218e-04, 2.11745720e-04, 2.17565021e-04,
                                        7.97756072e-06, -9.56753770e-08],
                                       [-2.53401872e-04, 1.94367290e-04, 1.96321084e-04,
                                        1.98639315e-04, 3.12561535e-04, 1.97440629e-04,
                                        1.98271627e-04, 2.01952018e-04, 2.03971780e-04,
                                        6.29181262e-06, -2.57004528e-08],
                                       [-2.53545012e-04, 1.93561823e-04, 1.95974290e-04,
                                        2.00358776e-04, 1.97440629e-04, 3.20607776e-04,
                                        1.99534150e-04, 2.05855338e-04, 2.09446226e-04,
                                        6.66273501e-06, -5.53303117e-08],
                                       [-2.81169571e-04, 1.93929484e-04, 1.97059881e-04,
                                        2.01640218e-04, 1.98271627e-04, 1.99534150e-04,
                                        4.04837719e-04, 2.07747990e-04, 2.11608257e-04,
                                        9.45143925e-06, -5.46040064e-08],
                                       [-2.43409544e-04, 1.87837851e-04, 1.93136341e-04,
                                        2.11745720e-04, 2.01952018e-04, 2.05855338e-04,
                                        2.07747990e-04, 4.85148555e-04, 2.46472592e-04,
                                        8.10725781e-06, -2.60737252e-07],
                                       [-2.12802803e-04, 1.84018218e-04, 1.90444087e-04,
                                        2.17565021e-04, 2.03971780e-04, 2.09446226e-04,
                                        2.11608257e-04, 2.46472592e-04, 7.90692415e-04,
                                        6.52409863e-06, -3.86785704e-07],
                                       [-2.71488782e-04, 5.78923328e-06, 1.16187824e-05,
                                        7.97756072e-06, 6.29181262e-06, 6.66273501e-06,
                                        9.45143925e-06, 8.10725781e-06, 6.52409863e-06,
                                        2.64461183e-05, 8.70031728e-08],
                                       [-1.17108280e-06, 8.62912701e-08, 7.68842070e-08,
                                        -9.56753770e-08, -2.57004528e-08, -5.53303117e-08,
                                        -5.46040064e-08, -2.60737252e-07, -3.86785704e-07,
                                        8.70031728e-08, 2.62593686e-09]])
        self.assertAlmostEqual(model.deviance, 15673.222613627502, delta=.0001)
        self.assertAlmostEqual(model.llf, -8126.9496609469106, delta=.0001)
        self.assertAlmostEqual(model.llnull, -88037.0499629, delta=.0001)
        np.testing.assert_allclose(model.pvalues,
                                   [2.23154436e-058,
                                    0.00000000e+000,
                                    0.00000000e+000,
                                    3.61133996e-009,
                                    1.05877746e-272,
                                    0.00000000e+000,
                                    2.69492058e-025,
                                    9.38664385e-213,
                                    9.08121216e-067,
                                    0.00000000e+000,
                                    0.00000000e+000])
        np.testing.assert_allclose(model.std_err,
                                   [5.49032782e-02,
                                    1.53743830e-02,
                                    1.53990555e-02,
                                    2.11181777e-02,
                                    1.76794099e-02,
                                    1.79055236e-02,
                                    2.01205795e-02,
                                    2.20260880e-02,
                                    2.81192535e-02,
                                    5.14257895e-03,
                                    5.12438958e-05])
        np.testing.assert_allclose(model.tvalues,
                                   [-16.10827734, 105.487554, 125.18435157, 5.90107805,
                                    35.28331128, 38.89641795, 10.39205191, 31.13025333,
                                    17.2620603, 173.52358548, -135.38293645])
        np.testing.assert_allclose(model.yhat,
                                   [1.79502279e+03, 2.77690999e+03, 1.77376340e+02,
                                    5.41058308e+02, 3.28265191e+02, 1.61020145e+02,
                                    9.55492240e+01, 3.54052486e+01, 1.48342439e+03,
                                    1.51642463e+04, 7.61962380e+02, 2.29718733e+03,
                                    2.29156465e+03, 9.24935115e+02, 5.55191561e+02,
                                    2.11833031e+02, 2.31106289e+03, 1.52712766e+04,
                                    8.29095427e+02, 2.57322937e+03, 2.28197035e+03,
                                    9.39377653e+02, 5.59026730e+02, 2.12153271e+02,
                                    1.78795020e+02, 9.29389609e+02, 1.00418629e+03,
                                    8.13086106e+02, 5.79191340e+02, 5.35268359e+02,
                                    4.24969889e+02, 1.60758895e+02, 5.47719688e+02,
                                    2.81394269e+03, 3.12998907e+03, 8.16565623e+02,
                                    1.15732912e+03, 7.21460431e+02, 4.41575377e+02,
                                    1.63374443e+02, 3.87326254e+02, 3.27181524e+03,
                                    3.23528198e+03, 6.77976176e+02, 1.34894643e+03,
                                    1.19916138e+03, 8.01876754e+02, 3.13863001e+02,
                                    1.75685709e+02, 1.22115852e+03, 1.23153422e+03,
                                    5.79386090e+02, 7.77596785e+02, 1.10887286e+03,
                                    7.06986190e+02, 2.63279368e+02, 4.96907636e+01,
                                    3.49378290e+02, 3.49326167e+02, 2.19253703e+02,
                                    2.26850151e+02, 3.53430501e+02, 3.36979293e+02,
                                    5.49332748e+02, 1.22952888e+01, 8.90162551e+01,
                                    8.85260032e+01, 5.53842615e+01, 5.60455225e+01,
                                    9.23759900e+01, 8.37976212e+01, 3.66824277e+02])
        self.assertAlmostEquals(model.D2, .910690541438)
        self.assertAlmostEquals(model.adj_D2, .896049646592)
        self.assertAlmostEquals(model.SSI, .750634498293)
        self.assertAlmostEquals(model.pseudoR2, .90768716507)
        self.assertAlmostEquals(model.adj_pseudoR2, .907562217676)
        self.assertAlmostEquals(model.SRMSE, 0.59478477816884223)

    def test_local_Attraction(self):
        model = Attraction(self.f, self.d, self.o_var, self.dij, 'exp')
        local = model.local(locs=np.unique(self.d))
        self.assertEqual(list(local.keys()).sort(), ['stde0',
                                                     'stde1',
                                                     'stde2',
                                                     'pvalue2',
                                                     'SRMSE',
                                                     'pvalue0',
                                                     'deviance',
                                                     'adj_pseudoR2',
                                                     'pvalue1',
                                                     'tvalue0',
                                                     'tvalue2',
                                                     'adj_D2',
                                                     'tvalue1',
                                                     'SSI',
                                                     'AIC',
                                                     'param1',
                                                     'param0',
                                                     'D2',
                                                     'pseudoR2',
                                                     'param2'].sort())

    def test_Doubly(self):
        model = Doubly(self.f, self.o, self.d,
                       self.dij, 'exp', constant=True)
        np.testing.assert_allclose(model.params,
                                   [6.20471518, 1.5449095, 2.4414292, 0.69924374,
                                    0.94869185, 1.28967637, 0.74270015, 1.19468573,
                                    0.98874193, 1.49709841, 2.18492741, 0.18784818,
                                    0.66434515, 0.74264938, 0.21334535, 0.66765781,
                                    0.39986094, -0.00791533], atol=1e-05)
        np.testing.assert_allclose(model.cov_params,
                                   [[5.01690795e-04, -2.67085869e-04, -2.85861407e-04,
                                     -2.47145002e-04, -2.56344375e-04, -2.47959694e-04,
                                     -2.51858026e-04, -2.34909872e-04, -2.31532205e-04,
                                     -2.12557582e-04, -2.30973877e-04, -1.98360054e-04,
                                     -2.04848380e-04, -1.97315240e-04, -2.04713619e-04,
                                     -1.92147501e-04, -1.90223393e-04, -2.88654296e-07],
                                    [-2.67085869e-04, 3.09818975e-04, 2.38201819e-04,
                                     2.40816440e-04, 2.44508571e-04, 2.41573651e-04,
                                     2.43213918e-04, 2.36475186e-04, 2.35264739e-04,
                                     2.70747190e-05, -4.06434204e-06, 3.54631504e-06,
                                     6.73127801e-06, 3.48881444e-07, 4.84106698e-06,
                                     -1.08816401e-07, -7.82227026e-07, 1.17369687e-07],
                                       [-2.85861407e-04, 2.38201819e-04, 3.15835404e-04,
                                        2.60714112e-04, 2.57317985e-04, 2.58287551e-04,
                                        2.61026738e-04, 2.62571867e-04, 2.63204233e-04,
                                        -1.46703716e-06, 6.08300790e-05, 2.43162304e-05,
                                        2.12302255e-05, 1.90685319e-05, 2.32782320e-05,
                                        2.47736982e-05, 2.53307733e-05, -1.06931968e-08],
                                       [-2.47145002e-04, 2.40816440e-04, 2.60714112e-04,
                                        4.73670174e-04, 2.57806206e-04, 2.62588091e-04,
                                        2.64062738e-04, 2.76642894e-04, 2.84509772e-04,
                                        1.49294364e-06, 2.51567145e-05, 1.84043845e-05,
                                        -2.99190057e-06, 1.28512200e-06, -1.26738274e-05,
                                        -7.38879982e-06, 8.72549111e-08, -1.47073267e-07],
                                       [-2.56344375e-04, 2.44508571e-04, 2.57317985e-04,
                                        2.57806206e-04, 3.74656775e-04, 2.56143467e-04,
                                        2.58077967e-04, 2.64765554e-04, 2.67330191e-04,
                                        5.69526013e-06, 2.17034297e-05, -1.88949992e-06,
                                        2.20220200e-05, 4.13804378e-06, 1.27479768e-06,
                                        6.91280894e-06, 9.28146927e-06, -7.56672892e-08],
                                       [-2.47959694e-04, 2.41573651e-04, 2.58287551e-04,
                                        2.62588091e-04, 2.56143467e-04, 3.54522394e-04,
                                        2.58778390e-04, 2.69502689e-04, 2.75078094e-04,
                                        -1.05126847e-06, 1.97946415e-05, 1.65176617e-06,
                                        3.76490799e-06, 1.56828518e-05, -1.12614285e-05,
                                        -3.07326187e-06, 1.83335365e-06, -1.15030115e-07],
                                       [-2.51858026e-04, 2.43213918e-04, 2.61026738e-04,
                                        2.64062738e-04, 2.58077967e-04, 2.58778390e-04,
                                        4.39566465e-04, 2.72270083e-04, 2.82540973e-04,
                                        1.91670954e-06, 2.37810191e-05, -1.14352620e-05,
                                        1.02053574e-06, -1.13429776e-05, 9.96630546e-06,
                                        -1.99290067e-05, -9.39196494e-06, -9.20519180e-08],
                                       [-2.34909872e-04, 2.36475186e-04, 2.62571867e-04,
                                        2.76642894e-04, 2.64765554e-04, 2.69502689e-04,
                                        2.72270083e-04, 5.48320166e-04, 2.87269673e-04,
                                        -4.65887794e-06, 2.58726435e-05, -8.69859569e-06,
                                        5.47577328e-06, -4.51656124e-06, -2.45399627e-05,
                                        1.73990976e-05, -1.18725973e-04, -2.41709452e-07],
                                       [-2.31532205e-04, 2.35264739e-04, 2.63204233e-04,
                                        2.84509772e-04, 2.67330191e-04, 2.75078094e-04,
                                        2.82540973e-04, 2.87269673e-04, 8.14375093e-04,
                                        -5.91338987e-06, 2.65660547e-05, -1.32176066e-06,
                                        7.75773294e-06, 5.22185282e-07, -1.46830696e-05,
                                        -1.23317429e-04, 1.69481305e-05, -2.70788264e-07],
                                       [-2.12557582e-04, 2.70747190e-05, -1.46703716e-06,
                                        1.49294364e-06, 5.69526013e-06, -1.05126847e-06,
                                        1.91670954e-06, -4.65887794e-06, -5.91338987e-06,
                                        2.42563703e-04, 1.84733809e-04, 1.89259451e-04,
                                        1.91681710e-04, 1.89152965e-04, 1.92231256e-04,
                                        1.87441436e-04, 1.86834624e-04, 1.13843139e-07],
                                       [-2.30973877e-04, -4.06434204e-06, 6.08300790e-05,
                                        2.51567145e-05, 2.17034297e-05, 1.97946415e-05,
                                        2.37810191e-05, 2.58726435e-05, 2.65660547e-05,
                                        1.84733809e-04, 2.63915732e-04, 2.06737361e-04,
                                        2.02941436e-04, 2.03812109e-04, 2.06876793e-04,
                                        2.08793972e-04, 2.09473765e-04, -1.94248549e-08],
                                       [-1.98360054e-04, 3.54631504e-06, 2.43162304e-05,
                                        1.84043845e-05, -1.88949992e-06, 1.65176617e-06,
                                        -1.14352620e-05, -8.69859569e-06, -1.32176066e-06,
                                        1.89259451e-04, 2.06737361e-04, 4.53792323e-04,
                                        2.01217128e-04, 2.05155865e-04, 2.05806138e-04,
                                        2.14332194e-04, 2.21550755e-04, -9.10506514e-08],
                                       [-2.04848380e-04, 6.73127801e-06, 2.12302255e-05,
                                        -2.99190057e-06, 2.20220200e-05, 3.76490799e-06,
                                        1.02053574e-06, 5.47577328e-06, 7.75773294e-06,
                                        1.91681710e-04, 2.02941436e-04, 2.01217128e-04,
                                        3.17290867e-04, 2.00121482e-04, 2.02120689e-04,
                                        2.06522637e-04, 2.08554008e-04, -4.37219119e-08],
                                       [-1.97315240e-04, 3.48881444e-07, 1.90685319e-05,
                                        1.28512200e-06, 4.13804378e-06, 1.56828518e-05,
                                        -1.13429776e-05, -4.51656124e-06, 5.22185282e-07,
                                        1.89152965e-04, 2.03812109e-04, 2.05155865e-04,
                                        2.00121482e-04, 3.26458468e-04, 2.01391450e-04,
                                        2.09628557e-04, 2.14889547e-04, -7.68167253e-08],
                                       [-2.04713619e-04, 4.84106698e-06, 2.32782320e-05,
                                        -1.26738274e-05, 1.27479768e-06, -1.12614285e-05,
                                        9.96630546e-06, -2.45399627e-05, -1.46830696e-05,
                                        1.92231256e-04, 2.06876793e-04, 2.05806138e-04,
                                        2.02120689e-04, 2.01391450e-04, 4.14676504e-04,
                                        2.11496728e-04, 2.21430978e-04, -2.25147281e-08],
                                       [-1.92147501e-04, -1.08816401e-07, 2.47736982e-05,
                                        -7.38879982e-06, 6.91280894e-06, -3.07326187e-06,
                                        -1.99290067e-05, 1.73990976e-05, -1.23317429e-04,
                                        1.87441436e-04, 2.08793972e-04, 2.14332194e-04,
                                        2.06522637e-04, 2.09628557e-04, 2.11496728e-04,
                                        5.06851801e-04, 2.14569472e-04, -1.33183180e-07],
                                       [-1.90223393e-04, -7.82227026e-07, 2.53307733e-05,
                                        8.72549111e-08, 9.28146927e-06, 1.83335365e-06,
                                        -9.39196494e-06, -1.18725973e-04, 1.69481305e-05,
                                        1.86834624e-04, 2.09473765e-04, 2.21550755e-04,
                                        2.08554008e-04, 2.14889547e-04, 2.21430978e-04,
                                        2.14569472e-04, 8.07696796e-04, -1.49419116e-07],
                                       [-2.88654296e-07, 1.17369687e-07, -1.06931968e-08,
                                        -1.47073267e-07, -7.56672892e-08, -1.15030115e-07,
                                        -9.20519180e-08, -2.41709452e-07, -2.70788264e-07,
                                        1.13843139e-07, -1.94248549e-08, -9.10506514e-08,
                                        -4.37219119e-08, -7.68167253e-08, -2.25147281e-08,
                                        -1.33183180e-07, -1.49419116e-07, 2.56252122e-09]])
        self.assertAlmostEqual(model.deviance, 9360.482092561484, delta=.0001)
        self.assertAlmostEqual(model.llf, -4970.5795707251054, delta=.0001)
        self.assertAlmostEqual(model.llnull, -88037.0499629, delta=.0001)
        np.testing.assert_allclose(model.pvalues,
                                   [0.00000000e+000,
                                    0.00000000e+000,
                                    0.00000000e+000,
                                    1.76331083e-226,
                                    0.00000000e+000,
                                    0.00000000e+000,
                                    7.21755436e-275,
                                    0.00000000e+000,
                                    4.88760299e-263,
                                    0.00000000e+000,
                                    0.00000000e+000,
                                    1.16346714e-018,
                                    1.88877600e-304,
                                    0.00000000e+000,
                                    1.10421926e-025,
                                    2.83322217e-193,
                                    5.83172788e-045,
                                    0.00000000e+000])
        np.testing.assert_allclose(model.std_err,
                                   [2.23984552e-02,
                                    1.76016753e-02,
                                    1.77717586e-02,
                                    2.17639650e-02,
                                    1.93560527e-02,
                                    1.88287651e-02,
                                    2.09658404e-02,
                                    2.34162372e-02,
                                    2.85372580e-02,
                                    1.55744567e-02,
                                    1.62454834e-02,
                                    2.13024018e-02,
                                    1.78126603e-02,
                                    1.80681617e-02,
                                    2.03636073e-02,
                                    2.25133694e-02,
                                    2.84200070e-02,
                                    5.06213514e-05])
        np.testing.assert_allclose(model.tvalues,
                                   [277.01531761, 87.77059404, 137.37690538, 32.12850872,
                                    49.0126713, 68.49500564, 35.42429674, 51.01954322,
                                    34.64740487, 96.12524108, 134.49445277, 8.81816885,
                                    37.29623407, 41.10265285, 10.47679558, 29.65605897,
                                    14.06969877, -156.36352821])
        np.testing.assert_allclose(model.yhat,
                                   [9.78988280e+02, 2.26003279e+03, 1.04038742e+02,
                                    3.38382580e+02, 1.90458075e+02, 8.67908467e+01,
                                    4.37554720e+01, 1.35532201e+01, 1.02693176e+03,
                                    1.43579537e+04, 5.02646536e+02, 1.61314478e+03,
                                    1.60124044e+03, 5.84144805e+02, 2.98377549e+02,
                                    9.55604104e+01, 2.92086883e+03, 1.76899160e+04,
                                    9.49267467e+02, 3.14910952e+03, 2.73315395e+03,
                                    1.01972797e+03, 5.15779061e+02, 1.64177257e+02,
                                    1.73496758e+02, 7.99088484e+02, 1.22486311e+03,
                                    9.31563443e+02, 6.29698756e+02, 5.91117070e+02,
                                    4.15424488e+02, 1.31747905e+02, 4.49674437e+02,
                                    2.04361676e+03, 3.23802841e+03, 7.42345992e+02,
                                    1.00234547e+03, 6.00432512e+02, 3.13590596e+02,
                                    9.69658353e+01, 3.29132064e+02, 2.63792996e+03,
                                    3.65458094e+03, 6.52540343e+02, 1.30346098e+03,
                                    1.16517842e+03, 6.73203489e+02, 2.21973821e+02,
                                    1.47356669e+02, 9.45479887e+02, 1.33962391e+03,
                                    6.01828982e+02, 7.67131590e+02, 1.14476805e+03,
                                    6.43385796e+02, 2.00425139e+02, 7.41169755e+01,
                                    4.81822820e+02, 6.76007805e+02, 4.21969575e+02,
                                    3.99722086e+02, 6.59873779e+02, 6.41890452e+02,
                                    9.85596546e+02, 2.44225078e+01, 1.64157859e+02,
                                    2.28909306e+02, 1.42362371e+02, 1.31485029e+02,
                                    2.31461478e+02, 2.12717926e+02, 1.04848355e+03])
        self.assertAlmostEquals(model.D2, .946661920897)
        self.assertAlmostEquals(model.adj_D2, .929870303401)
        self.assertAlmostEquals(model.SSI, .811852110904)
        self.assertAlmostEquals(model.pseudoR2, .943539912198)
        self.assertAlmostEquals(model.adj_pseudoR2, .943335452826)
        self.assertAlmostEquals(model.SRMSE, 0.37925654532618808)


if __name__ == '__main__':
    unittest.main()
