# Copyright 2020 Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Tests for tfx_bsl.types.tfx_namedtuple."""
import collections
import pickle
import sys
import typing

from tfx_bsl.types import tfx_namedtuple

from absl.testing import absltest

_cls = {}
_TFXNamedTuple = tfx_namedtuple.namedtuple('_TFXNamedTuple', ['attrib'])
_TFXTypedNamedTuple = tfx_namedtuple.TypedNamedTuple('_TFXTypedNamedTuple',
                                                     [('attrib', int)])
_NamedTuple = collections.namedtuple('_NamedTuple', ['attrib'])
_TypedNamedTuple = typing.NamedTuple('_TypedNamedTuple', [('attrib', int)])


class _TFXNamedTupleSubclass(_TFXNamedTuple):

  def __str__(self):
    return 'hi, I am _TFXNamedTupleSubclass'


class _TFXTypedNamedTupleSubclass(_TFXTypedNamedTuple):

  def __str__(self):
    return 'hi, I am _TFXTypedNamedTupleSubclass'


class _NamedTupleSubclass(_NamedTuple):

  def __str__(self):
    return 'hi, I am _NamedTupleSubclass'


class _TypedNamedTupleSubclass(_TypedNamedTuple):

  def __str__(self):
    return 'hi, I am _TypedNamedTupleSubclass'


def _restore(name, fields, value):
  """Restore an object of namedtuple."""
  key = (name, fields)
  cls = _cls.get(key)
  if cls is None:
    cls = collections.namedtuple(name, fields)
    _cls[key] = cls
  return cls(*value)


def _hack_namedtuple(cls):
  """Make class generated by namedtuple picklable."""
  name = cls.__name__
  fields = cls._fields

  def reduce(self):
    return (_restore, (name, fields, tuple(self)))

  cls.__reduce__ = reduce
  cls._is_namedtuple_ = True
  return cls


class TFXNamedtupleTest(absltest.TestCase):

  def testPickling(self):
    tfx_namedtuple_child = _TFXNamedTupleSubclass(1)
    tfx_typed_namedtuple_child = _TFXTypedNamedTupleSubclass(2)
    namedtuple_child = _NamedTupleSubclass(3)
    typed_namedtuple_child = _TypedNamedTupleSubclass(4)

    # This emulates hack that is done in PySpark.
    for obj in sys.modules['__main__'].__dict__.values():
      if (isinstance(obj, type) and obj.__base__ is tuple and
          hasattr(obj, '_fields') and '__reduce__' not in obj.__dict__):
        _hack_namedtuple(obj)  # hack inplace

    # Subclasses are preserved.
    self.assertIsInstance(tfx_namedtuple_child, _TFXNamedTupleSubclass)
    self.assertEqual(
        str(tfx_namedtuple_child), 'hi, I am _TFXNamedTupleSubclass')
    self.assertIsInstance(tfx_typed_namedtuple_child,
                          _TFXTypedNamedTupleSubclass)
    self.assertEqual(
        str(tfx_typed_namedtuple_child), 'hi, I am _TFXTypedNamedTupleSubclass')
    self.assertIsInstance(namedtuple_child, _NamedTupleSubclass)
    self.assertEqual(str(namedtuple_child), 'hi, I am _NamedTupleSubclass')
    self.assertIsInstance(typed_namedtuple_child, _TypedNamedTupleSubclass)
    self.assertEqual(
        str(typed_namedtuple_child), 'hi, I am _TypedNamedTupleSubclass')

    # Perform pickling.
    tfx_namedtuple_child = pickle.loads(pickle.dumps(tfx_namedtuple_child))
    tfx_typed_namedtuple_child = pickle.loads(
        pickle.dumps(tfx_typed_namedtuple_child))
    namedtuple_child = pickle.loads(pickle.dumps(namedtuple_child))
    typed_namedtuple_child = pickle.loads(pickle.dumps(typed_namedtuple_child))

    # `_TFXNamedTupleSubclass` and `_TFXTypedNamedTupleSubclass` are preserved,
    # but the `_NamedTupleSubclass` and `_TypedNamedTupleSubclass` instances
    # are pickled and unpickled as their parent classes.
    self.assertIsInstance(tfx_namedtuple_child, _TFXNamedTupleSubclass)
    self.assertEqual(
        str(tfx_namedtuple_child), 'hi, I am _TFXNamedTupleSubclass')
    self.assertIsInstance(tfx_typed_namedtuple_child,
                          _TFXTypedNamedTupleSubclass)
    self.assertEqual(
        str(tfx_typed_namedtuple_child), 'hi, I am _TFXTypedNamedTupleSubclass')
    self.assertNotIsInstance(namedtuple_child, _NamedTupleSubclass)
    self.assertEqual(str(namedtuple_child), '_NamedTuple(attrib=3)')
    self.assertNotIsInstance(typed_namedtuple_child, _TypedNamedTupleSubclass)
    self.assertEqual(str(typed_namedtuple_child), '_TypedNamedTuple(attrib=4)')


if __name__ == '__main__':
  absltest.main()
