# code: 
# * George Moschovitis  <gm@navel.gr>
# * Anastasios Koutoumanos  <ak@navel.gr>
# * Elias Karakoulakis  <ekarak@ktismata.com>
#
# (c) 2004 Navel, all rights reserved.
# $Id: string.rb 202 2005-01-17 10:44:13Z gmosx $

require "uri"

module N;

# = StringUtils
#
# General string utilities collection.
#
# === Design:
#
# Implement as a module to avoid class polution. You can
# still Ruby's advanced features to include the module in your
# class. Passing the object to act upon allows to check for nil,
# which isn't possible if you use self.
#
# === TODO:
#
# - implement a method that returns easy to remember
#   pseudo-random strings
# - add aliases for those methods in Kernel.
#
module StringUtils

	@@map_to_greeklish = {
		"" => "a", "" => "A", "" => "a", "" => "A",
		"" => "b", "" => "B",
		"" => "g", "" => "G",
		"" => "d", "" => "D",
		"" => "e", "" => "E", "" => "e", "" => "E",
		"" => "z", "" => "Z",
		"" => "h", "" => "H", "" => "h", "" => "H",
		"" => "8", "" => "8",
		"" => "i", "" => "I", "" => "i", "" => "I",
		"" => "k", "" => "K",
		"" => "l", "" => "L",
		"" => "m", "" => "M",
		"" => "n", "" => "N",
		"" => "3", "" => "3",
		"" => "o", "" => "O", "" => "o", "" => "O",
		"" => "p", "" => "P",
		"" => "r", "" => "R",
		"" => "s", "" => "s", "" => "S",
		"" => "t", "" => "T",
		"" => "y", "" => "Y", "" => "y", "" => "Y",
		"" => "f", "" => "F",
		"" => "x", "" => "X",
		"" => "ps","" => "PS",
		"" => "w", "" => "W", "" => "w", ""=>"W"
	}

	# Convert the input string to greeklish.
	#--
	# gmosx, TODO: remove from public distribution
	#++
	#
	def self.to_greeklish(input)
		return nil unless input
		output = ""
		# gmosx: also parse new lines
		input.scan(/./m) { |w|
			c = @@map_to_greeklish[w]
			output << (c.nil?? w: c)
		}
		return output
	end

	# Move this in String class?
	#
	# Tests a string for a valid value (non nil, not empty)
	#
	def self.valid?(string)
		return (not ((nil == string) or (string.empty?)))
	end

	# returns short abstract of long strings (first 'count'
	# characters, chopped at the nearest word, appended by '...')
	# force_cutoff: break forcibly at 'count' chars. Does not accept
	# count < 2.
	#
	def self.head(string, count = 128, force_cutoff = false, ellipsis="...")
		return nil unless string
		return nil if count < 2

		if string.size > count
			cut_at = force_cutoff ? count : (string.index(' ', count-1) || count)
			xstring = string.slice(0, cut_at)
			return xstring.chomp(" ") + ellipsis
		else
			return string
		end
	end

	# Apply a set of rules (regular expression matches) to the
	# string
	#
	# === Requirements:
	# - the rules must be applied in order! So we cannot use a
	#   hash because the ordering is not guaranteed! we use an
	#	array instead.
	#
	# === Input:
	# the string to rewrite
	# the array containing rule-pairs (match, rewrite)
	#
	# === Output:
	# the rewritten string

	MATCH = 0
	REWRITE = 1

	def self.rewrite(string, rules)
		return nil unless string

		# gmosx: helps to find bugs
		raise ArgumentError.new("the rules parameter is nil") unless rules

		rewritten_string = string.dup

		for rule in rules
			rewritten_string.gsub!(rule[MATCH], rule[REWRITE])
		end

		return (rewritten_string or string)
	end

	# Enforces a maximum width of a string inside an
	# html container. If the string exceeds this maximum width
	# the string gets wraped.
	#
	# Not really useful, better use the CSS overflow: hidden
	# functionality.
	#
	# === Input:
	# the string to be wrapped
	# the enforced width
	# the separator used for wrapping
	#
	# === Output:
	# the wrapped string
	#
	# === Example:
	#  text = "1111111111111111111111111111111111111111111"
	#  text = wrap(text, 10, " ")
	#  p text # => "1111111111 1111111111 1111111111"
	#
	# See the test cases to better understand the behaviour!
	#
	def self.wrap(string, width = 20, separator = " ")
		return nil unless string

		re = /([^#{separator}]{1,#{width}})/
		wrapped_string = string.scan(re).join(separator)

		return wrapped_string
	end

	# Replace dangerours chars in filenames
	#
	def self.rationalize_filename(filename)
		return nil unless filename
		# gmosx: rationalize a copy!!! (add unit test)
		xfilename = filename.dup()
		# gmosx: replace some dangerous chars!
		xfilename.gsub!(/ /, "-")
		xfilename.gsub!(/!/, "")
		xfilename.gsub!(/'/, "")
		xfilename.gsub!(/\(/, "")
		xfilename.gsub!(/\)/, "")
		xfilename = self.to_greeklish(xfilename)
		return xfilename
	end

	# Returns a random string. one possible use is
	# password initialization.
	#
	# === Input:
	# the maximum length of the string
	#
	# === Output:
	# the random string
	#
	def self.random(max_length = 8, char_re = /[\w\d]/)
    # gmosx: this is a nice example of input parameter checking.
		# this is NOT a real time called method so we can add this
		# check. Congrats to the author.
		raise ArgumentError.new("char_re must be a regular expression!") unless char_re.is_a?(Regexp)
		
		string = ""

		while string.length < max_length
				ch = rand(255).chr
				string << ch if ch =~ char_re
		end

		return string
	end

	# Screen an IP address
	#--
	# gmosx: copied this method from n1, check how it works!
	# probably deprecate?
	#++
	def self.screen_ip_address(address)
		if address
			return address.split(',').collect { |hostip|
				hostip.gsub(/\.[^\.]*$/, ".*")
			}.join(', ')
		else
			return "*.*.*.*"
		end
	end
	
end

end # module

