$:.unshift File.join(File.dirname(__FILE__), '..', 'lib')

require 'test/unit'

require 'og'

module Test # :nodoc: all

# bug: creates a table that fucks-up postgres if not
# correctly escaped.
class User
	prop_accessor :name, String
	prop_accessor :banned, TrueClass

	def initialize(name = nil)
		@name = name
	end
end

class Role
	prop_accessor :title, String
end

class Comment; end

class Article
	prop_accessor :title, String
	prop_accessor :body, String
	prop_accessor :owner_oid, Fixnum, :sql_index => true
	prop_accessor :another_oid, Fixnum
	sql_index 'owner_oid, another_oid'
	prop_accessor :options, Hash
	has_many :comments, Test::Comment
	
	def initialize(title = nil, body = nil)
		@title = title
		@body = body
		@options = {"hello" => "world"}
	end
	
	def og_pre_insert(conn)
		puts "-- PRE INSERT CALLED FOR ARTICLE"
	end
	
	def og_post_insert_update(conn)
		puts "-- POST INSERT UPDATE CALLED FOR ARTICLE"
	end

	def og_post_update(conn)
		puts "-- POST UPDATE CALLED FOR ARTICLE"
	end
end

class Comment
	prop_accessor :body, String
	prop_accessor :create_time, Time
	belongs_to :article, Test::Article, :extra_sql => 'NOT NULL' 
	belongs_to :author, Test::User
	
	def initialize(body = nil)
		@create_time = Time.now
		@body = body
	end
	
	def og_pre_insert(conn)
		puts "-- PRE INSERT CALLED FOR COMMENT"
	end

	def og_post_insert(conn)
		puts "-- POST INSERT CALLED FOR COMMENT"
	end	
end

module MyMixin
	prop_accessor :title, String
end

class MyClass
	include Test::MyMixin
end

class OrderItem
	prop_accessor :number, Fixnum
	refers_to :article, Article
end

class TC_N_OG < Test::Unit::TestCase

	def setup
		psql = true

		if psql
			config = {
				:backend => "psql",
				:address => "localhost",
				:database => "test",
				:user => "postgres",
				:password => "navelrulez",
				:connection_count => 1
			}
		else
			config = {
				:backend => "mysql",
				:address => "localhost",
				:database => "test",
				:user => "root",
				:password => "navelrulez",
				:connection_count => 1
			}
		end
		
		Og::Database.drop_db!(config)
		$og = Og::Database.new(config)
	end

	def teardown
		$og.shutdown()
	end

	# gmosx: hmm, implemented in one method to enforce order.
	#	
	def test_all
		$DBG = true

		$og.get_connection()
		
		article = Article.new("Title", "Here comes the body")
		$og << article
		
		article.title = "Changed"
		article.save!
		
		$og.pupdate("body='Hello'", article)

		article.update_properties "body='Hello'"
		
		another = Article[1]
		assert_equal(1, another.oid)
		# bug: yaml load problem.
		assert_equal("world", another.options["hello"])
		
		# bug: YAMLing nil property
		another.options = nil
		another.save!
		
		assert_equal(nil, $og.load(30000, Article))
		
		articles = $og.load_all(Article)

#		p articles
#		p Article[23]
		
		user = User.new("gmosx")
		user.save!
		
		user = User["gmosx"]
		
		assert_equal("gmosx", user.name)
		
		users1 = $og.select("name='gmosx' ORDER BY oid", User)
		
		users = $og.select("SELECT * FROM #{User::DBTABLE} WHERE name='gmosx' ORDER BY oid", User)
		
		users2 = User.select "name='gmosx' ORDER BY oid"
		
		assert_equal(users1.size, users2.size)
		
		article = Article.new("Title", "Body")
		article.save!

		comment = Comment.new("This is a comment")
		comment.article = article
		comment.author = User["gmosx"]
		comment.save!


		# test automatically generated add_commnet
		comment = Comment.new("This is another comment")
		comment.author = User["gmosx"]
		article.add_comment(comment)

		assert_equal(article.oid, comment.article_oid)
		
		comments = article.comments
		
		assert_equal(2, comments.size)		

		assert_equal("gmosx", comments[0].author.name)

		article.delete_all_comments
		assert article.comments.empty?

		# test cascading delete

		article.add_comment(Comment.new("hello"))
		article.add_comment(Comment.new("world"))

		Article.delete(article)
		
		assert Comment.all.empty?

		comment.delete!		

		# test extra_sql
		
		for p in Comment.__props
			if :article_oid == p.symbol
				assert_equal('NOT NULL', p.meta[:extra_sql])
				break
			end
		end

		assert($og.managed_classes.include?(Test::Article))
		
		# bug: indirectly managed (includes managed Module)
		assert($og.managed_classes.include?(Test::MyClass))

		# test create
		article = Article.create("title", "body")
		assert_equal(3, article.oid)

		# test refers_to

		oi = OrderItem.new
		oi.article = article
		oi.save

		assert_equal article.oid, oi.article_oid

		# bug

		user = User['gmosx']
		user.banned = true
		user.save!

		user = User['gmosx']
		assert_equal true, user.banned
		
	end

end

end
