# * George Moschovitis  <gm@navel.gr>
# (c) 2004-2005 Navel, all rights reserved.
# $Id: og.rb 248 2005-01-31 13:38:34Z gmosx $

require 'glue/logger'
require 'glue/attribute'
require 'glue/property'
require 'glue/array'
require 'glue/hash'
require 'glue/time'
require 'glue/pool'

# Og (ObjectGraph) is an efficient, yet simple Object-Relational 
# mapping library. 
#
# == Features
#
# The library provides the following features:
#
# + Object-Relational mapping.
# + Absolutely no configuration files.
# + Multiple backends (PostgreSQL, MySQL).
# + ActiveRecord-style meta language and db aware methods.
# + Deserialize to Ruby Objects or ResultSets.
# + Deserialize sql join queries to Ruby Objects.
# + Serialize arbitrary ruby object graphs through YAML.
# + Connection pooling.
# + Thread safety.
# + SQL transactions.
# + Lifecycle callbacks.
# + Transparent support for cascading deletes for all backends.
# + Hierarchical structures (preorder traversal, materialized paths)
# + Works safely as part of distributed application.
# + Simple implementation.
#
# == Meta language
#
# primary_key :pid (NOT IMPLEMENTED)
# name_key :name (NOT IMPLEMENTED)
# prop_accessor Fixnum, :pid, :sql => "smallint DEFAULT 1"
# has_many Child, :children
# many_to_many Role, :roles
# sql_index :pid
#
# == Design
#
# Keep the main classes backend agnostic. 
#--
# Try to make the methods work with oids. Do NOT implement descendants 
# use a root id (rid).
#++
# For class ids we use the name instead of a hash. Class ids are
# typically not used in querys, they are stored for completeness.
# If we store a hash we cannot reclaim the class thus invalidating
# the point. Instead of .name(), to_s() is used so the methods
# are more flexible (they accept class names too!!)
#
# Og allows the serialization of arbitrary Ruby objects. Just
# mark them as Object (or Array or Hash) in the prop_accessor
# and the engine will serialize a YAML dump of the object.
# Arbitrary object graphs are supported too.
#
# This is NOT a singleton, an application may access multiple
# databases.
#
# The og.xxx methods are more flexible and allow you to use
# multiple databases for example.
# 
# == Managed Objects Lifecycle Callbacks
#
# * og_pre_insert
# * og_post_insert
#	* og_pre_update
# * og_post_update
# * og_pre_insert_update
# * og_post_insert_update
# * self.og_pre_delete
#
# A class level callback is used for delete because typically you call
# delete with an oid and not an object to avoid a deserialization.
#
# == Future 
# 
# * Support prepared statements (pgsql)
# * Support stored procedures (pgsql)
# * Support caching.
# * Deserialize to OpenStruct.
# * Better documentation.

class Og

	# If true, only allow reading from the database. Usefull
	# for maintainance.
	
	cattr_accessor :read_only_mode, false

	# If true, the library automatically 'enchants' managed classes.
	# In enchant mode, special db aware methods are added to 
	# managed classes and instances.
	
	cattr_accessor :enchant_managed_classes, true
	
	# If true, use Ruby's advanced introspection capabilities to 
	# automatically manage classes tha define properties.
	
	cattr_accessor :auto_manage_classes, true

	# If true, automatically include the Og meta-language into Module. 
	#
	# By default this is FALSE, to avoid polution of the Module object.
	# However if you include a prop_accessor or a managed Mixin in your
	# object MetaLanguage gets automatically extended in the class.
	
	cattr_accessor :include_meta_language, true
	
	# Attach the following prefix to all generated SQL table names.
	# Usefull on hosting scenarios where you have to run multiple
	# web applications/sites on a single database.
	
	cattr_accessor :table_prefix, nil

	# The active database. Og allows you to access multiple
	# databases from a single application.
	
	cattr_accessor :db

	# Set the active database.

	def self.use(db)
		@@db = db
		@@db.get_connection
	end

end

# gmosx: leave this here.
require 'og/enchant'
require 'og/meta'

class Og
	
# Marker module. If included this in a class, the Og automanager
# ignores this class.

module Unmanageable; end

# Encapsulates an Og Database.

class Database
	include Og::Enchant

	# Managed class metadata
	
	class ManagedClassMeta
		# The managed class.
		attr_accessor :klass
		
		# A mapping of the database fields to the object properties.
		attr_accessor :field_index
		
		def initialize(klass = nil)
			@klass = klass
			@field_index = {}
		end
	end

	# hash of configuration options.

	attr_accessor :config

	# Pool of connections to the backend.
	
	attr_accessor :connection_pool

	# Managed classes.
	
	attr_accessor :managed_classes

	# Initialize the database interface.
	
	def initialize(config)
		@config = config

		# populate with default options if needed.
		@config[:connection_count] ||= 1
		
		# require the backend.		
		backend = @config[:backend] || "psql"
		require "og/backends/#{backend}"
		eval %{	@config[:backend] = #{backend.capitalize}Backend }
		
		@connection_pool = N::Pool.new
		@managed_classes = N::SafeHash.new
		
		Logger.info "Connecting to database '#{@config[:database]}' using backend '#{backend}'."
		
		@config[:connection_count].times do
			@connection_pool << Og::Connection.new(self)
		end
	
		# gmosx, FIXME: this automanage code is not elegant and slow
		# should probably recode this, along with glue/property.rb
		
		if Og.auto_manage_classes
			# automatically manage classes with properties and metadata.
			# gmosx: Any idea how to optimize this?
			classes_to_manage = []
			ObjectSpace.each_object(Class) do |c|
				if c.respond_to?(:__props) and c.__props
					classes_to_manage << c
				end
			end
			Logger.info "Og auto manages the following classes:"
			Logger.info "#{classes_to_manage.inspect}"
			manage_classes(*classes_to_manage)
		end

		# use the newly created database.
		Og.use(self)
	end
		
	# Shutdown the database interface.
	
	def shutdown
		for con in @connection_pool
			con.close()
		end
	end
	alias_method :close, :shutdown

	# Get a connection from the pool to access the database.
	# Stores the connection in a thread-local variable.

	def get_connection
		thread = Thread.current
		
		unless conn = thread[:og_conn]
			conn = @connection_pool.pop()
			thread[:og_conn] = conn
		end
		
		return conn
	end
	alias_method :connection, :get_connection
	
	# Restore an unused connection to the pool.
	
	def put_connection
		thread = Thread.current
		
		if conn = thread[:og_conn]
			thread[:og_conn] = nil
			return @connection_pool.push(conn)
		end
	end	
	
	# Utility method, automatically restores a connection to the pool.
	
	def connect(deserialize = nil, &block)
		result = nil
		
		begin
			conn = get_connection()
			conn.deserialize = deserialize unless deserialize.nil?
			
			result = yield(conn)
			
			conn.deserialize = true
		ensure
			put_connection()
		end

		return result
	end
	alias_method :open, :connect
	
	
	# Register a standard Ruby class as managed.
	
	def manage(klass)
		return if managed?(klass) or klass.ancestors.include?(Og::Unmanageable)
		
		@managed_classes[klass] = ManagedClassMeta.new(klass)
		
		# Add standard og methods to the class.
		convert(klass)
		
		# Add helper methods to the class.
		enchant(klass) if Og.enchant_managed_classes 
	end
	
	# Helper method to set multiple managed classes.
	
	def manage_classes(*klasses)
		for klass in klasses
			manage(klass)
		end
	end
	
	# Stop managing a Ruby class
	
	def unmanage(klass)
		@managed_classes.delete(klass)
	end
	
	# Is this class managed?
	#	
	def managed?(klass)
		return @managed_classes.include?(klass)
	end
	
	# Add standard og functionality to the class
	
	def convert(klass)
		# Grab backend class
		backend = @config[:backend]

		# gmosx: this check is needed to allow the developer to customize
		# the sql generated for oid
		backend.eval_og_oid(klass) unless klass.instance_methods.include?(:oid)
		
		klass.class_eval %{
			DBTABLE = "#{backend.table(klass)}"
			DBSEQ = "#{backend.table(klass)}_oids_seq" 
			
			def to_i()
				@oid
			end			
		}

		# Create the schema for this class if not available.
		create_table(klass)				

		# Precompile some code that gets executed all the time.
		# Deletion code is not precompiled, because it is not used
		# as frequently.
		backend.eval_og_insert(klass)
		backend.eval_og_update(klass)
		backend.eval_og_deserialize(klass, self)
	end
	
	# Automatically wrap connection methods.
	#
	def self.wrap_method(method, args)
		args = args.split(/,/)
		class_eval %{
			def #{method}(#{args.join(", ")})
				thread = Thread.current
				
				unless conn = thread[:og_conn]
					conn = @connection_pool.pop()
					thread[:og_conn] = conn
				end
			
				return conn.#{method}(#{args.collect {|a| a.split(/=/)[0]}.join(", ")})
			end
		}
	end
		
	wrap_method :create_table, "klass"
	wrap_method :drop_table, "klass"
	wrap_method :save, "obj";	alias_method :<<, :save; alias_method :put, :save
	wrap_method :insert, "obj"
	wrap_method :update, "obj"
	wrap_method :update_properties, "update_sql, obj_or_oid, klass = nil"
	wrap_method :pupdate, "update_sql, obj_or_oid, klass = nil"
	wrap_method :load, "oid, klass"; alias_method :get, :load	
	wrap_method :load_by_oid, "oid, klass"
	wrap_method :load_by_name, "name, klass"	
	wrap_method :load_all, "klass, extrasql = nil"
	wrap_method :select, "sql, klass"
	wrap_method :select_one, "sql, klass"
	wrap_method :count, "sql, klass = nil"
	wrap_method :delete, "obj_or_oid, klass = nil"	
	wrap_method :query, "sql"
	wrap_method :exec, "sql"

	class << self
		def create_db!(config)
			get_connection().db.create_db(config[:database], config[:user], 
					config[:password])
		end
		alias_method :create!, :create_db!

		def drop_db!(config)
			backend = config[:backend] || "psql"
			require "og/backends/#{backend}"
			eval %{
				#{backend.capitalize}Backend.drop_db(config[:database], config[:user], 
						config[:password])
			}
		end
		alias_method :drop!, :drop_db!
	end
end

end
