# * George Moschovitis  <gm@navel.gr>
# (c) 2004-2005 Navel, all rights reserved.
# $Id: meta.rb 248 2005-01-31 13:38:34Z gmosx $

require 'og/backend'
require 'glue/inflector'

class Og

# = MetaUtils
#
# Some useful meta-language utilities.

module MetaUtils # :nodoc: all

	# Convert the klass to a string representation
	# The leading module if available is removed.
	#--
	# gmosx, FIXME: unify with the ogutils.encode method?
	#++
	
	def self.expand(klass)
		return klass.name.gsub(/^.*::/, '').gsub(/::/, '_').downcase
	end

	# Infer the target klass for a relation. When defining
	# relations, tha target class is typically given. Some times
	# in order to avoid forward declarations a symbol is given instead
	# of a class. Other times on class is given at all, and 
	# the inflection mechanism is used to infer the class name.

	def self.resolve_class(name, klass)
		klass ||= N::Inflector.camelize(name)
		
		return klass if klass.is_a?(Class)
		
		unless klass.is_a?(String) or klass.is_a?(Symbol)
			raise 'Invalid class definition' 
		end
	
		unless Object.const_get(klass.intern)
			# Forward declaration.
			Object.class_eval("class #{klass}; end")
		end

		return Object.const_get(klass)
	end

end

# = MetaLanguage
#
# Implements a meta-language for manipulating og-managed objects
# and defining their relationships. The original idea comes
# from the excellent ActiveRecord library.
#
# Many more useful relations will be available soon.

module MetaLanguage

	# Defines an SQL index. Useful for defining indiced
	# over multiple columns.
	
	def sql_index(index, options = {})
		meta :sql_index, [index, options]
	end

	# Implements a 'belongs_to' relation.
	# Automatically enchants the calling class with helper methods.
	# 
	# Example:
	#
	# class MyObject
	#		belongs_to AnotherObject, :prop => :parent
	# end
	#
	# creates the code:
	#
	# prop_accessor Fixnum, :parent_oid
	# def parent; ... end
	# def parent=(obj_or_oid); ... end
	
	def belongs_to(name, klass, options = {})
		prop_eval = "prop_accessor Fixnum, :#{name}_oid"
		prop_eval << ", :sql => '#{options[:sql]}'" if options[:sql]
		prop_eval << ", :extra_sql => '#{options[:extra_sql]}'" if options[:extra_sql]

		meta :belongs_to, klass
		
		module_eval %{
			#{prop_eval}
			
			def #{name}
				Og.db.load_by_oid(@#{name}_oid, #{klass})
			end
			
			def #{name}=(obj_or_oid)
				@#{name}_oid = obj_or_oid.to_i
			end
		}
	end

	# Implements a 'has_one' relation.
	# Automatically enchants the calling class with helper methods.
	#
	# Example:
	#
	# class MyObject
	#		has_one :child, TheClass
	#		has_one :article
	# end
	#
	# creates the code:
	#
	# ...
	
	def has_one(name, klass = nil, options = {})

		# linkback is the property of the child object that 'links back' 
		# to this object. 

		linkback = options[:linkback] || "#{MetaUtils.expand(self)}_oid"
		
		meta :has, [klass, linkback]
		
		module_eval %{
			def #{name}(extrasql = nil)
				Og.db.select_one("SELECT * FROM #{Backend.table(klass)} WHERE #{linkback}=\#\@oid \#\{extrasql\}", #{klass})
			end

			def delete_#{name}(extrasql = nil)
				Og.db.exec("DELETE FROM #{Backend.table(klass)} WHERE #{linkback}=\#\@oid \#\{extrasql\}") 
			end			
		}
	end	
	
	# Implements a 'has_many' relation.
	# Automatically enchants the calling class with helper methods.
	# 
	# Example:
	#
	# class MyObject
	#		has_many :children, AnotherObject
	# end
	#
	# creates the code:
	#
	# def children; ... end
	
	def has_many(name, klass, options = {})
		name_s = N::Inflector.singularize(name.to_s)
		
		# linkback is the property of the child object that 'links back' 
		# to this object. 

		linkback = options[:linkback] || "#{MetaUtils.expand(self)}_oid"
		
		# keep belongs to metadata, useful for 
		# reflection/scaffolding.

		meta :has, [klass, linkback]

		module_eval %{
			def #{name}(extrasql = nil)
				Og.db.select("SELECT * FROM #{Backend.table(klass)} WHERE #{linkback}=\#\@oid \#\{extrasql\}", #{klass})
			end

			def #{name}_count(extrasql = nil)
				Og.db.count("SELECT COUNT(*) FROM #{Backend.table(klass)} WHERE #{linkback}=\#\@oid \#\{extrasql\}")
			end			

			def add_#{name_s}(obj, extra = nil)
				obj.#{linkback} = @oid
				obj.save!
			end
			
			def delete_all_#{name}(extrasql = nil)
				Og.db.exec("DELETE FROM #{Backend.table(klass)} WHERE #{linkback}=\#\@oid \#\{extrasql\}") 
			end			
		}
	end
	
	# Implements a 'many_to_many' relation.
	# Two objects are associated using an intermediate join table.
	# Automatically enchants the calling class with helper methods.
	#
	# Options:
	#
	# Example:
	#
	# class Article
	# 	many_to_many :categories, Category, :linkback => articles
	# end
	#
	# article.categories
	# article.del_category
	# article.add_category
	#	article.clear_categories
	#
	# category.articles
	# ...
	#--
	# FIXME: make more compatible with other enchant methods.
	#++
	
	def many_to_many(name, klass, options = {})
		list_o = name.to_s
		prop_o = N::Inflector.singularize(list_o)
		list_m = options[:linkback] || N::Inflector.plural_name(self) 
		prop_m = N::Inflector.singularize(list_m)
	
		# exit if the class is allready indirectly 'enchanted' from the
		# other class of the many_to_many relation.

		return if self.respond_to?(prop_m)
	
		# Add some metadata to the class to allow for automatic join table
		# calculation.

		meta :sql_join, [klass, options]

		meta :many_to_many, klass
		klass.meta :many_to_many, self 

		# enchant this class
		
		module_eval %{
			def #{list_o}(extrasql = nil)
				Og.db.select("SELECT d.* FROM #{Backend.table(klass)} AS d, #{Backend.join_table(self, klass)} AS j WHERE j.key1=\#\@oid AND j.key2=d.oid \#\{extrasql\}", #{klass})
			end

			def #{list_o}_count(extrasql = nil)
				Og.db.select("SELECT COUNT(*) FROM #{Backend.table(klass)} AS d, #{Backend.join_table(self, klass)} AS j WHERE j.key1=\#\@oid AND j.key2=d.oid \#\{extrasql\}", #{klass})
			end						
			
			def add_#{prop_o}(obj, extra = nil)
				Og.db.exec("INSERT INTO #{Backend.join_table(self, klass)} (key1, key2) VALUES (\#\@oid, \#\{obj.oid\})") 
			end
			
			def delete_#{prop_o}(obj_or_oid, extra = nil)
				Og.db.exec("DELETE FROM #{Backend.join_table(self, klass)} WHERE key2=\#\{obj_or_oid.to_i\}") 
			end

			def clear_#{list_o}
				Og.db.exec("DELETE FROM #{Backend.join_table(self, klass)} WHERE key1=\#\@oid") 
			end			
		}
		
		# indirectly enchant the other class of the relation.
		
		klass.module_eval %{
			def #{list_m}(extrasql = nil)
				Og.db.select("SELECT s.* FROM #{Backend.table(self)} AS s, #{Backend.join_table(self, klass)} AS j WHERE j.key2=\#\@oid AND j.key1=s.oid \#\{extrasql\}", #{self})
			end

			def #{list_m}_count(extrasql = nil)
				Og.db.select("SELECT COUNT(*) FROM #{Backend.table(self)} AS s, #{Backend.join_table(self, klass)} AS j WHERE j.key2=\#\@oid AND j.key1=s.oid \#\{extrasql\}", #{self})
			end
			
			def add_#{prop_m}(obj, extra = nil)
				Og.db.exec("INSERT INTO #{Backend.join_table(self, klass)} (key1, key2) VALUES (\#\{obj.oid\}, \#\@oid)") 
			end
			
			def delete_#{prop_m}(obj_or_oid, extra = nil)
				Og.db.exec("DELETE FROM #{Backend.join_table(self, klass)} WHERE key1=\#\{obj_or_oid.to_i\}") 
			end

			def clear_#{list_m}
				Og.db.exec("DELETE FROM #{Backend.join_table(self, klass)} WHERE key2=\#\@oid") 
			end			
		}
	end
	alias :has_and_belongs_to_many :many_to_many 

	# Implements a 'refers_to' relation.
	# This is a one-way version of the 'has_one'/'belongs_to' 
	# relations. The target object cannot link back to the source
	# object.
	# This is in fact EXACTLY the same as belongs_to with a 
	# different name (!!!!)
	#
	# Automatically enchants the calling class with helper methods.
	# 
	# 
	# Example:
	#
	# class MyObject
	#		refers_to article, Article
	# end
	#
	# creates the code:
	#
	# prop_accessor Fixnum, :article_oid
	# def article; ... end
	# def article=(obj_or_oid); ... end

	def refers_to(name, klass, options = {})
		prop_eval = "prop_accessor Fixnum, :#{name}_oid"
		prop_eval << ", :sql => '#{options[:sql]}'" if options[:sql]
		prop_eval << ", :extra_sql => '#{options[:extra_sql]}'" if options[:extra_sql]

		meta :refers_to, klass
		klass.meta :has, [self, "#{name}_oid"]
		
		module_eval %{
			#{prop_eval}
			
			def #{name}
				Og.db.load_by_oid(@#{name}_oid, #{klass})
			end
			
			def #{name}=(obj_or_oid)
				@#{name}_oid = obj_or_oid.to_i
			end
		}
	end

	# Declares that this class can join with another class.
	# The join parameters are given so the join-compatible 
	# methods are generated.

	def joins(klass, options = {})
		meta :joins, [klass, options]
	end

end

end

# Include the meta-language extensions into Module. If the flag is
# false the developer is responsible for including the MetaLanguage
# module where needed.
# 
# By default this is FALSE, to avoid polution of the Module object.
# However if you include a prop_accessor or a managed Mixin in your
# object MetaLanguage gets automatically extended in the class.

if Og.include_meta_language
	class Module # :nodoc: all
		include Og::MetaLanguage
	end
end
