#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Colours::Colours
#
# The ANSII colours are:
#
#   Black       0;30     Dark Gray     1;30
#   Red         0;31     Light Red     1;31
#   Green       0;32     Light Green   1;32
#   Brown       0;33     Yellow        1;33
#   Blue        0;34     Light Blue    1;34
#   Purple      0;35     Light Purple  1;35
#   Cyan        0;36     Light Cyan    1;36
#   Light Gray  0;37     White         1;37
#
# This list is for colours at the console. In xterm, the code 1;31
# is not "Light Red" but "Bold Red".
#
# Usage example:
#
#   Colours::Colours.new(ARGV)
#
# =========================================================================== #
# require 'colours/class/colours.rb'
# Colours.fancy_parse '<i><red>E. coli</red></i>'
# =========================================================================== #
require 'colours/base/base.rb'

module Colours

class Colours < Base # === Colours::Colours

  # ========================================================================= #
  # === LEADING_PART
  #
  # We don't include the "[" here.
  # ========================================================================= #
  LEADING_PART = "\033" # \x1b"

  # ========================================================================= #
  # === TRAILING_PART
  # ========================================================================= #
  TRAILING_PART = "\033[0m"

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(i = '')
    reset
    if i.is_a? Array
      i = i.join(' ').strip
    end
    @internal_hash[:raw_content] = i # Keep a reference-copy here.
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    # ======================================================================= #
    # === @internal_hash
    # ======================================================================= #
    @internal_hash = {}
    # ======================================================================= #
    # === :raw_content
    # ======================================================================= #
    @internal_hash[:raw_content] = nil
    # ======================================================================= #
    # === :leading_colour_component
    #
    # This can be '', '0' or '1'. The '1' means bright; the '0' is for
    # regular colours. The default is ''.
    # ======================================================================= #
    @internal_hash[:leading_colour_component] = ''.dup
    # ======================================================================= #
    # === :use_italic
    # ======================================================================= #
    @internal_hash[:use_italic] = false
    # ======================================================================= #
    # === :use_underline
    # ======================================================================= #
    @internal_hash[:use_underline] = false
    # ======================================================================= #
    # === :slow_blink
    # ======================================================================= #
    @internal_hash[:slow_blink] = false
    # ======================================================================= #
    # === :use_this_background_colour
    # ======================================================================= #
    @internal_hash[:use_this_background_colour] = nil
    # ======================================================================= #
    # === :use_this_colour
    #
    # We must use a default colour.
    # ======================================================================= #
    @internal_hash[:use_this_colour] = '0'
    # ======================================================================= #
    # === :reversed
    #
    # If this is true then "7" will be used, which will invert foreground
    # colour and background colour.
    # ======================================================================= #
    @internal_hash[:reversed] = false
  end

  # ========================================================================= #
  # === reversed?
  # ========================================================================= #
  def reversed?
    @internal_hash[:reversed]
  end

  # ========================================================================= #
  # === use_this_background_colour?
  #
  # Background colours go from e. g. 40 to 47, give or take.
  # ========================================================================= #
  def use_this_background_colour?
    @internal_hash[:use_this_background_colour]
  end

  # ========================================================================= #
  # === set_use_this_background_colour
  # ========================================================================= #
  def set_use_this_background_colour(i)
    @internal_hash[:use_this_background_colour] = i
  end

  # ========================================================================= #
  # === reversed
  # ========================================================================= #
  def reversed
    @internal_hash[:reversed] = true
    self
  end

  # ========================================================================= #
  # === set_use_underline_to_true
  # ========================================================================= #
  def set_use_underline_to_true
    @internal_hash[:use_underline] = true
  end

  # ========================================================================= #
  # === set_slow_blink_to_true
  # ========================================================================= #
  def set_slow_blink_to_true
    @internal_hash[:slow_blink] = true
  end

  # ========================================================================= #
  # === raw_content?
  # ========================================================================= #
  def raw_content?
    @internal_hash[:raw_content]
  end; alias content? raw_content? # === content?
       alias string?  raw_content? # === string?

  # ========================================================================= #
  # === use_this_colour?
  # ========================================================================= #
  def use_this_colour?
    @internal_hash[:use_this_colour]
  end

  # ========================================================================= #
  # === use_underline?
  # ========================================================================= #
  def use_underline?
    @internal_hash[:use_underline]
  end

  # ========================================================================= #
  # === slow_blink?
  # ========================================================================= #
  def slow_blink?
    @internal_hash[:slow_blink]
  end

  # ========================================================================= #
  # === display
  # ========================================================================= #
  def display
    print build_main_string(content?)
  end; alias report display# === report

  # ========================================================================= #
  # === set_content
  # ========================================================================= #
  def set_content(i)
    @internal_hash[:raw_content] = i.dup
  end

  # ========================================================================= #
  # === trailing_end?
  # ========================================================================= #
  def trailing_end?
    TRAILING_PART
  end

  # ========================================================================= #
  # === leading_part?
  # ========================================================================= #
  def leading_part?
    "#{LEADING_PART}"
  end

  # ========================================================================= #
  # === slow_blink
  # ========================================================================= #
  def slow_blink
    set_slow_blink_to_true
    self
  end

  # ========================================================================= #
  # === underline
  # ========================================================================= #
  def underline
    set_use_underline_to_true
    self
  end; alias underlined underline # === underlined

  # ========================================================================= #
  # === increased_intensity?
  # ========================================================================= #
  def increased_intensity?
    _ = @internal_hash[:leading_colour_component]
    return (_ and !_.empty?)
  end

  # ========================================================================= #
  # === string_increased_intensity?
  # ========================================================================= #
  def string_increased_intensity?
    _ = @internal_hash[:leading_colour_component]
    if _ and !_.empty?
      return "#{_}" # ;"
    end
    return ''
  end

  # ========================================================================= #
  # === bright
  # ========================================================================= #
  def bright
    @internal_hash[:leading_colour_component] = '1'
    self
  end; alias bold bright # === bold

  # ========================================================================= #
  # === background
  # ========================================================================= #
  def background(i = :yellow)
    i = i.to_s
    unless i.start_with? 'background_'
      i = i.dup if i.frozen?
      i.prepend('background_')
    end
    i = i.to_sym
    unless HASH_ANSI_COLOURS.has_key? i
      e 'Warning: the key '+i.to_s+' is not registered in the main Hash.'
    end
    i = HASH_ANSI_COLOURS[i].to_s
    set_use_this_background_colour(i)
    self
  end; alias bg background # === bg

  # ========================================================================= #
  # === set_use_this_colour
  # ========================================================================= #
  def set_use_this_colour(i)
    @internal_hash[:use_this_colour] = i
  end

  # ========================================================================= #
  # === +
  # ========================================================================= #
  def +(i = '')
    append(i)
  end

  # ========================================================================= #
  # === append
  # ========================================================================= #
  def append(i)
    "#{to_str}#{i.to_str}"
  end

  # ========================================================================= #
  # === build_the_main_string (main tag)
  #
  # This method must always rebuild the full, modified content.
  # ========================================================================= #
  def build_the_main_string
    result = ''.dup
    result << leading_part?
    result << '['
    result << string_increased_intensity?
    result << ";#{use_this_colour?}" unless use_this_colour? == '0'
    if use_this_background_colour?
      result << ";#{use_this_background_colour?}"
    end
    result << ';3' if use_italic?
    result << ';4' if use_underline?
    result << ';5' if slow_blink?
    result << ';7' if reversed?
    result << 'm'
    result << raw_content?
    result << trailing_end?
    return result
  end; alias to_str            build_the_main_string # === to_str
       alias to_s              build_the_main_string # === to_s
       alias build_main_string build_the_main_string # === build_main_string

  # ========================================================================= #
  # === red
  # ========================================================================= #
  def red
    real_name_of_the_method = HASH_ANSI_COLOURS[__callee__]
    set_use_this_colour(real_name_of_the_method)
    self
  end; alias blue          red # === blue
       alias black         red # === black
       alias green         red # === green
       alias brown         red # === brown
       alias yellow        red # === yellow
       alias purple        red # === purple
       alias magenta       red # === magenta
       alias cyan          red # === cyan
       alias light_blue    red # === light_blue
       alias light_gray    red # === light_gray
       alias light_green   red # === light_green
       alias light_magenta red # === light_magenta
       alias white         red # === white
       alias light_red     red # === light_red
       alias grey          red # === grey

  # ========================================================================= #
  # === italic
  # ========================================================================= #
  def italic
    @internal_hash[:use_italic] = true
    self
  end

  # ========================================================================= #
  # === use_italic?
  # ========================================================================= #
  def use_italic?
    @internal_hash[:use_italic]
  end

  # ========================================================================= #
  # === Colours::Colours[]
  # ========================================================================= #
  def self.[](i = ARGV)
    new(i)
  end

end

# =========================================================================== #
# === Colours.fancy_parse
#
# This method can be used to parse a line containing special tags,
# such as <i> or <teal>, and replace them with the corresponding
# ASCII code for the commandline. The second example, e. g. <teal>,
# is known as a HTML colour tag (htmlcolour).
#
# The idea for this method is to simply use it to quickly colourize
# some arbitrary text.
#
# The method will only return the resulting, sanitizing String;
# if you wish to display this on the terminal/commandline then
# you will have to use puts or print on your own.
#
# Usage examples:
#
#   alias e puts
#   e Colours.fancy_parse '<i><red>E. coli</red></i>'
#   e Colours.fancy_parse '<i><lightgreen>E. coli</lightgreen></i>'
#   e Colours.fancy_parse "<lightgreen><i>hey</i></lightgreen> <teal>there</teal>"
#   e Colours.fancy_parse "<tomato>hey</tomato> <teal>there</teal>"
#   e Colours.fancy_parse "<tomato><i>hey</i></tomato> <teal>there</teal>"
#   e Colours.fancy_parse "<tomato><b>Hello world.</b></tomato>"
#   e Colours.fancy_parse "<tomato>Hello world.</tomato>"
#   e Colours.fancy_parse '<i>hey</i> <teal>there</teal>'
#   e Colours.fancy_parse '<i>hey</i> <teal>there</teal> ok'
#   e Colours.fancy_parse "<lightgreen><i>hey</i></lightgreen> <teal>there</teal>"
#   e Colours.fancy_parse "<tomato>hey</tomato> <teal>there</teal>"
#   e Colours.fancy_parse "<tomato><i>hey</i></tomato> <teal>there</teal>"
#   e Colours.fancy_parse "<tomato><b><i>hey</i></b></tomato> <teal>there</teal>"
#   e Colours.fancy_parse "<tomato><b><i>hey</i></b></tomato> <teal>there</teal> <b>how do you do</b>"
#   e Colours.fancy_parse "<tomato><b>Hello world.</b></tomato>"
#
# =========================================================================== #
def self.fancy_parse(
    i               = '',
    use_this_colour = :default_colour
  )
  if i.is_a? Array
    i = i.join # We do not want an Array past this point.
  end
  # ========================================================================= #
  # === Handle italic tags, aka <i>
  #
  # The old code was:
  #
  #  i.gsub!(use_this_regex, return_italic('\1'))
  #
  # Up until November 2022, then it was changed.
  # ========================================================================= #
  if i.include?('<i>') and i.include?('</i>')
    # ======================================================================= #
    # The following regex is defined at:
    #
    #   https://rubular.com/r/k5tE9xBzky8muR
    #
    # ======================================================================= #
    use_this_regex = /<i>([<>\/\.A-Za-zöäüÖÄÜ\s-]*?)<\/i>/
    i = i.dup if i.frozen?
    i.gsub!(
      use_this_regex,
      ::Colours::Colours.new('\1').italic.to_s+ # or return_italic('\1')
      ::Colours.rev
    )
  end
  # ========================================================================= #
  # === Handle bold tags, aka <b>
  # ========================================================================= #
  if i.include?('<b>') and i.include?('</b>')
    use_this_regex = /<b>(.+)<\/b>/
    i.gsub!(
      use_this_regex,
      ::Colours::Colours.new('\1').bold.to_s # or return_bold('\1')
    )
  end
  # ========================================================================= #
  # Finally, handle all HTML colours in the given line.
  # ========================================================================= #
  i = replace_all_html_colours_in_this_line(i, use_this_colour)
  return i
end

# =========================================================================== #
# === Colours.new
#
# Easier toplevel-instantiation.
# =========================================================================== #
def self.new(i = ARGV)
  ::Colours::Colours.new(i)
end

end

# =========================================================================== #
# === Colours()
#
# And a toplevel method that can be used to create a new instance of
# class Colours::Colours.
# =========================================================================== #
def Colours(i)
  ::Colours::Colours.new(i)
end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  _ = 'Hey there'
  e Colours(_).red
  e Colours(_).blue
  e Colours(_).light_magenta
  e Colours(_).yellow
  e Colours(_).purple
  e Colours(_).white
  e Colours(_).light_red
  # e Colours(_).light_green
  # e Colours(_).light_blue
  # e Colours(_).grey
  # e Colours(_).cyan
  e
  e Colours('This will be shown in the colour red.').red
  e
  e Colours('This is red').red+
    ' and '+
    Colours('this on yellow bg').bg(:yellow)
  e
  e Colours('This is red').red+' and '+
    Colours('this should be tilted').italic+'.'
  e
  e Colours('This is red, tilted').red.italic+'.'
  e
  e Colours('This is blue, tilted').blue.italic+'.'
  e
  e Colours('This is green, tilted').green.italic+'.'
  e
  e Colours('This is brown, tilted').brown.italic+'.'
  e
  e Colours('This is purple, tilted').purple.italic+'.'
  e
  e Colours('This is cyan, tilted').cyan.italic+'.'
  e
  e Colours('This is light gray, tilted').light_gray.italic+'.'
  e
  e Colours('This is underlined').underlined+'.'
  e
  e Colours('This is red').red +
    ' and '+
    Colours("this on yellow bg").bg(:yellow)+
    ' and '+
    Colours("even bright underlined!").underline.bright
  e
  e Colours('This is underlined and italic').underlined.italic+'.'
  e
  e Colours('This is red, underlined').red.underlined+'.'
  e
  e Colours('This is blue, underlined').blue.underlined+'.'
  e
  e Colours('This is green, underlined').green.underlined+'.'
  e
  e Colours('This is brown, underlined').brown.underlined+'.'
  e
  e Colours('This is purple, underlined').purple.underlined+'.'
  e
  e Colours('This is cyan, underlined').cyan.underlined+'.'
  e
  e Colours('This is light gray, underlined').light_gray.underlined+'.'
  e
  e Colours('This is blue, underlined').blue.underline+'.'
  e
  e Colours('This is blue.').blue
  e
  e Colours('This is bright blue.').blue.bright
  e
  e Colours('This is bright blue, slow blinking.').blue.bright.slow_blink
  e
end # fancy_parse "<tomato><b><i>hey</i></b></tomato> <teal>there</teal>"