#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Colours::Module256Colours
# =========================================================================== #
# require 'colours/module_256_colours/module_256_colours.rb'
# include ::Colours::Module256Colours
# =========================================================================== #
module Colours

module Module256Colours # === Colours::Module256Colours

  require 'yaml'
  require 'colours/constants/constants.rb'
  # require 'colours/autogenerated/256_colours_methods.rb'
  # require 'colours/autogenerated/256_colours_instance_methods.rb'

  # ========================================================================= #
  # === Colours::Module256Colours::BLOCK_START
  # ========================================================================= #
  BLOCK_START = "\x1b[48;5;"

  # ========================================================================= #
  # === Colours::Module256Colours::BLOCK_END
  # ========================================================================= #
  BLOCK_END = "\x1b[0m"

  # ========================================================================= #
  # === Colours::Module256Colours.autogenerate_256_colours_methods (autogenerate tag)
  #
  # This method will autogenerate all methods that can be used via
  # Colours::Module256Colours.
  #
  # Commandline invocation example:
  #
  #   colours --autogenerate_256_colours_methods
  #
  # ========================================================================= #
  def self.autogenerate_256_colours_methods(
      into = HOME_DIRECTORY_OF_USER_X+
             'programming/ruby/src/'\
             'colours/lib/colours/autogenerated/'\
             '256_colours_methods.rb'
    )
    require 'save_file'
    dataset_from_the_yaml_file = YAML.load_file(::Colours.file_256_colours)
    _ = "#{RUBY_HEADER}\n".dup
    _ << "# require 'colours/autogenerated/#{File.basename(into)}'\n"
    _ << "# =========================================================================== #\n"
    _ << "# include ::Colours::Module256Colours\n"
    _ << "# =========================================================================== #\n"
    _ << "module Colours\n\n"
    _ << "module Module256Colours # === ::Colours::Module256Colours\n\n"
    dataset_from_the_yaml_file.keys.each {|this_html_colour|
      this_html_colour = this_html_colour.dup if this_html_colour.frozen?
      pointer = dataset_from_the_yaml_file[this_html_colour.to_s] # Pointer must come first.
      this_html_colour.downcase! # To turn 'LightGoldenrod1' into 'lightgoldenrod1'.
      # ===================================================================== #
      # We must obtain the proper entry from the .yml file next:
      # ===================================================================== #
      _ << "  # ========================================================================= #\n"
      _ << "  # === Colours::Module256Colours.#{this_html_colour}\n"
      _ << "  # ========================================================================= #\n"
      _ << "  def self.#{this_html_colour}(i = '', &block)\n"
      _ << '    return_this_256_colour('+pointer.first.to_s+', i)'+"\n"
      _ << "  end\n\n"
    }
    _ << "end\n\n"
    _ << "end\n"
    puts "Storing into the file `#{into}` next."
    SaveFile.write_what_into(_, into)
    autogenerate_256_colours_methods_for_instance_methods
  end

  # ========================================================================= #
  # === Colours::Module256Colours.autogenerate_256_colours_methods_for_instance_methods
  #
  # This method will generate all instance-methods for the 256-colours
  # part.
  # ========================================================================= #
  def self.autogenerate_256_colours_methods_for_instance_methods(
      into = HOME_DIRECTORY_OF_USER_X+
             'programming/ruby/src/'\
             'colours/lib/colours/autogenerated/'\
             '256_colours_instance_methods.rb'
    )
    require 'save_file'
    dataset_from_the_yaml_file = YAML.load_file(::Colours.file_256_colours)
    _ = "#{RUBY_HEADER}\n".dup
    _ << "# require 'colours/autogenerated/#{File.basename(into)}'\n"
    _ << "# =========================================================================== #\n"
    _ << "# include ::Colours::Module256Colours\n"
    _ << "# =========================================================================== #\n"
    _ << "module Colours\n\n"
    _ << "module Module256Colours # === ::Colours::Module256Colours\n\n"
    dataset_from_the_yaml_file.keys.each {|this_html_colour|
      this_html_colour = this_html_colour.dup if this_html_colour.frozen?
      this_html_colour.downcase! # To turn 'LightGoldenrod1' into 'lightgoldenrod1'.
      # ===================================================================== #
      # We must obtain the proper entry from the .yml file next:
      # ===================================================================== #
      _ << "  # ========================================================================= #\n"
      _ << "  # === #{this_html_colour}\n"
      _ << "  # ========================================================================= #\n"
      _ << "  def #{this_html_colour}(i = '', &block)\n"
      _ << '    ::Colours::Module256Colours.'+this_html_colour.to_s+'(i)'+"\n"
      _ << "  end\n\n"
    }
    _ << "end\n\n"
    _ << "end\n"
    puts "Storing into the file `#{into}` next."
    SaveFile.write_what_into(_, into)
  end

  # ========================================================================= #
  # === Colours::Module256Colours.load_the_default_dataset
  # ========================================================================= #
  def self.load_the_default_dataset(
      i = FILE_256_COLOURS
    )
    # ======================================================================= #
    # === @dataset
    #
    # This is the dataset that holds the content of the .yml file specifying
    # the (xterm) 256 colours.
    #
    # We will keep the keys in a downcased variant, as this will make it
    # easier to use them as method-names.
    # ======================================================================= #
    if File.exist?(i) # Must safeguard.
      @dataset = YAML.load_file(i).
                      transform_keys(&:downcase)
    else
      @dataset = nil
    end
  end

  # ========================================================================= #
  # === Colours::Module256Colours.return_this_256_colour         (return tag)
  #
  # The first argument to this method should be a number from 0 to 255.
  #
  # For example, 255 is also known as "Grey93" - a variant of grey.
  #
  # It appears to work better if you use print() rather than puts() for
  # whatever is returned here - perhaps due to the leading \u001b part.
  #
  # You can also input the colour variant, such as 'Grey93'. 
  #
  # Invocation examples:
  #
  #   Colours::Module256Colours.return_this_256_colour(255, "Hello world in grey!\n")
  #   print Colours::Module256Colours.return_this_256_colour(255, (Roebe.block_character*5)+"  Hello world in grey!  "+(Roebe.block_character*5)+" \n")
  #   print Colours::Module256Colours.return_this_256_colour('DarkTurquoise', (Roebe.block_character*5)+"  Hello world in DarkTurquoise!  "+(Roebe.block_character*5)+" \n")
  #
  # ========================================================================= #
  def self.return_this_256_colour(
      id            = 9,
      use_this_text = "Hello world!",
      dataset       = @dataset
    )
    id = id.to_s.downcase
    if (id !~ /^\d+/) and
      dataset and
      dataset.has_key?(id)
      id = dataset[id] # Query the real value from the dataset.
    end
    "\u001b[38;5;#{id}m#{use_this_text}#{REVERT}"
  end

  # ========================================================================= #
  # === Colours::Module256Colours.dataset?
  # ========================================================================= #
  def self.dataset?
    @dataset
  end

  # ========================================================================= #
  # === Colours::Module256Colours.is_this_a_256_colour?
  #
  # Invocation example:
  #
  #   Colours::Module256Colours.is_this_a_256_colour? 'LightSteelBlue3'
  #
  # ========================================================================= #
  def self.is_this_a_256_colour?(
      i,
      dataset = dataset?
    )
    if dataset
      dataset.keys.map(&:downcase).include?(i.to_s.downcase) # The input argument will be downcased as well.
    else
      false
    end
  end

  # ========================================================================= #
  # === Colours::Module256Colours.display_this_256_colour
  #
  # The first argument should be a number from 0 to 255.
  #
  # You can also batch-output all colours, by using something like:
  #
  #   Colours.display_this_256_colour('0-255',"Hello world, in a batch!\n")
  #
  # Generic usage examples:
  #
  #   Colours::Module256Colours.display_this_256_colour(33, 'yo there') { :newline }
  #   Colours::Module256Colours.display_this_256_colour(55, 'yo there') { :newline }
  #   Colours::Module256Colours.display_this_256_colour(77, 'yo there') { :newline }
  #
  # ========================================================================= #
  def self.display_this_256_colour(
      id            = 9,
      use_this_text = "Hello world!\n",
      &block
    )
    # ======================================================================= #
    # === Handle blocks given to this method next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :newline
      # ===================================================================== #
      when :newline
        use_this_text = use_this_text.dup if use_this_text.frozen?
        use_this_text << "\n"
      end
    end
    if id.is_a?(String) and id.include?('-')
      # ===================================================================== #
      # Assume pseudo-range input.
      # ===================================================================== #
      splitted = id.split('-')
      range = (splitted.first.to_i .. splitted.last.to_i).to_a
      range.each {|this_range_id|
        display_this_256_colour(this_range_id, use_this_text, &block)
      }
    else # else print the 256-colour
      print ::Colours::Module256Colours.return_this_256_colour(id, use_this_text)
    end
  end; self.instance_eval { alias display_this_in_256_colour display_this_256_colour } # === Colours::Module256Colours.display_this_in_256_colour

  # ========================================================================= #
  # === is_this_a_256_colour?
  # ========================================================================= #
  def is_this_a_256_colour?(i)
    ::Colours::Module256Colours.is_this_a_256_colour?(i)
  end

  # ========================================================================= #
  # === Colours::Module256Colours.write_this_in_256_colours
  #
  # Similar to the other write method, but has the arguments reversed. 
  # ========================================================================= #
  def self.write_this_in_256_colours(
      this_text = "Hello world!\n",
      id        = rand(256),
      &block
    )
    display_this_256_colour(id, this_text, &block)
  end; self.instance_eval { alias write_in_256_colour write_this_in_256_colours } # === Colours::Module256Colours.write_in_256_colour

  # ========================================================================= #
  # === write_in_256_colour
  #
  # Include-able method to the above module-level instance.
  # ========================================================================= #
  def write_in_256_colour(
      i  = 'hi there',
      id = 33
    )
    write_this_in_256_colours(i, id)
  end

  # ========================================================================= #
  # === Colours::Module256Colours.show_256_colour_cube
  #
  # This will show the 256-colours cube. Colours 16-231 are a
  # 6x6x6 color cube.
  # ========================================================================= #
  def self.show_256_colour_cube
    e
    e 'Color cube, 6x6x6 blocks:'
    e
    0.upto(5) {|green|
      0.upto(5) {|red|
        0.upto(5) {|blue|
          colour = 16 + (red * 36) + (green * 6) + blue
          print "#{BLOCK_START}#{colour}m  " # This creates a block.
        }
        print "#{BLOCK_END} "
      }
      e
    }
  end

  # ========================================================================= #
  # === Colours::Module256Colours.show_all_256_colours
  # ========================================================================= #
  def self.show_all_256_colours(&block)
    yielded = nil
    if block_given?
      yielded = yield
    end
    0.upto(255).each {|this_number|
      result = this_number.to_s.rjust(4)+' '+
               return_this_256_colour(this_number).to_s.dup
      if yielded and yielded == :newline
        result << "\n"
      end
      print result
    }
  end

end

# =========================================================================== #
# === Colours.show_256_colour_cube
# =========================================================================== #
def self.show_256_colour_cube
  ::Colours::Module256Colours.show_256_colour_cube
end

# ========================================================================= #
# === Colours.display_this_256_colour
# ========================================================================= #
def self.display_this_256_colour(
    id            = 9,
    use_this_text = "Hello world!\n",
    &block
  )
  Colours::Module256Colours.display_this_256_colour(
    id,
    use_this_text,
    &block
  )
end

# =========================================================================== #
# === Colours.is_this_a_256_colour?
#
# Invocation example:
#
#   Colours.is_this_a_256_colour? 'LightSteelBlue3'
#
# =========================================================================== #
def self.is_this_a_256_colour?(
    i,
    dataset = ::Colours::Module256Colours.dataset?
  )
  ::Colours::Module256Colours.is_this_a_256_colour?(i, dataset)
end

# =========================================================================== #
# === Colours.show_all_256_colours
# =========================================================================== #
def self.show_all_256_colours(&block)
  ::Colours::Module256Colours.show_all_256_colours(&block)
end

# =========================================================================== #
# === Colours.map_this_symbol_to_that_256_colour
#
# This method will take input such as :bold_green and associate it with
# a corresponding 256-colour. Thus, the resulting Symbol that is
# found here should correspond to a method on the module
# Module256Colours.
# =========================================================================== #
def self.map_this_symbol_to_that_256_colour(i)
  case i
  # ========================================================================= #
  # === :boldgreen
  # ========================================================================= #
  when :boldgreen,
       :bold_green
    :chartreuse2
  # ========================================================================= #
  # === :teal
  # ========================================================================= #
  when :teal
    :teal
  # ========================================================================= #
  # === :bold_yellow
  # ========================================================================= #
  when :bold_yellow
    :yellow2
  # ========================================================================= #
  # === :gold
  # ========================================================================= #
  when :gold
    :gold3
  # ========================================================================= #
  # === :bold_blue
  # ========================================================================= #
  when :bold_blue
    :lightslateblue
  when :bold_brown
    :darkred
  when :brown
    :maroon
  when :pink
    :hotpink3
  when :lightgrey,
       :light_grey
    :grey66
  when :bold_red
    :red1
  when :lightblue,
       :light_blue
    :deepskyblue3
  when :grey
    :grey53
  when :green
    :chartreuse4
  else
    e "Not known input to method #{__method__.to_s}: #{i}"
  end
end

# =========================================================================== #
# === Colours.write_this_in_256_colours
#
# Similar to the other write method, but has the arguments reversed. 
# =========================================================================== #
def self.write_this_in_256_colours(
    this_text = "Hello world!\n",
    id        = rand(256),
    &block
  )
  ::Colours::Module256Colours.display_this_256_colour(id, this_text, &block)
end; self.instance_eval { alias write_in_256_colour write_this_in_256_colours } # === Colours.write_in_256_colour

# =========================================================================== #
# === Colours.autogenerate_256_colours_methods
# =========================================================================== #
def self.autogenerate_256_colours_methods
  Colours::Module256Colours.autogenerate_256_colours_methods
end; self.instance_eval { alias autogenerate_the_module_for_the_256_colours autogenerate_256_colours_methods } # === Colours.autogenerate_the_module_for_the_256_colours

end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  Colours::Module256Colours.load_the_default_dataset
  dataset = Colours::Module256Colours.dataset?
  e 'We have '+dataset.keys.size.to_s+' entries.'
  Colours.display_this_256_colour(55,  'yo there') { :newline }
  Colours.display_this_256_colour(75,  'yo there') { :newline }
  Colours.display_this_256_colour(125, 'yo there') { :newline }
  e 'Ok - the above three lines tested the method Colours.display_this_256_colour().'
  e
  e 'Next we will call Colours.show_256_colour_cube() and other related methods.'
  Colours.show_256_colour_cube
  # The next line can be found in: colours/test/testing_256_colours_support.rb
  # Colours.test_256_colours_support
  e
  Colours.display_this_256_colour('0-255',"Hello world, in a batch!\n")
  e
  print "\x1b[48;5;15000m      TESTING    \x1b[0m" # This creates a block.
  e
end # supportfor256colours