#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'colours/sfile_sdir_sfancy_swarn_simp_scomments_and_ssymlink/sfile_sdir_sfancy_swarn_simp_scomments_and_ssymlink.rb'
# =========================================================================== #
module Colours

  require 'colours/toplevel_methods/rev.rb'
  require 'colours/module_256_colours/module_256_colours.rb'
  require 'colours/toplevel_methods/map_this_symbol_to_that_html_colour.rb'

  # ========================================================================= #
  # === @use_html_colours
  #
  # This is a separate variable so that we can selectively disable or
  # enable the colours for the Konsole submodule. Some terminals do
  # not respond nicely to the R,G,B values that the Konsole submodule
  # may require.
  #
  # Since as of November 2023 this variable is semi-deprecated, but
  # not entirely disabled.
  # ========================================================================= #
  @use_html_colours = true

  # ========================================================================= #
  # === @colour_table                      (default tag, default colours tag)
  #
  # This toplevel instance variable has the default colours assigned to
  # the different methods. If you ever wish to change the default colours
  # in use, you can do so by manipulating this instance variable.
  #
  # The colour-table is a table that maps from e. g. sfancy() to the
  # corresponding colour at hand.
  #
  # Keep it sorted alphabetically.
  # ========================================================================= #
  @colour_table = {
    default_colour:  :green,
    sfancy:          :bold_green,   # === sfancy()
    # sargument:       :bold_green,   # === sargument()
    scomments:       :grey,         # === scomments()
    sdir:            :chocolate,    # === sdir()
    sfile:           :saddlebrown,  # === sfile()
    simportant:      :gold,         # === simportant()
    ssymlink:        :lightblue,    # === ssymlink()
    swarn:           :bold_red,     # === swarn()
    # snormal:         :light_grey    # === snormal()
  }

  # ========================================================================= #
  # === Colours::HASH_NEW_COLOURS
  #
  # This is the new hash, which we can transition-into if necessary.
  #
  # Since as of the year 2018, this Hash is rarely used anymore, though.
  #
  # If you want to use these colours nonetheless, you can use this simpler API:
  #
  #    Colourss.set_method_to_colour_table(:new_colours)
  #
  # ========================================================================= #
  HASH_NEW_COLOURS = {
    default_colour: :green,        
    # sargument:    :bold_brown,       # === sarg()      # removed in November 2023.
    # scomments:    :bold_blue,        # === scomments() # removed in November 2023.
    sdir:           :thistle,          # === sdir()
    sfancy:         :rosybrown,        # === sfancy()
    sfile:          :brown,            # === sfile()
    simportant:     :pink,             # === simportant()
    ssymlink:       :teal,             # === ssymlink()
    swarn:          :olivedrab,        # === swarn
    snormal:        :mediumaquamarine, # === snormal()
    scomments:      :grey,             # === scomments()
  }

  # ========================================================================= #
  # === Colours.update_the_colour_codes_with                     (update tag)
  # ========================================================================= #
  def self.update_the_colour_codes_with(
      i = HASH_NEW_COLOURS
    )
    @colour_table.update(i)
  end; self.instance_eval { alias update_colour_table_with update_the_colour_codes_with } # === Colours.update_colour_table_with
       self.instance_eval { alias update_the_colour_codes  update_the_colour_codes_with } # === Colours.update_the_colour_codes
       self.instance_eval { alias update_colour_codes      update_the_colour_codes_with } # === Colours.update_colour_codes

  # ========================================================================= #
  # === Colours.sfancy
  #
  # Invocation example:
  #
  #   Colours.sfancy('Hello world!')
  #
  # ========================================================================= #
  def self.sfancy(
      i            = '',
      make_newline = false
    )
    # ======================================================================= #
    # Fetch the appropriate replacement-colour next:
    # ======================================================================= #
    replacement_colour = @colour_table[__method__.to_sym].to_sym
    unless ::Colours::HtmlColoursMethods.respond_to?(replacement_colour.to_sym)
      replacement_colour = ::Colours.map_this_symbol_to_that_html_colour(replacement_colour.to_sym)
    end
    return ::Colours::HtmlColoursMethods.send(replacement_colour, i)
  end

  # ========================================================================= #
  # === use_new_colour_codes
  #
  # Use this method to assign new colour codes.
  # We assume that you must pass a hash to this method.
  #
  # Usage example:
  #
  #   new_hash = {
  #     :sfancy         => 'green',
  #     :simportant     => 'teal',
  #     :default_colour => 'grey',
  #     :sfile          => 'magenta',
  #     :sdir           => 'cyan'
  #   }
  #   Colours.assign_new_colour_codes(new_hash)
  #
  # ========================================================================= #
  def use_new_colour_codes(
      i = HASH_NEW_COLOURS
    )
    @colour_table.update(i)
  end; alias assign_new_colour_codes use_new_colour_codes # === assign_new_colour_codes

  # ========================================================================= #
  # === Colours.set_colour_table
  #
  # This method can be used to modify the instance variable
  # @colour_table.
  #
  # The input to this method should be a Hash. It can also be a Symbol,
  # in which case we will try to find a Hash that corresponds to that
  # Symbol.
  #
  # If in doubt, pass in a Hash though.
  # ========================================================================= #
  def self.set_colour_table(
      i = :new_colours
    )
    case i
    # ======================================================================= #
    # === :new_colours
    # ======================================================================= #
    when :new_colours,
         :new_colour_table,
         :default
      i = HASH_NEW_COLOURS
    end
    # ======================================================================= #
    # Note that we must ensure that all the necessary keys are there
    # in the given input. If this is not the case, then we will
    # determine which entries are missing, and assign them from the
    # instance variable @colour_table.
    # ======================================================================= #
    missing_entries = (
      i.keys - @colour_table.keys
    ).flatten
    unless missing_entries.empty?
      # ===================================================================== #
      # Push them onto i in this case.
      # ===================================================================== #
      missing_entries.each {|this_key|
        if i.has_key? this_key
          # All fine, pass through in this case.
        else
          add_this = missing_entries[this_key]
          i[key] = add_this

        end
      }
    end
    @colour_table = i
  end; self.instance_eval { alias assign_new_colour_codes    set_colour_table } # === Colours.assign_new_colour_codes
       self.instance_eval { alias set_method_to_colour_table set_colour_table } # === Colours.set_method_to_colour_table

  # ========================================================================= #
  # === Colours.sfile
  #
  # This method can be explicitely used to colour files, such as
  # '/opt/foobar.rb'.
  #
  # Invocation examples:
  #
  #   Colours.sfile('/opt/foobar.rb')
  #   Colours.sfile('/tmp/test.md')
  #   Colours.sfile('Hello world!')
  #
  # ========================================================================= #
  def self.sfile(
      i            = '',
      make_newline = false
    )
    replacement_colour = @colour_table[__method__.to_sym].to_sym
    if ::Colours.const_defined?(:HtmlColoursMethods)
      return ::Colours::HtmlColoursMethods.send(replacement_colour, i)
    end
  end

  # ========================================================================= #
  # === Colours.colour_table?
  #
  # This method will feedback the available colour-table.
  #
  # To invoke this method, try the following code:
  #
  #   pp Colours.colours?
  #
  # ========================================================================= #
  def self.colour_table?
    @colour_table
  end; self.instance_eval { alias method_to_colour_table                  colour_table? } # === Colours.method_to_colour_table
       self.instance_eval { alias method_to_colour_table?                 colour_table? } # === Colours.method_to_colour_table?
       self.instance_eval { alias colours?                                colour_table? } # === Colours.colours?
       self.instance_eval { alias table?                                  colour_table? } # === Colours.table?
       self.instance_eval { alias table_colour_methods_to_specific_colour colour_table? } # === Colours.table_colour_methods_to_specific_colour
       self.instance_eval { alias colour_methods_to_specific_colour?      colour_table? } # === Colours.colour_methods_to_specific_colour?

  # ========================================================================= #
  # === Colours.use_new_colour_table
  # ========================================================================= #
  def self.use_new_colour_table
    set_colour_table(:new_colour_table)
  end

  # ========================================================================= #
  # === Colours.show_the_colour_table
  #
  # This will simply pretty-print the current colour table.
  # ========================================================================= #
  def self.show_the_colour_table
    pp @colour_table
  end

  # ========================================================================= #
  # === Colours.shuffle
  #
  # Shuffle the above table. The available colours to take here will
  # be from the HtmlColours namespace.
  # ========================================================================= #
  def self.shuffle
    # ======================================================================= #
    # Point to the html "table" to use.
    # ======================================================================= #
    _ = ::Colours::HtmlColours.all?.shuffle
    @colour_table = {
      simp:      _.shift,
      sdir:      _.shift,
      sfile:     _.shift,
      swarn:     _.shift,
      sfancy:    _.shift,
      ssymlink:  _.shift
    }
  end

  # ========================================================================= #
  # === edir
  # ========================================================================= #
  def edir(
      i            = '',
      make_newline = false
    )
    e ::Colours.sdir(i, make_newline)
  end

  # ========================================================================= #
  # === Colours.ssymlink
  # ========================================================================= #
  def self.ssymlink(
      i            = '',
      make_newline = false
    )
    replacement_colour = @colour_table[__method__.to_sym].to_sym
    return ::Colours::HtmlColoursMethods.send(replacement_colour, i)
  end; self.instance_eval { alias sym  ssymlink } # === Colours.sym
       self.instance_eval { alias ssym ssymlink } # === Colours.ssym

  # ========================================================================= #
  # === Colours.default_colour
  #
  # Invocation example:
  #
  #   puts Colours.default_colour('Hello world!')
  #
  # ========================================================================= #
  def self.default_colour(
      i            = '',
      make_newline = false
    )
    if use_colours?
      if make_newline
        i = "#{i}#{N}"
      end
      if @use_html_colours
        replacement_colour = @colour_table[__method__.to_sym].to_sym
        return ::Colours::HtmlColoursMethods.send(replacement_colour, i)
      end
    end
    return i
  end

  # ========================================================================= #
  # === Colours.colour_for_symlinks
  #
  # This method will simply return the colour for symlinks.
  # ========================================================================= #
  def self.colour_for_symlinks
    return @colour_table[:ssymlink].to_sym
  end

  # ========================================================================= #
  # === ssymlink
  #
  # Note that a symlink called symlink() exists to this method, but it
  # is not exactly clear whether this alias will be kept, as it may
  # interfere with some other methods when we do an include-action.
  # ========================================================================= #
  def ssymlink(i = '')
    ::Colours.ssymlink(i)
  end; alias symlink ssymlink # === symlink
       alias ssym    ssymlink # === ssym
       alias slink   ssymlink # === slink

  # ========================================================================= #
  # === default_colour
  # ========================================================================= #
  def default_colour(i = '')
    ::Colours.default_colour(i)
  end

  # ========================================================================= #
  # === edefault_colour
  # ========================================================================= #
  def edefault_colour(
      i            = '',
      make_newline = false
    )
    e ::Colours.default_colour(i, make_newline)
  end

  # ========================================================================= #
  # === Colours.ecomment
  #
  # This method was added on Nov 2013.
  #
  # In April 2014, the ability to split at any arbitrary character was
  # added, which can be controlled via the second argument given to
  # that method. By default it will split on the token '#'.
  # ========================================================================= #
  def self.ecomment(
      i,
      optional_split_at_this_character = '#', # ← Specify which character to split at.
      first_colour_to_use         = :steelblue,
      second_colour_to_use        = :seagreen,
      colour_to_use_for_the_token = :slategray
    )
    _ = i.to_s
    if optional_split_at_this_character.is_a? Hash
      # ===================================================================== #
      # === :token
      # ===================================================================== #
      if optional_split_at_this_character.has_key? :token
        optional_split_at_this_character = optional_split_at_this_character.delete :token
      end
    end
    optional_split_at_this_character = optional_split_at_this_character.to_s
    # ======================================================================= #
    # === If we can find a token to split at
    # ======================================================================= #
    if _.include? optional_split_at_this_character
      _ = _.dup if _.frozen?
      splitted = _.split(optional_split_at_this_character)
      first_part  = ::Colours::HtmlColoursMethods.send(first_colour_to_use, splitted.first)
      middle_part = ::Colours::HtmlColoursMethods.send(colour_to_use_for_the_token, optional_split_at_this_character)
      second_part = ::Colours::HtmlColoursMethods.send(second_colour_to_use, splitted.last)
      _ = first_part+
          middle_part+
          second_part+
          ::Colours.rev
    end
    e _
  end

  # ========================================================================= #
  # === Colours.scomments
  #  
  # Invocation example:
  #
  #   Colours.scomments('Hello world!')
  #
  # ========================================================================= #
  def self.scomments(
      i            = '',
      make_newline = false
    )
    replacement_colour = @colour_table[__method__.to_sym].to_sym
    return ::Colours::HtmlColoursMethods.send(replacement_colour, i)
  end; self.instance_eval { alias comment scomments } # === Colours.comment

  # ========================================================================= #
  # === Colours.scomments?
  #
  # This method will usually return :grey.
  # ========================================================================= #
  def self.scomments?
    colour_table?[:scomments]
  end

  # ========================================================================= #
  # === scomments
  # ========================================================================= #
  def scomments(i = '')
    ::Colours.scomments(i) # Delegate towards Colours.scomments? here.
  end; alias scomment scomments # === scomment

  # ========================================================================= #
  # === ecomment
  #
  # Just a wrapper over the Colours.ecomment() functionality.
  # ========================================================================= #
  def ecomment(
      i,
      optional_split_at_this_character = '#'
    )
    ::Colours.ecomment(i, optional_split_at_this_character)
  end

  # ========================================================================= #
  # === Colours.swarn
  #
  # Invocation example:
  #
  #   Colours.swarn('Hello world!')
  #
  # ========================================================================= #
  def self.swarn(
      i            = '',
      make_newline = false
    )
    replacement_colour = @colour_table[__method__.to_sym].to_sym
    return ::Colours::HtmlColoursMethods.send(replacement_colour, i)
  end

  # ========================================================================= #
  # === swarn
  # ========================================================================= #
  def swarn(
      i            = '',
      make_newline = false
    )
    ::Colours.swarn(i, make_newline)
  end

  # ========================================================================= #
  # === Colours.esymlink
  # ========================================================================= #
  def self.esymlink(
      i            = '',
      make_newline = false
    )
    e ::Colours.ssymlink(i, make_newline)
  end

  # ========================================================================= #
  # === esymlink
  # ========================================================================= #
  def esymlink(
      i            = '',
      make_newline = false
    )
    ::Colours.esymlink(i, make_newline)
  end

  # ========================================================================= #
  # === Colours.ewarn
  # ========================================================================= #
  def self.ewarn(
      i            = '',
      make_newline = false
    )
    e swarn(i, make_newline)
  end

  # ========================================================================= #
  # === ewarn
  # ========================================================================= #
  def ewarn(
      i            = '',
      make_newline = false
    )
    ::Colours.ewarn(i, make_newline)
  end

  # ========================================================================= #
  # === Colours.simportant
  #
  #This method is the one to denote "important" text components.
  #
  # Invocation examples:
  #
  #   Colours.simportant('Hello world!')
  #   Colours.simportant('yo there')
  #
  # ========================================================================= #
  def self.simportant(
      i            = '',
      make_newline = false
    )
    require 'colours/essentials/essentials.rb'
    replacement_colour = @colour_table[__method__.to_sym].to_sym
    _ = ::Colours::HtmlColoursMethods
    if _.respond_to? replacement_colour
      return _.send(replacement_colour, i)
    else
      ::Colours::Essentials.send(replacement_colour, i)
    end
  end; self.instance_eval { alias simp simportant } # === Colours.simp

  # ========================================================================= #
  # === simportant
  #
  # Delegate towards Colours.simportant() here.
  # ========================================================================= #
  def simportant(
      i            = '',
      make_newline = false
    )
    ::Colours.simportant(i, make_newline)
  end; alias simp simportant # === simp
       alias si   simportant # === si

  # ========================================================================= #
  # === Colours.eimp
  # ========================================================================= #
  def self.eimp(
      i            = '',
      make_newline = false
    )
    e simp(i, make_newline)
  end

  # ========================================================================= #
  # === eimportant
  # ========================================================================= #
  def eimportant(
      i            = '',
      make_newline = false
    )
    ::Colours.eimp(i, make_newline)
  end; alias eimp eimportant # === eimp
       alias cii  eimportant # === cii

  # ========================================================================= #
  # === ciif
  #
  # Alias to the above method, basically, aka eimportant().
  # ========================================================================= #
  def ciif(i)
    eimportant(i, false)
  end

  # ========================================================================= #
  # === Colours.efancy
  #
  # This simply will output the result from sfancy(), defined above.
  # ========================================================================= #
  def self.efancy(
      i            = '',
      make_newline = false
    )
    e sfancy(i, make_newline)
  end

  # ========================================================================= #
  # === efancy
  # ========================================================================= #
  def efancy(
      i            = '',
      make_newline = false
    )
    ::Colours.efancy(i, make_newline)
  end; alias f efancy # === f (f tag)

  # ========================================================================= #
  # === sfancy
  # ========================================================================= #
  def sfancy(i = '')
    ::Colours.sfancy(i)
  end

  # ========================================================================= #
  # === cif
  #
  # This is efancy, but we won't use newlines.
  # ========================================================================= #
  def cif(i)
    efancy(i, false)
  end; alias ff cif # === ff (alias to the above ^^^)

  # ========================================================================= #
  # === Colours.sdir
  # ========================================================================= #
  def self.sdir(
      i            = '',
      make_newline = false
    )
    replacement_colour = @colour_table[__method__.to_sym].to_sym
    return ::Colours::HtmlColoursMethods.send(replacement_colour, i)
  end

  # ========================================================================= #
  # === sdir
  # ========================================================================= #
  def sdir(
      i            = '',
      make_newline = false
    )
    ::Colours.sdir(i, make_newline)
  end

  # ========================================================================= #
  # === sfile
  # ========================================================================= #
  def sfile(
      i            = '',
      make_newline = false
    )
    ::Colours.sfile(i, make_newline)
  end

  # ========================================================================= #
  # === efile
  # ========================================================================= #
  def efile(
      i            = '',
      make_newline = false
    )
    e sfile(i, make_newline)
  end

  # ========================================================================= #
  # === Colours.sfile_or_sdir
  #
  # This method will directly delegate to either Colours.sfile() or
  # Colours.sdir(), depending on whether we have a file at hand or
  # whether we have a directory at hand.
  #
  # Usage examples:
  #
  #   Colours.sfile_or_sdir('/home/x/Temp/MyGems/')
  #   Colours.sfile_or_sdir('/home/x/Temp/opened_files.yml')
  #
  # ========================================================================= #
  def self.sfile_or_sdir(
      i            = '',
      make_newline = false
    )
    if File.exist?(i) and File.file?(i)
      return ::Colours.sfile(i, make_newline)
    elsif File.directory?(i)
      return ::Colours.sdir(i, make_newline)
    end
  end; self.instance_eval { alias sdir_or_sfile sfile_or_sdir } # === Colours.sdir_or_sfile

  # ========================================================================= #
  # === Update the colour-table next
  # ========================================================================= #
  if @use_html_colours and File.exist?(FILE_USE_THESE_VALUES_FOR_THE_COLOUR_METHODS)
    these_colours = YAML.load_file(FILE_USE_THESE_VALUES_FOR_THE_COLOUR_METHODS).
                    transform_keys(&:to_sym)
    update_colour_table_with(these_colours)
  end

  # ========================================================================= #
  # === Colours.col                                                 (col tag)
  #
  # Careful - this method may collide with other methods named col().
  #
  # Usage examples:
  #
  #   Colours.col '/Depot/j/geojgirjh'
  #   Colours.col '/home/x/songs/'
  #   Colours.col '/home/x/songs/Westbam_Sunshine.mp3'
  #
  # ========================================================================= #
  def self.col(
      i,
      optional_arg_not_in_use_right_now = ''
    )
    if File.exist? i
      ftype = File.ftype(i)
      # ===================================================================== #
      # e 'The ftype is: '+ftype # <- This could be used for debugging.
      # ===================================================================== #
      case ftype
      # ===================================================================== #
      # === file
      # ===================================================================== #
      when 'file'
        return sfile(i)
      # ===================================================================== #
      # === directory
      # ===================================================================== #
      when 'directory'
        return sdir(i)
      # ===================================================================== #
      # === link
      # ===================================================================== #
      when 'link'
        return ssymlink(i)
      else
        e "module Colours: We do not know the filetype `#{ftype}`"
      end
    else # Else return the input a bit changed.
      return sfile(i) # We modify it because that is better.
    end
  end

end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  e ::Colours.sdir('Hello world!')
  e ::Colours.sfile('Hello world!')+' And this should appear in another colour.'
  e ::Colours.simportant('Hello world!')
  e ::Colours.simp('Hello world!')
  e ::Colours.swarn('Hello world!')
  e ::Colours.ssymlink('Hello world!')
  e ::Colours.sfancy('Hello world!')
  e ::Colours.scomments('Hello world!')
  e ::Colours.col '/Depot/j/geojgirjh.md'
  e ::Colours.col '/home/x/songs/'
  e ::Colours.col '/home/x/songs/Westbam_Sunshine.mp3'
  ::Colours.ecomment('Hello world! # this is the associated comment')
  e
  pp Colours::HASH_NEW_COLOURS
  e
  e ::Colours.default_colour('Hello world!')
end # ruby sfile_sdir*.rb