#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal =  true
# =========================================================================== #
# This file primarily stores constants, including AnsiColours-related
# escape sequences. Additionally it will also store methods can refer
# or make use of these constants, such as Colours.teal(), and so forth.
#
# The different escape sequences can be combined into a single escape
# sequence by separating them via semicolons. For example, to create
# a section of text that is bold text with a yellow foreground and
# blue background, the escape sequence would be:
#
#   ^[[1;33;44m
#
# Keep in mind that in bash the ESC code can be either of the following:
#
#    \e
#    \033 (octal)
#    \x1B (hexadecimal)
#
# BOLD colour constants will have a "1" as part of their escape sequence.
# Note that the bold colours are sometimes also called the "bright colours".
# =========================================================================== #
# To see which colours are supported/supportable, for each terminal,
# have a look at this link:
#
#   https://misc.flogisoft.com/bash/tip_colors_and_formatting#terminals_compatibility
#
# This wikipedia link is also somewhat useful for ANSI escape sequences
# in general:
#
#   http://en.wikipedia.org/wiki/ANSI_escape_code#Colors
#
# =========================================================================== #
# This file is tested in the file
# colours/test/testing_the_basic_colours_of_the_colours_project.rb
# =========================================================================== #
# The following table shows the colour-codes for the terminal:
#
#   Black       0;30   Dark Gray     1;30
#   Red         0;31   Light Red     1;31
#   Green       0;32   Light Green   1;32
#   Yellow      0;33   Brown         1;33
#   Blue        0;34   Light Blue    1;34
#   Magenta     0;35   Light Magenta 1;35
#   Cyan        0;36   Light Cyan    1;36
#   Light Gray  0;37   White         1;37
#
# These are also stored in the Hash called HASH_SIMPLE_COLOURS.
#
# =========================================================================== #
# require 'colours/constants/constants.rb'
# Colours.file_html_colours
# Colours.file_html_colours_to_rgb?
# =========================================================================== #
module Colours

  require 'colours/project/project.rb'

  # ========================================================================= #
  # === Colours::N
  # ========================================================================= #
  N = "\n"

  # ========================================================================= #
  # === Colours::USE_THIS_COLOUR_FOR_THE_DEFAULT_COLOUR
  #
  # Here we specify the default colour to be used, in order to revert
  # to the prior colour in use. This can be modified by the user,
  # in order to conform to other colours if said user wishes to
  # do so.
  # ========================================================================= #
  USE_THIS_COLOUR_FOR_THE_DEFAULT_COLOUR = :grey

  # ========================================================================= #
  # === Colours::ESC
  #
  # Escape sequences start with the character ESC (ASCII decimal 27/hex
  # 0x1B/octal 033).
  #
  # The characters ESC and [ (left bracket) are called CSI, for "Control
  # Sequence Initiator".
  #
  # Keep in mind that "\e" can be represented via "\u001b" in Ruby, too.
  # ========================================================================= #
  ESC = "\e["

  # ========================================================================= #
  # === Colours::COLOUR_NC
  #
  # Note that this will actually reset all attributes.
  #
  # The "\e[0m" sequence removes all attributes, including formatting
  # and colours. It may be useful to add it to the end of each colour
  # text - and this is what the Colours project is essentially doing.
  #
  # An alternative may be to assign REVERT to WHITE, but this seems
  # to be incorrect according to the specification.
  # ========================================================================= #
  COLOUR_NC        = "#{ESC}0m" # No Colour. This is the reset code, aka \033[0m
    REVERT         = COLOUR_NC    # === REVERT
    REV            = COLOUR_NC    # === REV
    CLOSING_TAG    = COLOUR_NC    # === CLOSING_TAG (also represented as "#{ESC}0m")
    RESET          = COLOUR_NC    # === RESET
    RESET_TERMINAL = COLOUR_NC    # === RESET_TERMINAL
    CLEAR          = COLOUR_NC    # === CLEAR
    COLOUR_RESET   = COLOUR_NC    # === COLOUR_RESET
    DEFAULT_COLOUR = COLOUR_RESET # === DEFAULT_COLOUR

  # ========================================================================= #
  # === RESET_ALL_ATTRIBUTES
  #
  # This is essentially the same as COLOUR_NC / RESET.
  # ========================================================================= #
  RESET_ALL_ATTRIBUTES = "\x1b[0m"

  # ========================================================================= #
  # === Colours::HASH_ANSI_COLOURS
  # ========================================================================= #
  HASH_ANSI_COLOURS = {
    black:             '30',
    red:               '31',
    green:             '32',
    brown:             '33', # This is also known as yellow.
    yellow:            '33',
    blue:              '34',
    purple:            '35', # Also known as magenta.
    magenta:           '35',
    light_magenta:     '35',
    cyan:              '36',
    light_gray:        '37',
    grey:              '37',
    white:             '37',
    light_red:         '41',
    background_black:  '40',
    background_red:    '41',
    background_green:  '42',
    background_brown:  '43',
    background_yellow: '43',
    background_blue:   '44',
    light_blue:        '44', # Alias to the above ^^^.
    background_purple: '45',
    background_cyan:   '46',
    background_white:  '47'
  }

  # ========================================================================= #
  # === REGEX_REMOVE_ANSI_ESCAPE_CHARACTERS
  #
  # This regex can be used to get rid of (some) AnsiEscape characters.
  # ========================================================================= #
  REGEX_REMOVE_ANSI_ESCAPE_CHARACTERS =
    /\033\[.*?m/
    REMOVE_ANSI_ESCAPE_CHARACTERS = REGEX_REMOVE_ANSI_ESCAPE_CHARACTERS # === REMOVE_ANSI_ESCAPE_CHARACTERS

  # ========================================================================= #
  # === REGEX_FOR_HTML_COLOURS
  #
  # This regex supports the use case where we can match against HTML
  # colours. (The name is a bit of a misnomer, though.)
  #
  # For the following regex, see this entry at rubular:
  #
  #   https://rubular.com/r/o5r4DneMTGOsbX
  #
  # ========================================================================= #
  REGEX_FOR_HTML_COLOURS =
    /<([()a-zA-Z0-9²→\n]+?)>/

  # ========================================================================= #
  # === GENERIC_RUBY_HEADER
  # ========================================================================= #
  GENERIC_RUBY_HEADER = '#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #'
    RUBY_HEADER = GENERIC_RUBY_HEADER # === RUBY_HEADER

  # ========================================================================= #
  # === UNDERLINE_CODE
  #
  # To quickly test the underline-functionality, you can try:
  #
  #   echo -e "\e[4mone two three\e[23m" 
  #
  # ========================================================================= #
  UNDERLINE_CODE = '4'

  # ========================================================================= #
  # === ITALIC_CODE
  #
  # If you wish to test italic output quickly on the terminal, then you
  # can use something like this:
  #
  #   echo -e "\e[3mone two three\e[23m and this is not in italic"
  #
  # ========================================================================= #
  ITALIC_CODE = '3'

  # ========================================================================= #
  # === Colours::FILE_HTML_COLOURS
  # ========================================================================= #
  FILE_HTML_COLOURS =
    "#{PROJECT_BASE_DIRECTORY}yaml/html_colours.yml"

  # ========================================================================= #
  # === Colours::FILE_KDE_COLOUR_PALETTE
  # ========================================================================= #
  FILE_KDE_COLOUR_PALETTE =
    "#{PROJECT_BASE_DIRECTORY}yaml/kde_colour_palette.yml"

  # ========================================================================= #
  # === Colours::FILE_256_COLOURS
  # ========================================================================= #
  FILE_256_COLOURS =
    "#{PROJECT_BASE_DIRECTORY}yaml/256_colours.yml"

  # ========================================================================= #
  # === Colours::FILE_BASIC_COLOURS
  # ========================================================================= #
  FILE_BASIC_COLOURS =
    "#{PROJECT_BASE_DIRECTORY}yaml/basic_colours.yml"

  # ========================================================================= #
  # === Colours::FILE_USE_THESE_VALUES_FOR_THE_COLOUR_METHODS
  # ========================================================================= #
  FILE_USE_THESE_VALUES_FOR_THE_COLOUR_METHODS =
    "#{PROJECT_BASE_DIRECTORY}yaml/use_these_values_for_the_colour_methods.yml"

  # ========================================================================= #
  # === Colours::FILE_PREFER_THIS_COLOUR_SCHEMATA
  # ========================================================================= #
  FILE_PREFER_THIS_COLOUR_SCHEMATA =
    "#{PROJECT_BASE_DIRECTORY}yaml/prefer_this_colour_schemata.yml"

  # ========================================================================= #
  # === Colours.file_html_colours_to_rgb?
  #
  # Easier toplevel-getter method to determine where the file for
  # the html-colours-to-rgb conversion resides.
  # ========================================================================= #
  def self.file_html_colours_to_rgb?
    "#{PROJECT_BASE_DIRECTORY}yaml/html_colours.yml"
  end; self.instance_eval { alias file_html_colours  file_html_colours_to_rgb? } # === Colours.file_html_colours
       self.instance_eval { alias file_html_colours? file_html_colours_to_rgb? } # === Colours.file_html_colours?

  # ========================================================================= #
  # === Colours.available_file_constants
  # ========================================================================= #
  def self.available_file_constants
    [
      FILE_HTML_COLOURS,       # colours/yaml/html_colours.yml
      FILE_KDE_COLOUR_PALETTE, # colours/yaml/kde_colour_palette.yml
      FILE_256_COLOURS         # colours/yaml/256_colours.yml
    ]
  end

  # ========================================================================= #
  # === Colours::REGISTERED_COLOUR_METHODS
  #
  # These methods constitute the registered colour methods.
  #
  # These have a corresponding method, such as snormal() or sdir() or
  # sfile().
  #
  # snormal() stands for the "normal" colour, aka the "default
  # colour".
  # ========================================================================= #
  REGISTERED_COLOUR_METHODS = %i(
    snormal
    sargument
    scomments
    ssymlink
    sdir
    sfile
    sfancy
    simportant
    swarn
  )

  # ========================================================================= #
  # === Colours::REGISTERED_COLOUR_METHODS_ALIASES
  #
  # This Array should hold all aliases to the main colour-methods,
  # such as simportant() or sargument().
  # ========================================================================= #
  REGISTERED_COLOUR_METHODS_ALIASES = %i(
    argument
    ssym
    simp
    sarg
    normal
    comments
    symlink
    dir
    file
    fancy
    important
    warn
  )

  # ========================================================================= #
  # === Colours::HASH_SIMPLE_COLOURS
  #
  # This hash will only store one name for the corresponding colour code.
  # ========================================================================= #
  HASH_SIMPLE_COLOURS = {
    black:         '0;30',
    red:           '0;31',
    green:         '0;32',
    yellow:        '0;33',
    blue:          '0;34',
    magenta:       '0;35',
    cyan:          '0;36',
    light_gray:    '0;37',
    # ======================================================================= #
    # Next, the secondary-colours:
    # ======================================================================= #
    dark_gray:     '1;30',
    light_red:     '1;31',
    light_green:   '1;32',
    brown:         '1;33',
    light_blue:    '1;34',
    light_magenta: '1;35',
    light_cyan:    '1;36',
    white:         '1;37'
  }

  # ========================================================================= #
  # === Colours::MAIN_COLOURS
  #
  # The main colours come next.
  #
  # Eight main colours are currently defined, according to:
  #
  #   http://en.wikipedia.org/wiki/ANSI_escape_code#Colors
  #
  # The range is:
  #
  #   Intensity:     0    1    2      3     4      5      6    7
  #   Normal:     Black  Red Green Yellow  Blue  Magenta Cyan White
  #
  # Yellow, in my opinion, looks more like brown.
  # ========================================================================= #
  MAIN_COLOURS = %i(
    black red green yellow blue magenta teal white
  ) # ^^^ An Array of Symbols.

  # ========================================================================= #
  # === Colours.available_main_colours?
  #
  # Feedback which main colours are available.
  # ========================================================================= #
  def self.available_main_colours?
    MAIN_COLOURS
  end; self.instance_eval { alias available_basic_colours? available_main_colours? } # === Colours.available_basic_colours?

  # ========================================================================= #
  # === main_colours
  #
  # This will return an Array with the symbols such as
  # :black, :red etc.. 
  # ========================================================================= #
  def main_colours
    ::Colours.available_main_colours?
  end

  # ========================================================================= #
  # === Colours::SECONDARY_COLOURS
  #
  # The secondary Colours.
  #
  # Colours which are not part of the MAIN_COLOURS Array.
  #
  # The secondary colours are:
  #
  #   Bright Black
  #   Bright Red
  #   Bright Green
  #   Bright Yellow
  #   Bright Blue
  #   Bright Magenta
  #   Bright Cyan
  #   Bright White
  #
  # ========================================================================= #
  SECONDARY_COLOURS = [
    :bright_black,
    :bright_red,
    :bright_green,
    :bright_yellow,
    :bright_blue,
    :bright_magenta,
    :bright_cyan,
    :bright_white
  ]

  # ========================================================================= #
  # === ALL_COLOURS
  #
  # Simply combine the two Arrays, MAIN_COLOURS and SECONDARY_COLOURS,
  # into one aggregate Array.
  # ========================================================================= #
  ALL_COLOURS = MAIN_COLOURS +
                SECONDARY_COLOURS

  # ========================================================================= #
  # === Colours.really_all_colours
  #
  # This method will return an Array that may look like this:
  #
  #   [:black, :red, :green, :yellow, :blue, :magenta, :teal,
  #    :white, :bright_black, :bright_red, :bright_green,
  #    :bright_yellow, :bright_blue, :bright_magenta,
  #    :bright_cyan, :bright_white]
  #
  # ========================================================================= #
  def self.really_all_colours
    ALL_COLOURS # This is the conjoint Array.
  end; self.instance_eval { alias all_available_colours really_all_colours } # === Colours.all_available_colours

  # ========================================================================= #
  # === Colours.is_a_valid_colour?
  #
  # This method allows you to check if a given input-string is a
  # valid colour.
  #
  # A "valid" colour is one that the colours project supports.
  #
  # For example, the string "lightblue" is a valid colour.
  #
  # Usage example:
  #
  #   Colours.is_a_valid_colour? 'lightblue'
  #
  # ========================================================================= #
  def self.is_a_valid_colour?(i)
    ALL_COLOURS.include? i.to_sym
  end

  # ========================================================================= #
  # === Colours::REDRAW_THE_PROMPT
  #
  # The following constant is necessary in order for the background colour
  # to apply to the whole line and in order for the foreground colour to
  # apply to the cursor.
  # ========================================================================= #
  REDRAW_THE_PROMPT = "\e[K"

  # ========================================================================= #
  # === Colours::HOME_DIRECTORY_OF_USER_X
  #
  # This constant is only useful on my home system.
  # ========================================================================= #
  HOME_DIRECTORY_OF_USER_X = '/home/x/'

  # ========================================================================= #
  # === Colours::DEFAULT_TEXT
  # ========================================================================= #
  DEFAULT_TEXT = 'Hello World!'

  # ========================================================================= #
  # === Colours::IS_A_TEST
  # ========================================================================= #
  IS_A_TEST = ' This is just a simple colour test.' 

  # ========================================================================= #
  # === Colours::TEST_FILE
  # ========================================================================= #
  TEST_FILE =
    "#{HOME_DIRECTORY_OF_USER_X}programming/ruby/src/colours/test/testing_konsole_submodule.rb"

  # ========================================================================= #
  # === LEFT
  #
  # Leading on the left side, for RGB value setting. This is the beginning
  # of a special instruction. Note that "\e[38" should be equal to "\x1b",
  # but IRB reports that this is not the case so.
  #
  # Examples on the www show usage such as this:
  #
  #   printf "\x1b[${bg};2;${red};${green};${blue}m\n"
  #   printf "\x1b[38;2;255;100;0mTRUECOLOR\x1b[0m\n"
  # 
  # ========================================================================= #
  LEFT = "\e[38;2;".dup

  # ========================================================================= #
  # === Colours.left?
  # ========================================================================= #
  def self.left?
    LEFT
  end

  # ========================================================================= #
  # === left?
  # ========================================================================= #
  def left?
    Colours.left?
  end

  # ========================================================================= #
  # === COLOUR_BLACK
  # ========================================================================= #
  COLOUR_BLACK        = "#{ESC}0;30m"
    COLOUR_GRAY       = COLOUR_BLACK # === COLOUR_GRAY
    BLACK             = COLOUR_BLACK # === BLACK

  # ========================================================================= #
  # === COLOUR_LIGHT_CYAN
  # ========================================================================= #
  COLOUR_LIGHT_CYAN   = "#{ESC}1;36m"
    BOLD_CYAN         = COLOUR_LIGHT_CYAN # === BOLD_CYAN
    COLOUR_BOLD_TEAL  = COLOUR_LIGHT_CYAN # === COLOUR_BOLD_TEAL
    BRIGHT_CYAN       = COLOUR_LIGHT_CYAN # === BRIGHT_CYAN
    BRIGHT_TEAL       = COLOUR_LIGHT_CYAN # === BRIGHT_TEAL
    CYANB             = COLOUR_LIGHT_CYAN # === CYANB
    BOLD_TEAL         = COLOUR_LIGHT_CYAN # === BOLD_TEAL

  # ========================================================================= #
  # === COLOUR_CYAN
  #
  # This is also called TEAL, sort of.
  # ========================================================================= #
  COLOUR_CYAN         = "#{ESC}0;36m"
    TEAL              = COLOUR_CYAN # === TEAL
    CYAN              = COLOUR_CYAN # === CYAN
    COLOUR_TEAL       = COLOUR_CYAN # === COLOUR_TEAL

  # ========================================================================= #
  # === COLOUR_BROWN
  # ========================================================================= #
  COLOUR_BROWN         = "#{ESC}1;33m"
    BOLD_YELLOW        = COLOUR_BROWN # === BOLD_YELLOW
    GOLD               = COLOUR_BROWN # === GOLD
    BRIGHT_BROWN       = COLOUR_BROWN # === BRIGHT_BROWN
    BRIGHT_YELLOW      = COLOUR_BROWN # === BRIGHT_YELLOW
    BOLDYELLOW         = COLOUR_BROWN # === BOLDYELLOW
    BYEL               = COLOUR_BROWN # === BYEL
    BYELLOW            = COLOUR_BROWN # === BYELLOW
    COLOUR_BOLD_YELLOW = COLOUR_BROWN # === COLOUR_BOLD_YELLOW

  # ========================================================================= #
  # === COLOUR_LIGHT_GRAY
  #
  # This is also known as white.
  # ========================================================================= #
  COLOUR_LIGHT_GRAY   = "#{ESC}0;37m"
    WHITE             = COLOUR_LIGHT_GRAY # === WHITE
    COLOUR_LIGHT_GREY = COLOUR_LIGHT_GRAY # === COLOUR_LIGHT_GREY

  # ========================================================================= #
  # === COLOUR_WHITE
  #
  # This is bold white, actually.
  # ========================================================================= #
  COLOUR_WHITE        = "#{ESC}1;37m"
    BOLD_WHITE        = COLOUR_WHITE # === BOLD_WHITE 
    BRIGHT_WHITE      = COLOUR_WHITE # === BRIGHT_WHITE
    BWHITE            = COLOUR_WHITE # === BWHITE
    COLOUR_BOLD_WHITE = COLOUR_WHITE # === COLOUR_BOLD_WHITE

  # ========================================================================= #
  # === COLOUR_DARK_GRAY
  #
  # This is also called "black bold".
  # ========================================================================= #
  COLOUR_DARK_GRAY   = "#{ESC}1;30m"
    GREY             = COLOUR_DARK_GRAY # === GREY
    GRAY             = COLOUR_DARK_GRAY # === GRAY
    CGREY            = COLOUR_DARK_GRAY # === CGREY
    BRIGHT_BLACK     = COLOUR_DARK_GRAY # === BRIGHT_BLACK
    BOLD_BLACK       = COLOUR_DARK_GRAY # === BOLD_BLACK

  # ========================================================================= #
  # == COLOUR_RED                                                   (red tag)
  # ========================================================================= #
  COLOUR_RED          = "#{ESC}0;31m" # This is thin, darkred.
    RED  = COLOUR_RED # === RED
    CRED = COLOUR_RED # === CRED

  # ========================================================================= #
  # === COLOUR_GREEN
  # ========================================================================= #
  COLOUR_GREEN        = "#{ESC}0;32m" # This is lightgreen.
    GREEN     = COLOUR_GREEN # === GREEN
    CGREEN    = COLOUR_GREEN # === CGREEN
    DARKGREEN = COLOUR_GREEN # === DARKGREEN

  # ========================================================================= #
  # === COLOUR_BLUE
  # ========================================================================= #
  COLOUR_BLUE         = "#{ESC}0;34m"
    BLUE = COLOUR_BLUE # === BLUE

  # ========================================================================= #
  # === COLOUR_LIGHT_GREEN
  # ========================================================================= #
  COLOUR_LIGHT_GREEN  = "#{ESC}1;32m"
    BOLD_GREEN        = COLOUR_LIGHT_GREEN # === BOLD_GREEN
    COLOUR_BOLD_GREEN = COLOUR_LIGHT_GREEN # === COLOUR_BOLD_GREEN
    BRIGHT_GREEN      = COLOUR_LIGHT_GREEN # === BRIGHT_GREEN
    LIGHTGREEN        = COLOUR_LIGHT_GREEN # === LIGHTGREEN
    BGREEN            = COLOUR_LIGHT_GREEN # === BGREEN

  # ========================================================================= #
  # === COLOUR_MAGENTA
  # ========================================================================= #
  COLOUR_MAGENTA    = "#{ESC}0;35m"
    PURPLE        = COLOUR_MAGENTA # === PURPLE
    COLOUR_PURPLE = COLOUR_MAGENTA # === COLOUR_PURPLE
    MAGENTA       = COLOUR_MAGENTA # === MAGENTA

  # ========================================================================= #
  # === COLOUR_LIGHT_BLUE
  #
  # This is also known as "bold blue".
  # ========================================================================= #
  COLOUR_LIGHT_BLUE   = "#{ESC}1;34m"
    BOLD_BLUE         = COLOUR_LIGHT_BLUE # === BOLD_BLUE
    COLOUR_BOLD_BLUE  = COLOUR_LIGHT_BLUE # === COLOUR_BOLD_BLUE 
    BRIGHT_BLUE       = COLOUR_LIGHT_BLUE # === BRIGHT_BLUE
    LIGHT_BLUE        = COLOUR_LIGHT_BLUE # === LIGHT_BLUE
    LIGHTBLUE         = COLOUR_LIGHT_BLUE # === LIGHTBLUE
    BBLUE             = COLOUR_LIGHT_BLUE # === BBLUE
    BOLDBLUE          = COLOUR_LIGHT_BLUE # === BOLDBLUE

  # ========================================================================= #
  # === Colours::BLACK_ON_WHITE_BACKGROUND
  #
  # Usage example:
  #
  #   puts Colours::BLACK_ON_WHITE_BACKGROUND+'Hello world!'+Colours::RESET
  #
  # ========================================================================= #
  BLACK_ON_WHITE_BACKGROUND = "#{ESC}0;30;47m"
    BLACK_ON_WHITE_BG = BLACK_ON_WHITE_BACKGROUND # === BLACK_ON_WHITE_BG

  # ========================================================================= #
  # === COLOUR_YELLOW
  # ========================================================================= #
  COLOUR_YELLOW       = "#{ESC}0;33m"
    BROWN             = COLOUR_YELLOW # === BROWN
    YEL               = COLOUR_YELLOW # === YEL
    YELLOW            = COLOUR_YELLOW # === YELLOW
    BOLD_BROWN        = COLOUR_YELLOW # === BOLD_BROWN

  # ========================================================================= #
  # === COLOUR_LIGHT_MAGENTA
  #
  # This is not the "normal" magenta, but the light variant; also
  # called light purple.
  # ========================================================================= #
  COLOUR_LIGHT_MAGENTA = "#{ESC}1;35m"
    LIGHT_PURPLE        = COLOUR_LIGHT_MAGENTA # === LIGHT_PURPLE
    COLOUR_LIGHT_PURPLE = COLOUR_LIGHT_MAGENTA # === COLOUR_LIGHT_PURPLE
    PINK                = COLOUR_LIGHT_MAGENTA # === PINK
    COLOUR_PINK         = COLOUR_LIGHT_MAGENTA # === COLOUR_PINK
    BRIGHT_PINK         = COLOUR_LIGHT_MAGENTA # === BRIGHT_PINK
    BRIGHT_MAGENTA      = COLOUR_LIGHT_MAGENTA # === BRIGHT_MAGENTA

  # ========================================================================= #
  # === Colours::COLOUR_LIGHT_RED
  # ========================================================================= #
  COLOUR_LIGHT_RED    = "#{ESC}1;31m"
    LIGHT_RED         = COLOUR_LIGHT_RED # === LIGHT_RED
    BOLD_RED          = COLOUR_LIGHT_RED # === BOLD_RED
    COLOUR_BOLD_RED   = COLOUR_LIGHT_RED # === COLOUR_BOLD_RED 
    BRIGHT_RED        = COLOUR_LIGHT_RED # === BRIGHT_RED
    BOLDRED           = COLOUR_LIGHT_RED # === BOLDRED
    REDB              = COLOUR_LIGHT_RED # === REDB
    BRED              = COLOUR_LIGHT_RED # === BRED

  # ========================================================================= #
  # === ARRAY_REGISTERED_ANSI_COLOURS
  #
  # This array exists mostly for testing-purposes. 
  # ========================================================================= #
  ARRAY_REGISTERED_ANSI_COLOURS = [

    COLOUR_BLACK,
    COLOUR_RED,
    COLOUR_GREEN,
    COLOUR_YELLOW,
    COLOUR_BLUE,
    COLOUR_MAGENTA,
    COLOUR_TEAL,
    COLOUR_LIGHT_GREY,

    COLOUR_BOLD_RED,
    COLOUR_BOLD_GREEN,
    COLOUR_BOLD_YELLOW,
    COLOUR_BOLD_BLUE,
    COLOUR_PINK,
    COLOUR_BOLD_TEAL,
    COLOUR_BOLD_WHITE,

    COLOUR_RESET

  ]

  # ========================================================================= #
  # === Colours.file_256_colours
  #
  # This method will return the path to the .yml file containing the
  # data for the 256-colours variant.
  # ========================================================================= #
  def self.file_256_colours
    FILE_256_COLOURS
  end
  
end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  include Colours
  e
  # pp Colours::COLOUR_NC == "\033[0m" # yup, this is true, as confirmed in November 2023.
  pp Colours.available_file_constants
  e
  e 'The registered colour methods are:'
  e
  pp Colours::REGISTERED_COLOUR_METHODS
  e
  e 'My home directory: '+Colours::HOME_DIRECTORY_OF_USER_X
  e
  e Colours::COLOUR_LIGHT_RED+'Testing light red here.'
  e
  e CYAN+'Hi there.'+RESET
  e BWHITE+'Hi there.'+BBLUE+' Yo there.'+RESET
  e WHITE+'Hi there.'+BBLUE+' Yo there.'+RESET
  e BLACK_ON_WHITE_BACKGROUND+'Hello world!'+RESET
  e BRIGHT_RED+'Hi there.'+BBLUE+' Yo there.'+RESET
  e BRIGHT_BLUE+'Hi there.'+BWHITE+' Yo there.'+RESET
  e
  e 'And testing individual colours for the characters next:'
  e
  e BLACK+' A '+
    RED+' B '+
    GREEN+' C '+
    BROWN+' D '+
    BLUE+' E '+
    MAGENTA+' F '+
    TEAL+' G '+
    WHITE+' H '+
    RESET
  e
  e 'Same as above but with the bright colours instead:'
  e
  e BRIGHT_BLACK+' A '+BRIGHT_RED+' B '+BRIGHT_GREEN+' C '+
    BRIGHT_BROWN+' D '+BRIGHT_BLUE+' E '+BRIGHT_MAGENTA+
    ' F '+BRIGHT_TEAL+' G '+BRIGHT_WHITE+' H '+RESET
  e
end