#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'colours/kde_colour_palette/kde_colour_palette.rb'
# =========================================================================== #
module Colours

  require 'yaml'
  require 'colours/constants/constants.rb'
  require 'colours/rgb/rgb.rb'

  # ========================================================================= #
  # === Colours::KDE_COLOUR_PALETTE
  #
  # The format for the following Hash is this:
  #
  #   - The key is the name of the colour at hand
  #   - There are two values per name. The first is the hex value,
  #     and the second is the R,G,B value stored as a Hash.
  #
  # ========================================================================= #
  if File.exist? FILE_KDE_COLOUR_PALETTE
    KDE_COLOUR_PALETTE = YAML.load_file(FILE_KDE_COLOUR_PALETTE)
  else
    KDE_COLOUR_PALETTE = {}
  end

  # ========================================================================= #
  # Next, we will batch-define the KDE colour palette as class methods.
  #
  # To test this, you can try:
  #
  #   Colours.kde_colour_palette_icon_yellow 'Hello world!'
  #
  # ========================================================================= #
  KDE_COLOUR_PALETTE.keys.each {|this_key|
    short_symbol_name_for_the_key = this_key.downcase.tr(' ','_').to_sym
    use_this_as_method_name = ('kde_colour_palette_'+short_symbol_name_for_the_key.to_s).to_sym
    unless Colours.respond_to? use_this_as_method_name
      self.class.class_eval {
        define_method(use_this_as_method_name) {|output_this_text = ''| # Allow input.
          ::Colours.return_this_via_kde_colour_palette(output_this_text, short_symbol_name_for_the_key)
        }
        # =================================================================== #
        # The next enables colours such as:
        #
        #   Colours.pimpinella 'Hello cats!'
        #
        # =================================================================== #
        define_method(short_symbol_name_for_the_key.to_sym) {|output_this_text = ''| # Allow input.
          ::Colours.return_this_via_kde_colour_palette(output_this_text, short_symbol_name_for_the_key.to_sym)
        }
        # =================================================================== #
        # The next enables colours such as:
        #   Colours.epimpinella 'Hello cats!'
        # =================================================================== #
        define_method(('e'+short_symbol_name_for_the_key.to_s).to_sym) {|output_this_text = ''| # Allow input.
          ::Colours.write_this_via_kde_colour_palette(output_this_text, short_symbol_name_for_the_key.to_sym)
        }
      }
    end
  }

  # ========================================================================= #
  # === Colours.return_this_via_kde_colour_palette
  #
  # To invoke this method, try:
  #
  #   Colours.return_this_via_kde_colour_palette 'Hello world!'
  #   Colours.return_this_via_kde_colour_palette 'Hello world!', :random
  #   Colours.return_this_via_kde_colour_palette 'Hello world!', :plasma_blue
  #
  # ========================================================================= #
  def self.return_this_via_kde_colour_palette(
      this_text   = '',
      this_colour = 'Pimpinella'
    )
    use_this_rgb_value_array = []
    case this_colour
    when :random # Pick a random colour in this case.
      this_colour = KDE_COLOUR_PALETTE.keys.sample
    else
      if this_colour.is_a? Symbol
        if this_colour.to_s.include? '_'
          # ================================================================= #
          # In this case, try to convert it.
          # ================================================================= #
          this_colour = this_colour.to_s.split('_').map(&:capitalize).join(' ')
        else
          this_colour = this_colour.to_s.capitalize
        end
      end
    end
    if KDE_COLOUR_PALETTE.has_key? this_colour
      use_this_rgb_value_array = KDE_COLOUR_PALETTE[this_colour][1]
    end
    # ======================================================================= #
    # Finally return the text, in the corresponding colour at hand.
    # ======================================================================= #
    return ::Colours.rgb_value_as_escape_code_string(use_this_rgb_value_array)+
           this_text+
           rev
  end

  # ========================================================================= #
  # === Colours.write_this_via_kde_colour_palette
  #
  # To invoke this method, try any of the following:
  #
  #   Colours.write_this_via_kde_colour_palette 'Hello world!'
  #   Colours.write_this_via_kde_colour_palette 'Hello world!', :random
  #   Colours.write_this_via_kde_colour_palette 'Hello world!', :plasma_blue
  #
  # ========================================================================= #
  def self.write_this_via_kde_colour_palette(
      this_text      = '',
      this_colour    = 'Pimpinella',
      append_newline = true
    )
    result = ::Colours.return_this_via_kde_colour_palette(this_text, this_colour)
    if append_newline
      result << "#{N}"
    end
    print result
  end

end

if __FILE__ == $PROGRAM_NAME
  pp Colours::KDE_COLOUR_PALETTE.keys
end