#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'colours/map_symbol_to_corresponding_colour/map_symbol_to_corresponding_colour.rb'
# =========================================================================== #
module Colours

  require 'colours/constants/constants.rb'
  require 'colours/toplevel_methods/rev.rb'

  # ========================================================================= #
  # === Colours.map_symbol_to_corresponding_colour                 (menu tag)
  #
  # This method will accept a symbol as input, such as :bold_blue, and
  # return the corresponding CONSTANT (or rather, the value this constant
  # refers to). This allows us to "map" the respective symbol to the
  # corresponding constant.
  #
  # For example, the Symbol :bold_blue will correspond to the BOLD_BLUE
  # constant. The Symbol :bold_green will correspond to the BOLD_GREEN
  # constant (or COLOUR_BOLD_GREEN; there are some "aliases" on the
  # constant-level as well). This constant thus represents the symbol
  # at hand.
  #
  # Take note that this method here has several aliases:
  #
  #   Colours.translate()
  #   Colours.beautify()
  #   Colours.convert()
  #   Colours.parse()
  #   Colours.map_colour_to_constant()
  #
  # This is partially due to historic reasons; and partially simply
  #  because some of them are shorter and hence easier to type.
  #
  # Usage example for this method:
  #
  #   Colours.beautiful(:bold_blue)
  #   Colours.beautiful(:brown)
  #
  # ========================================================================= #
  def self.map_symbol_to_corresponding_colour(i)
    if i.is_a? String # Handle Strings as well.
      # ===================================================================== #
      # === random
      # ===================================================================== #
      if i == 'random' # Fetch a random entry in this case.
        i = MAIN_COLOURS.sample
      end
      i = i.tr(' ','_').to_sym
    end
    case i # case tag
    # ======================================================================= #
    # === :bright_cyan
    # ======================================================================= #
    when :bright_cyan,
         :cyanb,
         :cyan
      return BOLD_CYAN
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default,
         :si,
         :default_colour,
         :defaultcolour,
         :normal,
         nil
      return DEFAULT_COLOUR
    # ======================================================================= #
    # === :light_grey
    # ======================================================================= #
    when :light_grey,
         :grey
      return COLOUR_LIGHT_GREY
    # ======================================================================= #
    # === :bold_green
    # ======================================================================= #
    when :bold_green,
         :boldgreen,
         :bright_green,
         :greenb,
         :lightgreen
      return BOLD_GREEN # => "\e[1;32m"
    # ======================================================================= #
    # === :bold_brown
    # ======================================================================= #
    when :bold_brown,
         :bbrown,
         :gold,
         :saddlebrown
      return BOLD_BROWN
    # ======================================================================= #
    # === :white
    # ======================================================================= #
    when :white,
         :whitebonblack
      return WHITE
    # ======================================================================= #
    # === :red
    # ======================================================================= #
    when :red
      return RED
    # ======================================================================= #
    # === :pink
    # ======================================================================= #
    when :pink,
         :bright_magenta
      return BRIGHT_MAGENTA
    # ======================================================================= #
    # === :brown
    # ======================================================================= #
    when :brown
      return BROWN
    # ======================================================================= #
    # === :yellow
    # ======================================================================= #
    when :yellow,
         :yel
      return YELLOW
    # ======================================================================= #
    # === :green
    # ======================================================================= #
    when :green,
         :greengreen
      return GREEN
    # ======================================================================= #
    # === :blue
    # ======================================================================= #
    when :blue
      return BLUE
    # ======================================================================= #
    # === :magenta
    # ======================================================================= #
    when :magenta
      return MAGENTA
    # ======================================================================= #
    # === :teal
    # ======================================================================= #
    when :teal
      return TEAL
    # ======================================================================= #
    # === :revert
    # ======================================================================= #
    when :revert,
         :reset,
         :clear
      return REVERT
    # ======================================================================= #
    # === :bold_red
    # ======================================================================= #
    when :bold_red,
         :boldred,
         :bright_red,
         :lightred,
         :redb
      return BOLD_RED
    # ======================================================================= #
    # === :lightblue
    # ======================================================================= #
    when :lightblue
      return LIGHTBLUE
    # ======================================================================= #
    # === :bold_yellow
    # ======================================================================= #
    when :bold_yellow,
         :boldyellow,
         :yelb,
         :bright_yellow,
         :byellow
      return BOLD_YELLOW
    # ======================================================================= #
    # === :bold_blue
    # ======================================================================= #
    when :bold_blue,
         :boldblue,
         :bright_blue,
         :blueb
      return BOLD_BLUE
    # ======================================================================= #
    # === :bold_white
    # ======================================================================= #
    when :bold_white,
         :boldwhite,
         :bright_white
      return BOLD_WHITE
    # ======================================================================= #
    # === :black
    # ======================================================================= #
    when :black,
         :bold_black,
         :boldblack,
         :bright_black
      return COLOUR_BLACK
    else
      puts "Unknown symbol given as input: #{i}"
      pp i
    end
    i # else return the input-argument again.
  end; self.instance_eval { alias beautiful              map_symbol_to_corresponding_colour } # === Colours.beautiful
       self.instance_eval { alias beautify               map_symbol_to_corresponding_colour } # === Colours.beautify
       self.instance_eval { alias translate              map_symbol_to_corresponding_colour } # === Colours.translate
       self.instance_eval { alias convert                map_symbol_to_corresponding_colour } # === Colours.convert
       self.instance_eval { alias parse                  map_symbol_to_corresponding_colour } # === Colours.parse
       self.instance_eval { alias map_colour_to_constant map_symbol_to_corresponding_colour } # === Colours.map_colour_to_constant
       self.instance_eval { alias map_this_symbol        map_symbol_to_corresponding_colour } # === Colours.map_this_symbol
       self.instance_eval { alias map_symbol_to_colour   map_symbol_to_corresponding_colour } # === Colours.map_symbol_to_colour
       self.instance_eval { alias corresponding_colour?  map_symbol_to_corresponding_colour } # === Colours.corresponding_colour?  

  # ========================================================================= #
  # === map_symbol_to_corresponding_colour
  #
  # This maps a colour (string) to the respective constant.
  # ========================================================================= #
  def map_symbol_to_corresponding_colour(i)
    ::Colours.map_symbol_to_corresponding_colour(i)
  end; alias map_colour_to_constant map_symbol_to_corresponding_colour # === map_colour_to_constant

end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  e Colours.beautiful(:bold_blue)+'OK! Bold blue colour.'+Colours.rev
  e Colours.beautiful(:bold_brown)+'OK! Bold brown colour.'+Colours.rev
end