#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
module Colours

module RainbowColours

  # ========================================================================= #
  # === Colours::RainbowColours.rainbow
  #
  # Do the colour calculations next, via the rainbow() method.
  #
  # The frequency is typically a value such as 0.1.
  #
  # We will calculate the RGB values here (R, G, B).
  #
  # Invocation example:
  #
  #   Colours::RainbowColours.rainbow(:default, 3)  # => "#A5D604"
  #   Colours::RainbowColours.rainbow(:default, 12) # => "#F66C1C"
  #
  # ========================================================================= #
  def self.rainbow(
      frequency = 0.1, i
    )
    case frequency
    when nil,
         :default
      frequency = 0.1
    end
    red   = Math.sin(frequency * i + 0) * 127 + 128
    green = Math.sin(frequency * i + 2 * Math::PI/3) * 127 + 128
    blue  = Math.sin(frequency * i + 4 * Math::PI/3) * 127 + 128
    '#%02X%02X%02X' % [ red, green, blue ]
  end

end

# =========================================================================== #
# === Colours.cat
#
# The first argument may be ARGF. It ought to respond to .each anyway.
#
# Note that .cat() will delegate towards print_line().
# =========================================================================== #
def self.cat(
  file_descriptor, hash_options = {}
  )
  print "\e[?25l" if hash_options[:animate] # Print this if we animate (enabling psychedelics).
  # ========================================================================= #
  # Iterate over the file_descriptor passed.
  # ========================================================================= #
  file_descriptor.each { |line|
    hash_options[:os] += 1 # Increase the output-count.
    # ======================================================================= #
    # Next, colourize the line. This depends on the method called
    # print_rainbow_line().
    # ======================================================================= #
    RainbowColours.print_rainbow_line(
      line, hash_options
    ) # Delegate towards print_line.
  }
ensure
  print "\e[?25h" if hash_options[:animate]
end

end