#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'gtk_paradise/colours/parse.rb'
# =========================================================================== #
module Gtk

module Colours # === Gtk::Colours

  # ========================================================================= #
  # === Gtk::Colours.parse                                        (parse tag)
  #
  # Gdk::Color.parse() will handle colours given to this method.
  #
  # This is mostly relevant only for ruby-gtk2; ruby-gtk3 and ruby-gtk4
  # have other ways to handle/parse colours.
  # ========================================================================= #
  def self.parse(i)
    begin
      Gdk::Color.parse(i)
    rescue NameError => error
      puts 'An error did occur at line 20, in the file '\
           'gtk_paradise/colours/parse.rb'
      pp error
    end
  end

  # ========================================================================= #
  # === Gtk::Colours.validate_gtk_colour
  #
  # Use this method to try and validate a colour. Valid input can be a
  # string, or a Gdk::Color object.
  # ========================================================================= #
  def self.validate_gtk_colour(i)
    if i.is_a? String and i =~ /\d/
      i.prepend '#' unless i.start_with? '#'
    end
    return parse(i)
  end

end

# =========================================================================== #
# === Gtk.colour_parse
# =========================================================================== #
def self.colour_parse(i)
  begin
    if ::Gtk.use_gtk4?
      e 'Gdk::Color.parse() is no longer available in ruby-gtk4.'
    else # this is for gtk3.
      Gdk::Color.parse(i.to_s)
    end
  rescue Exception => error
    pp 'Error on line '+__LINE__.to_s+' in the file parse.rb'
    pp error
    puts 'The caller() stacktrace was as follows:'
    e
    pp caller()
    e
  end
end

# =========================================================================== #
# === Gtk.convert_gdk_colour_to_gdk_rgba_colour
#
# This method can be used to specifically convert Gdk::Color
# into Gdk::RGBA.
# =========================================================================== #
def self.convert_gdk_colour_to_gdk_rgba_colour(
    colour
  )
  if colour.is_a? Gdk::Color
    # ======================================================================= #
    # Convert it into RGB values next.
    # ======================================================================= #
    array = [
      (colour.red   * 100.0 / 65535) / 100,
      (colour.green * 100.0 / 65535) / 100,
      (colour.blue  * 100.0 / 65535) / 100,
      1.0
    ]
    colour = Gdk::RGBA.new(*array)
  end
  return colour
end; self.instance_eval { alias convert_gdk_color_to_gdk_rgba_color convert_gdk_colour_to_gdk_rgba_colour } # === Gtk.convert_gdk_color_to_gdk_rgba_color

end