#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Button
#
# This file contains modification to a Gtk::Button widget.
# =========================================================================== #
# require 'gtk_paradise/core_classes/button.rb'
# =========================================================================== #
module Gtk

class Button # === Gtk::Button

  if respond_to?(:tooltip=)
    # ======================================================================= #
    # === original_set_tooltip
    # ======================================================================= #
    alias original_set_tooltip tooltip= # set_tooltip

    # ======================================================================= #
    # === set_tooltip
    # ======================================================================= #
    alias set_tooltip fancy_tooltip=
  end

  # ========================================================================= #
  # The next line is necessary to determine which functionality works
  # for the different gtk-variants.
  # ========================================================================= #
  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'

  # ========================================================================= #
  # === set_halign
  # ========================================================================= #
  unless Gtk.do_we_use_gtk4?
    alias set_halign set_xalign if respond_to?(:set_xalign) # === set_halign
  end

  # ========================================================================= #
  # === no_border
  # ========================================================================= #
  def no_border
    set_relief(Gtk::ReliefStyle::NONE) # At the least for gtk3.
  end

  # ========================================================================= #
  # === align_to_the_left_side
  #
  # This will focus the text on the button towards the left side.
  # ========================================================================= #
  def align_to_the_left_side
    self.xalign = 0.0
  end; alias align_to_the_left        align_to_the_left_side # === align_to_the_left
       alias text_is_on_the_left_side align_to_the_left_side # === text_is_on_the_left_side
       alias text_is_left             align_to_the_left_side # === text_is_left
       alias left_align               align_to_the_left_side # === left_align
       alias align_towards_the_left   align_to_the_left_side # === align_towards_the_left

  # ========================================================================= #
  # === hover_colour=
  #
  # "hover" in this context refers to the mouse cursor being positioned
  # over the button at hand.
  #
  # The argument to this method should be the colour that you want to
  # use, ideally as a symbol.
  # ========================================================================= #
  def hover_colour=(
      i        = :paleturquoise,
      prelight = :prelight # Assume ruby-gtk3 by default.
    )
    # prelight = ::Gtk::STATE_PRELIGHT if ::Gtk.use_gtk2?
    modify_bg(
      prelight,
      Gdk::Color.parse(i.to_s)
    )
  end; alias prelight_colour hover_colour= # === prelight_colour
       alias hover_colour    hover_colour= # === hover_colour

  # ========================================================================= #
  # === on_click_event
  # ========================================================================= #
  def on_click_event(&block)
    signal_connect(:clicked, &block)
  end; alias on_clicked  on_click_event # === on_click_event
       alias on_click    on_click_event # === on_click
       alias on_click_do on_click_event # === on_click_do
       # alias clicked    on_click_event # <-- This one unfortunately does not work.

  # ========================================================================= #
  # === modify_bg
  #
  # This method is mostly used to allow :symbols as second arguments,
  # as alias for a colour.
  # ========================================================================= #
  def modify_bg(
      a = ::Gtk::STATE_NORMAL,
      b = :slateblue
    )
    if b.is_a? Symbol
      b = Gdk::Color.parse(b.to_s)
    end
    # if ::Gtk.use_gtk2?
    #   super(a, b)
    if b.is_a? Gdk::Color
      # ===================================================================== #
      # Convert it into RGB values next.
      # ===================================================================== #
      array = [
        (b.red   * 100.0 / 65535) / 100,
        (b.green * 100.0 / 65535) / 100,
        (b.blue  * 100.0 / 65535) / 100,
        1.0
      ]
      b = Gdk::RGBA.new(*array)
    end
    override_background_color(a, b)
  end

  # ========================================================================= #
  # === set_text
  #
  # In the past, the method here had a "if respond_to? :label=", but since
  # as of November 2020 it just exists as a method anway. Otherwise we
  # may depend on gtk being "properly" initialized".
  # ========================================================================= #
  def set_text(
      i,
      use_mnemonics: false
    )
    set_label(i)
    if use_mnemonics
      set_use_underline(true)
    end
  end; alias text= set_text # === text=

  # ========================================================================= #
  # === do_click_this_button
  # ========================================================================= #
  def do_click_this_button
    signal_emit(:clicked)
  end

  # ========================================================================= #
  # === text?
  # ========================================================================= #
  alias text? label

  # ========================================================================= #
  # === apply_mnemonic
  # ========================================================================= #
  def apply_mnemonic
    set_label(text?)
    set_use_underline(true)
  end; alias apply_markup apply_mnemonic # === apply_markup

  # ========================================================================= #
  # === do_use_mnemonic
  # ========================================================================= #
  def do_use_mnemonic
    self.use_underline = true
  end

  # ========================================================================= #
  # === do_markify
  # ========================================================================= #
  def do_markify
    self.children.first.do_markify
  end

  # ========================================================================= #
  # === do_use_underline
  # ========================================================================= #
  def do_use_underline
    self.use_underline = true
  end; alias enable_markup do_use_underline # === enable_markup

  # ========================================================================= #
  # === relief_none
  # ========================================================================= #
  def relief_none(
      i = :use_the_appropriate_variant_for_the_current_gtk_version
    )
    case i
    # ======================================================================= #
    # === :use_the_appropriate_variant_for_the_current_gtk_version
    # ======================================================================= #
    when :use_the_appropriate_variant_for_the_current_gtk_version
      i = :none
    end
    set_relief(i)
  end; alias no_relief relief_none # === no_relief
    
  # ========================================================================= #
  # === do_focus
  # ========================================================================= #
  def do_focus
    set_focus(true)
  end

  # ========================================================================= #
  # === toggle_grey
  # ========================================================================= #
  def toggle_grey
    if sensitive?
      set_sensitive(false)
    else
      set_sensitive(true)
    end
  end

  # ========================================================================= #
  # === enable_the_button
  # ========================================================================= #
  def enable_the_button
    set_sensitive(true)
  end; alias is_active enable_the_button # === is_active

  # ========================================================================= #
  # === disable_the_button
  # ========================================================================= #
  def disable_the_button
    set_sensitive(false)
  end; alias is_inactive disable_the_button # === is_inactive

  # ========================================================================= #
  # === background_colours
  # ========================================================================= #
  def background_colours(
      normal   = nil,
      prelight = nil
    )
    if normal
      modify_bg(::Gtk::StateType::NORMAL,   normal)
    end
    if prelight
      modify_bg(::Gtk::StateType::PRELIGHT, prelight)
    end
  end

  # ========================================================================= #
  # === label_widget
  #
  # Convenience method to obtain the label-widget.
  # ========================================================================= #
  def label_widget
    label_widget = self.children.first
    return label_widget
  end

  # ========================================================================= #
  # === relief_half
  # ========================================================================= #
  def relief_half
    self.relief = :half
  end

  # ========================================================================= #
  # === no_relief
  #
  # This method will remove the relief that can be found associated
  # with buttons normally.
  # ========================================================================= #
  def no_relief
    self.relief = :none
  end

  # ========================================================================= #
  # === make_bold
  #
  # This method will make the text of the button at hand bold.
  # ========================================================================= #
  def make_bold(
      retain_mnemonic = :default
    )
    child = self.children.first
    unless child.respond_to?(:make_bold)
      require 'gtk_paradise/core_classes/label.rb' # ← Necessary for .make_bold() below.
    end
    child.make_bold(
      :use_the_default_text,
      retain_mnemonic
    ) # ← ^^^ This will delegate to Gtk::Label, and calls .make_bold() there.
  end

  # ========================================================================= #
  # === attach_this_image
  # ========================================================================= #
  # def attach_this_image(i)
  #   @button.set_image(@image)
  # end
  # alias attach_this_image set_image # === attach_this_image

end

# =========================================================================== #
# === Gtk.create_button
#
# Simpler method to create a new Gtk button.
#
# The first argument to this method should be the text for the button.
# This is the text that will appear on that button.
#
# It may be a Hash too, though, in particular in ruby-gtk3, so this
# method has to handle these cases as well.
# =========================================================================== #
def self.create_button(
    optional_use_this_text = '',
    use_underline          = false, # This is whether we will use a mnemonic or not.
    &block
  )
  # ========================================================================= #
  # === Handle Hashes first
  # ========================================================================= #
  if optional_use_this_text.is_a? Hash
    # ======================================================================= #
    # Here we handle input such as:
    #   { label: "TEST" }
    # ======================================================================= #
    if optional_use_this_text.has_key? :label
      optional_use_this_text = optional_use_this_text.delete(:label)
    end
  end
  if optional_use_this_text.respond_to?(:start_with?) and
     optional_use_this_text.start_with? '_'
    use_underline = true
  end
  hash = {}
  # ======================================================================= #
  # We could also add stock_id:
  # ======================================================================= #
  if optional_use_this_text.is_a? Hash
    hash.merge!(optional_use_this_text)
  end
  unless hash.has_key? :label
    hash.merge!(label: optional_use_this_text) if optional_use_this_text.is_a?(String)
  end
  unless hash.has_key? :use_underline
    hash.merge!(use_underline: use_underline) if optional_use_this_text.is_a?(String)
  end
  # ========================================================================= #
  # Instantiate a new gtk-button next, by passing the Hash:
  # ========================================================================= #
  button = ::Gtk::Button.new(hash)
  # ========================================================================= #
  # === Handle blocks next
  # ========================================================================= #
  if block_given?
    yielded = yield
    # ======================================================================= #
    # === Symbols passed as blocks are treated in a special way
    # ======================================================================= #
    if yielded.is_a? Symbol
      case yielded
      # ===================================================================== #
      # === :on_click_exit
      #
      # This entry point is used to exit, on clicking that button.
      # ===================================================================== #
      when :on_click_exit,
           :exit_on_click_event
        button.on_clicked {
          ::Gtk.main_quit
        }
      end
    elsif yielded.is_a? Hash
      # ===================================================================== #
      # === :tooltip
      #
      # This here can be triggered by passing a block such as:
      #
      #   {{ tooltip: "Hello world!" }}
      #
      # ===================================================================== #
      if yielded.has_key? :tooltip
        button.fancy_tooltips = yielded.delete(:tooltip)
      end
      # ===================================================================== #
      # === :on_click_do
      #
      # Note: this was added in August 2020, but it probably does not
      # work and could be deleted.
      #
      # Handle entries such as:
      #
      #   on_click_do: update_the_liststore
      #
      # ===================================================================== #
      if yielded.has_key? :on_click_do
        button.signal_connect(:clicked) {|widget, event|
          send(yielded.delete(:on_click_do))
        }
      end
    end
  end
  return button
end; self.instance_eval { alias create_gtk_button create_button } # === Gtk.create_gtk_button
     self.instance_eval { alias return_button     create_button } # === Gtk.return_button
     self.instance_eval { alias button            create_button } # === Gtk.button

end