#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::HBox
#
# This variant should only be pulled in if we are using ruby-gtk2.
# =========================================================================== #
# require 'gtk_paradise/core_classes/hbox.rb'
# =========================================================================== #
require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'

return unless Gtk.do_we_use_gtk2?

module Gtk

class HBox

  if respond_to?(:pack_start)
    alias pack pack_start # === pack
  end

  # ========================================================================= #
  # === add_these
  #
  # This method can be used to batch-add other widget into the VBox,
  # such as if you'd normally call add() yourself.
  # ========================================================================= #
  def add_these(*widgets)
    widgets.each {|this_widget|
      add(this_widget)
    }
  end

end

# =========================================================================== #
# === Gtk.padded_hbox
# =========================================================================== #
def self.padded_hbox(
    n_padding_to_use = 8
  )
  _ = hbox
  _.set_border_width(n_padding_to_use)
  return _
end

# =========================================================================== #
# === Gtk.create_paned_hbox
#
# This method will create a HBox with several widgets inside of it.
#
# The widgets go into the only parameter for this method, aptly
# called *widgets.
#
# You can pass in a block to specify the (optional) padding.
#
# An example for this would be:
#
#   Gtk.create_paned_hbox(widget1, widget2) { 10 }
#
# Usage examples:
#
#   Gtk.create_paned_hbox(Gtk::Button.new,Gtk::Button.new, Gtk::Button.new)
#   Gtk.create_paned_hbox(widget1, widget2)
#   Gtk.paned(a, b, c)
#
# =========================================================================== #
def self.create_paned_hbox(*widgets)
  widgets.flatten!
  default_padding = block_given? ? yield : 2
  # if do_we_use_gtk2?
    hbox = Gtk::HBox.new
  # else
  #   hbox = Gtk::Box.new(:horizontal)
  # end
  widgets.size.times { |entry| # Next batch-add it onto the hbox here.
    hbox.pack_start(widgets[entry],true,true, default_padding)
  }
  return hbox.show
end; self.instance_eval { alias chbox       create_paned_hbox } # === Gtk.chbox
     self.instance_eval { alias create_hbox create_paned_hbox } # === Gtk.create_hbox
     self.instance_eval { alias hbox        create_paned_hbox } # === Gtk.hbox
     self.instance_eval { alias add_hbox    create_paned_hbox } # === Gtk.add_hbox
     self.instance_eval { alias hpaned      create_paned_hbox } # === Gtk.hpaned
     self.instance_eval { alias paned_hbox  create_paned_hbox } # === Gtk.paned_hbox

# =========================================================================== #
# === Gtk.create_paned_hbox_last_big
#
# First widget takes up most space here. hbox means "horizontal box".
#
# Usage example via the gtk-paradise gem:
#
#   Gtk.create_paned_hbox_last_bigg(gtk_button, gtk_button, gtk_button)
#
# =========================================================================== #
def self.create_paned_hbox_last_big(*widgets)
  how_many_slots = widgets.size
  hbox = ::Gtk::HBox.new
  how_many_slots.times { |t|
    case t
    when 0
      hbox.pack_start(widgets[t], false, true,1)
    else
      hbox.pack_start(widgets[t], true, true, 1)
    end
  }
  return hbox.show
end

# =========================================================================== #
# === Gtk.create_paned_hbox_first_big
#
# First widget takes up most space here.
#
#   Gtk.create_paned_hbox_first_big(Gtk::Button.new, Gtk::Button.new, Gtk::Button.new)
#
# =========================================================================== #
def self.create_paned_hbox_first_big(*widgets)
  widgets.flatten! # Don't forget to flatten here.
  hbox = ::Gtk::HBox.new
  widgets.size.times { |t|
    if t == 0
      hbox.pack_start(widgets[t], true, true, 1)
    else
      hbox.pack_start(widgets[t], false, true,1)
    end
  }
  return hbox.show
end

# =========================================================================== #
# === Gtk.create_hbox
# =========================================================================== #
def self.create_hbox(i = ARGV)
  # The arguments are currently ignored.
  ::Gtk::HBox.new
end; self.instance_eval { alias hbox        create_hbox } # === Gtk.hbox
     self.instance_eval { alias return_hbox create_hbox } # === Gtk.return_hbox

# =========================================================================== #
# === Add an alias to HBox if we use ruby-gtk2
# =========================================================================== #
Hbox = HBox

end # end the gtk2-check.