#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::TextBuffer
# =========================================================================== #
# require 'gtk_paradise/core_classes/text_buffer.rb'
# =========================================================================== #
module Gtk

class TextBuffer # === Gtk::TextBuffer

  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'

  alias cursor_position? cursor_position # === cursor_position?

  # ========================================================================= #
  # Add an alias from set_buffer() to set_text().
  # ========================================================================= #
  alias set_buffer set_text # === set_buffer

  # ========================================================================= #
  # === smiley
  # ========================================================================= #
  def smiley
    append_this_image(:smiley)
  end

  # ========================================================================= #
  # === append_this_image
  #
  # This method can be used to append an image to a text-buffer.
  # ========================================================================= #
  def append_this_image(
      path_to_the_image
    )
    case path_to_the_image
    # ======================================================================= #
    # === :smiley
    # ======================================================================= #
    when :smiley
      path_to_the_image = "#{::Gtk.project_base_directory?}images/icons/icon_smile.gif"
    end
    pixbuf = ::GdkPixbuf::Pixbuf.new(file: path_to_the_image)
    insert(end_iter, pixbuf)
  end

  # ========================================================================= #
  # === read_only
  #
  # This method will disallow making any further changes to the given
  # gtk-text-buffer at hand.
  # ========================================================================= #
  def read_only
    begin
      set_property('editable', false)
    rescue GLib::NoPropertyError => error
      pp 'Error in line 32:'
      pp error
    end
  end; alias it_can_not_be_edited read_only # === it_can_not_be_edited

  # ========================================================================= #
  # === place_cursor_at_this_position
  #
  # This method can be used to place the cursor at the specific position
  # in the text-buffer. You have to provide a number, such as 5
  # or 10. For the proper number, use other methods to calculate it,
  # based on the string-content of the text-buffer at hand.
  # ========================================================================= #
  def place_cursor_at_this_position(
      position = 10
    )
    if position.is_a? Numeric
      iter = get_iter_at(offset: position)
    else
      iter = position
    end
    place_cursor(iter)
    return iter # We must return the iter, so that other code parts can re-use it.
  end; alias place_cursor_at place_cursor_at_this_position # === place_cursor_at

  # ========================================================================= #
  # === append_after_the_cursor
  # ========================================================================= #
  def append_after_the_cursor(i)
    insert_at_cursor(i.to_s)
  end
 
  # ========================================================================= #
  # === on_cursor_position_changed
  # ========================================================================= #
  def on_cursor_position_changed(&block)
    signal_connect('notify::cursor-position', &block)
  end

  # ========================================================================= #
  # === delete_the_selected_text
  # ========================================================================= #
  def delete_the_selected_text(
      a = false, b = false
    )
    delete_selection(a, b)
  end

  # ========================================================================= #
  # === return_bold_tag
  # ========================================================================= #
  def return_bold_tag
    create_tag(nil, { weight: :heavy })
  end; alias bold_tag return_bold_tag # === bold_tag

  # ========================================================================= #
  # === onto_the_end
  # ========================================================================= #
  def onto_the_end(
      text = '', hash
    )
    insert(end_iter, text, hash)
  end; alias append_onto_the_end onto_the_end # === append_onto_the_end

  # ========================================================================= #
  # === file=
  #
  # This method will read in the passed file, into the text-buffer.
  # ========================================================================= #
  def file=(i)
    set_text(
      File.read(i)
    )
  end

  # ========================================================================= #
  # === selected_text?
  # ========================================================================= #
  def selected_text?
    start_position, end_position = selection_bounds # Returns: [start, end, selected]
    text = get_text(start_position, end_position, false)
    return text
  end

  # ========================================================================= #
  # === sync_placeholder_text
  # ========================================================================= #
  def sync_placeholder_text
    if @placeholder_text and !@placeholder_text.empty?
      set_text(
        @placeholder_text.last.to_s
      )
    end 
  end

  # ========================================================================= #
  # === set_placeholder_text
  # ========================================================================= #
  def set_placeholder_text(i)
    @placeholder_text ||= []
    @placeholder_text << i
  end

  # ========================================================================= #
  # === placeholder_text?
  # ========================================================================= #
  def placeholder_text?
    @placeholder_text
  end

  # ========================================================================= #
  # === text?
  #
  # This method is guaranteed to return a String.
  # ========================================================================= #
  def text?
    text.to_s
  end

  # ========================================================================= #
  # === return_underline_tag
  # ========================================================================= #
  def return_underline_tag
    create_tag(nil, { underline: :single })
  end; alias underline_tag return_underline_tag # === return_underline_tag

  # ========================================================================= #
  # === clear
  #
  # Empty the text-buffer via this method.
  # ========================================================================= #
  def clear
    set_text('')
  end

  # ========================================================================= #
  # === append_markup
  # ========================================================================= #
  def append_markup(
      this_text
    )
    insert_markup(end_iter, this_text, -1)
  end

  # ========================================================================= #
  # === return_current_iter
  # ========================================================================= #
  def return_current_iter
    get_iter_at(offset: -1)
  end

  # ========================================================================= #
  # === append_italic
  #
  # This will append the given input, via a <i> tag.
  # ========================================================================= #
  def append_italic(
      this_text = ''
    )
    insert_markup(
      end_iter,
      "<i>#{this_text}</i>",
      -1
    )
  end

  # ========================================================================= #
  # === append
  #
  # Use this method to append text onto a Gtk::TextBuffer instance. This
  # method will only append simple text; if you need markup-text then
  # you have to use another method.
  # ========================================================================= #
  def append(i)
    _ = text # Grab the old text.
    set_text(_+i)
  end

  # ========================================================================= #
  # === do_markify
  # ========================================================================= #
  def do_markify
    old_text = text
    clear
    insert_markup(end_iter, old_text, -1)
  end

end

# =========================================================================== #
# === Gtk.text_buffer
#
# This method will instantiate a new Gtk::TextBuffer.
# =========================================================================== #
def self.text_buffer
  Gtk::TextBuffer.new
end; self.instance_eval { alias create_text_buffer text_buffer } # === Gtk.create_text_buffer

end