#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Drawing::Tool
# =========================================================================== #
# require 'gtk_paradise/drawing/tool.rb'
# =========================================================================== #
module Gtk

module Drawing

class Tool # === Drawing::Tool

  attr_accessor :width

  alias e puts

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(i = W)
    @window = i
    @drawable = @window.window
    @gc = Gdk::GC.new(@drawable)
    reset
  end

  # ========================================================================= #
  # === reset                                                   (reset tag)
  # ========================================================================= #
  def reset
    # ===================================================================== #
    # === @font_description
    # ===================================================================== #
    @font_description = Pango::FontDescription.new
    # ===================================================================== #
    # === @colormap
    # ===================================================================== #
    @colormap = Gdk::Colormap.system
    # ===================================================================== #
    # === @width
    # ===================================================================== #
    @width  = 0
    # ===================================================================== #
    # === @height
    # ===================================================================== #    
    @height = 0
    # ===================================================================== #
    # === @color
    # ===================================================================== #
    @color = Gdk::Color.new(0, 0, 0)
  end

  # ========================================================================= #
  # === window?
  # ========================================================================= #
  def window?
    @window
  end; alias window window? # === window

  # ========================================================================= #
  # === rectangle
  # ========================================================================= #
  def rectangle(
      fill, x, y, width, height, color = nil
    )
    set_color(color)
    # ======================================================================= #
    # Note that @drawable is a Gtk::Window.
    # ======================================================================= #
    @drawable.draw_rectangle(@gc, fill, x, y, width, height)
  end

  # ========================================================================= #
  # === clear
  #
  # This kind of resets the whole area.
  # ========================================================================= #
  def clear(color = nil)
    colour ||= Gdk::Color.new(0, 0, 0)
    set_colour(colour)
    rectangle(true, 0, 0, @width, @height)
  end

  # ========================================================================= #
  # === colour
  # ========================================================================= #
  def colour(r, g, b)
    @color = Gdk::Color.new(r, g, b)
    @colormap.alloc_color(@color, false, true)
    @color
  end; alias color colour # === color

  # ========================================================================= #
  # === arc
  # ========================================================================= #
  def arc(
      fill,
      x,
      y,
      width,
      height,
      d1,
      d2,
      color = nil
    )
    set_color(color)
    @drawable.draw_arc(@gc, fill, x, y, width, height, d1, d2)
  end

  # ========================================================================= #
  # === circle
  # ========================================================================= #
  def circle(
      fill,
      x,
      y,
      r,
      color = nil
    )
    arc(fill, x - r, y - r, 2 * r, 2 * r, 0, 64 * 360, color)
  end

  # ========================================================================= #
  # === point
  # ========================================================================= #
  def point(
      x,
      y,
      color = nil
    )
    set_color(color)
    @drawable.draw_point(@gc, x, y)
  end
 
  # ========================================================================= #
  # === line
  # ========================================================================= #
  def line(
      x1,
      y1,
      x2,
      y2,
      color = nil
    )
    set_color(color)
    @drawable.draw_lines(@gc, [[x1, y1], [x2, y2]])
  end

  # ========================================================================= #
  # === lines
  # ========================================================================= #
  def lines(
      array,
      color = nil
    )
    set_color(color)
    @drawable.draw_lines(@gc, array)
  end

  # ========================================================================= #
  # === polygon
  # ========================================================================= #
  def polygon(fill, array, color = nil)
    set_color(color)
    @drawable.draw_polygon(@gc, fill, array)
  end

  # ========================================================================= #
  # === text
  # ========================================================================= #
  def text(
      str,
      x,
      y,
      size,
      color = nil
    )
    set_color(color)
    # ===================================================================== #
    # Set the new size here:
    # ===================================================================== #
    @font_description.set_size(size)
    layout = Pango::Layout.new(W.pango_context)
    layout.font_description = @font_description
    layout.text = str
    @drawable.draw_layout(@gc, x, y, layout)
  end

  # ========================================================================= #
  # === set_colour
  # ========================================================================= #
  def set_colour(color)
    @color = color if color
    @gc.set_foreground(@color)
  end; private :set_colour
    alias set_color set_colour # === set_color
    
  # ========================================================================= #
  # === load_pic
  # ========================================================================= #
  def load_pic(use_this_filename)
    GdkPixbuf::Pixbuf.new(file: use_this_filename)
  end

  # ========================================================================= #
  # === save_pic
  # ========================================================================= #
  def save_pic(
      img, filename, type = 'png'
    )
    img.save(filename, type)
  end

  # ========================================================================= #
  # === show_pic
  # ========================================================================= #
  def show_pic(img, x, y)
    @drawable.draw_pixbuf(
      @gc, img, 0, 0, x, y, img.width, img.height, Gdk::RGB::DITHER_NONE, 0, 0
    )
  end

  # ========================================================================= #
  # === get_pic
  # ========================================================================= #
  def get_pic(x, y, width, height)
    GdkPixbuf::Pixbuf.from_drawable(nil, @drawable, x, y, width, height)
  end
    
  # ========================================================================= #
  # === timer_stop
  # ========================================================================= #
  def timer_stop(id)
    Gtk.timeout_remove(id)
  end

  # ========================================================================= #
  # === star
  # ========================================================================= #
  def star(
      fill, x1, y1, x2, y2, color = nil
    )
    set_color(color)
    Star.new(fill, x1, y1, x2, y2, @color).draw
  end

  # ========================================================================= #
  # === get_window_size
  # ========================================================================= #
  def get_window_size
    W.size
  end

  # ========================================================================= #
  # === height?
  # ========================================================================= #
  def height?
    @height
  end; alias height height? # === height

  # ========================================================================= #
  # === set_height
  # ========================================================================= #
  def set_height(i)
    @height = i
  end; alias height= set_height # === height=

end; end; end