#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# The idea of this class is to showcase a Gtk::ComboBox.
# =========================================================================== #
require 'gtk2'

class ComboBoxExample < ::Gtk::VBox # === ComboBoxExample

  alias e puts

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize
    super()
    reset
    create_combo_box_entry
    @main_window = Gtk::Window.new('A Gtk::ComboBox sample')
    @main_window.signal_connect(:destroy) { Gtk.main_quit }
    @main_window.set_default_size(600, 600)
    @button_quit = Gtk::Button.new('_Quit')
    @button_quit.signal_connect(:clicked) { Gtk.main_quit }
    @button_clear = Gtk::Button.new('_Clear List')
    @button_clear.signal_connect(:clicked) { 
      e 'Clearing list'
      @combo_box_entry.model.clear
    }
    add_debug_button
    @v_box = Gtk::VBox.new
    @v_box.add(@combo_box_entry).
           pack_start(@button_quit,false, false).
           pack_start(@button_clear, false, false).
           pack_start(@button_debug, false, false)
    @big_vbox.add(@v_box)
    run
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    @big_vbox = Gtk::VBox.new
  end

  # ========================================================================= #
  # === big_vbox?
  # ========================================================================= #
  def big_vbox?
    @big_vbox
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    combo1 = ::Gtk::ComboBox.new
    %w( one two three four five ).each { |val|
      combo1.append_text(val)
    }
    combo1.active = 1 # Second entry is active, which is 'two'.
    combo1.signal_connect(:changed) {
      p "combo1: #{combo1.active}, #{combo1.active_iter[0]}"
    }
    big_vbox?.pack_start(combo1, true, true, 2)
    # =========================================================================== #
    # Text only, for our first combo-box. This is the simplest combo-box
    # variant.
    # =========================================================================== #

    # =========================================================================== #
    # Icon and texts in use for the following combo-box.
    # =========================================================================== #
    model = Gtk::ListStore.new(GdkPixbuf::Pixbuf, String)
    [[Gtk::Stock::QUIT,   'quit'],
     [Gtk::Stock::CANCEL, 'cancel'],
     [Gtk::Stock::OK,     'ok']].each { |stock, name|
      iter = model.append
      iter[0] = @main_window.render_icon(stock, Gtk::IconSize::MENU, "icon")
      iter[1] = name
    }

    combo2 = Gtk::ComboBox.new(model)
    combo2.signal_connect(:changed) {
      p "combo2 was changed: #{combo2.active}, #{combo2.active_iter[0]}"
    }

    # =========================================================================== #
    # column 1
    # =========================================================================== #
    renderer = Gtk::CellRendererPixbuf.new
    combo2.pack_start(renderer, false)
    combo2.set_attributes(renderer, pixbuf: 0)

    # =========================================================================== #
    # column 2
    # =========================================================================== #
    renderer = Gtk::CellRendererText.new
    combo2.pack_start(renderer, true)
    combo2.set_attributes(renderer, text: 1)
    combo2.active = 2
    combo2.signal_connect(:changed) {
      p "combo2: #{combo2.active}, #{combo2.active_iter[1]}"
    }
    @big_vbox.pack_start(combo2, true, true)
    @main_window.add(@big_vbox)
    @main_window.show_all
    Gtk.main
  end

  # ========================================================================= #
  # === add_debug_button
  # ========================================================================= #
  def add_debug_button
    @button_debug = Gtk::Button.new('_Debug')    
    @button_debug.signal_connect(:clicked) {
      e 'Debug: '+@combo_box_entry.active.to_s
    }
  end

  # ========================================================================= #
  # === create_combo_box_entry
  #
  # This will create a Gtk::ComboBoxEntry instance.
  # ========================================================================= #
  def create_combo_box_entry
    @combo_box_entry = ::Gtk::ComboBoxEntry.new
    %w(
      combo_box_entry1
      combo_box_entry2
      combo_box_entry3
      combo_box_entry4
    ).each { |val| 
      @combo_box_entry.append_text(val) 
    }
    @combo_box_entry.active = 0
    @combo_box_entry.signal_connect(:changed) {
      if @combo_box_entry.active_iter
        p "@combo_box_entry: #{@combo_box_entry.active}, #{@combo_box_entry.active_iter[0]}"
      end
    }
  end
  # combo_text.model.clear
end

if __FILE__ == $PROGRAM_NAME
  window = Gtk::Window.new
  window.add(ComboBoxExample.new)
  window.move(0, 0)
  window.set_size_request(800, 500)
  window.show_all
  window.signal_connect(:destroy) { Gtk.main_quit }
  Gtk.main
end # $RUBY_GTK/showcase_combo_box.rb