#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
require 'gtk2'

class Canvas < Gtk::DrawingArea

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize
    super()
    handle_the_signals
    reset
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    @buffer = nil
    @bgc = nil
  end

  # ========================================================================= #
  # === handle_the_signals
  # ========================================================================= #
  def handle_the_signals
    signal_connect(:expose_event)    { |w, e| expose_event(w,e)    }
    signal_connect(:configure_event) { |w, e| configure_event(w,e) }
  end

  # ========================================================================= #
  # === expose_event
  # ========================================================================= #
  def expose_event(w,e)
    unless @buffer.nil?
      rec = e.area
      w.window.draw_drawable(@bgc, @buffer, rec.x, rec.y,
        rec.x, rec.y, rec.width, rec.height
      )
    end
    false
  end

  # ========================================================================= #
  # === clear
  # ========================================================================= #
  def clear(b = @buffer)
    return if b.nil?
    g = b.size
    @bgc = self.style.bg_gc(self.state) if @bgc.nil?
    if (g[0] > 0 && g[1] > 0)
      b.draw_rectangle(@bgc, true, 0,0, g[0], g[1])
    end
  end

  # ========================================================================= #
  # === configure_event
  # ========================================================================= #
  def configure_event(w,e)
    g = w.window.geometry
    if (g[2] > 0 && g[3] > 0)
      b = Gdk::Pixmap::new(w.window, g[2], g[3], -1)
      clear(b)
      if not @buffer.nil?
        g = @buffer.size
        b.draw_drawable(@bgc, @buffer, 0, 0, 0, 0, g[0], g[1])
      end
      @buffer = b
    end
    true
  end
end

class A < Canvas

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize
    super
    reset
    signal_connect(:button_press_event) { |w,e| pressed(w,e) }
    set_events(Gdk::Event::BUTTON_PRESS_MASK)
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    @last = nil
  end

  # ========================================================================= #
  # === pressed
  # ========================================================================= #
  def pressed(widget, ev)
    if not @last.nil?
      @buffer.draw_line(widget.style.fg_gc(widget.state),
        @last.x, @last.y, ev.x, ev.y
      )

      x1,x2 = if (@last.x < ev.x)
        then [@last.x, ev.x]
        else [ev.x,    @last.x]
        end
      y1,y2 = if (@last.y < ev.y)
      then [@last.y, ev.y]
      else [ev.y,    @last.y]
      end
      widget.queue_draw_area(x1, y1, x2 - x1 + 1, y2 - y1 + 1)
    end
    @last = nil
    @last = ev
    true
  end

end

if __FILE__ == $PROGRAM_NAME
  window = Gtk::Window.new('drawing test')
  window.signal_connect(:destroy) { Gtk.main_quit }
  canvas = A.new
  window.add(canvas)
  window.move(0, 0)
  window.show_all
  Gtk.main
end