#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# This widget shows how to intercept mouse-drag and mouse-drop events.
# =========================================================================== #
# require '061_mouse_movement.rb'
# =========================================================================== #
require 'gtk2'

module Gtk

class ValueDragger < ::Gtk::EventBox
  
  attr_accessor :val
  attr_accessor :divX
  attr_accessor :divY
  attr_accessor :widget

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      widget = Label.new('value = 0'),
      divX = 1,
      divY = 1,
      &block
    )
    super()
    self.signal_connect('motion-notify-event')  { | w, e |
      on_motion_notify_event( w, e )
    }
    self.signal_connect(:button_press_event)    { on_button_press_event }
    self.signal_connect('button-release-event') { on_button_release_event }
    self.add(@widget = widget)
    @startVal = 0
    @should_change_val = false
    @startX = nil
    @startY = nil
    @block  = block
    @divX   = divX
    @divY   = divY
    @divX   = 1 if divX.to_f == 0
    @divY   = 1 if divY.to_f == 0
  end

  # ========================================================================= #
  # === on_motion_notify_event
  # ========================================================================= #
  def on_motion_notify_event(w, e)
    if @should_change_val
      _win, x, y, state = e.window.pointer
      p e.class
      @val = ( @startVal + ( ( x - @startX ) / @divX ) +
             ( ( @startY - y ) / @divY ) ).round
      @block.call
      return true
    end
  end

  # ========================================================================= #
  # === on_button_press_event
  # ========================================================================= #
  def on_button_press_event
    win, @startX, @startY, state = self.window.pointer
    @should_change_val = true
  end

  # ========================================================================= #
  # === on_button_release_event
  # ========================================================================= #
  def on_button_release_event
    @should_change_val = false
    @startX = nil
    @startY = nil
    @startVal = @val
  end
  
  # ========================================================================= #
  # === get_value
  # ========================================================================= #
  def get_value
    return @val
  end

  # ========================================================================= #
  # === get_widget
  # ========================================================================= #
  def get_widget
    return @widget
  end

  # ========================================================================= #
  # === set_value
  # ========================================================================= #
  def set_value(i)
    @startVal = i
  end

  # ========================================================================= #
  # === set_divX
  # ========================================================================= #
  def set_divX(i)
    i = 1 if i.to_f == 0
    @divX = i
  end; alias set_x set_divX # === set_x

  # ========================================================================= #
  # === set_divY
  # ========================================================================= #
  def set_divY(i)
    i = 1 if i.to_f == 0
    @divY = i
  end; alias set_y set_divY

  # ========================================================================= #
  # === set_div
  #
  # Here we set both x and y in one method.
  # ========================================================================= #
  def set_div( x, y )
    set_x(x)
    set_y(y)
  end

  # ========================================================================= #
  # === new_block
  # ========================================================================= #
  def new_block(&block)
    @block = block
  end
end

end

class MainApp < Gtk::Window

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize
    super
    self.signal_connect(:destroy) { Gtk.main_quit }
    f = Gtk::Frame.new('')
    label = Gtk::Label.new( 'Wert = 0' )
    dragger = Gtk::ValueDragger.new( f.add(label) ) {
      label.set_label("Wert = #{ dragger.get_value }")
    }
    vbox = Gtk::VBox.new
    label = Gtk::Label.new(
      'Click the mouse button in order to modify the value.'
    )
    vbox.pack_start(label, false, false, 10)
    entry_X = Gtk::Entry.new
    entry_Y = Gtk::Entry.new 
    vbox.pack_start(entry_X, false)
    vbox.pack_start(entry_Y, false)
    button = Gtk::Button.new(Gtk::Stock::OK) 
    vbox.pack_start(button, false)
    vbox.pack_start(dragger)
    eX.text = 'X-Teiler hier...'
    eY.text = 'Y-Teiler hier...'
    self.add( vbox ).show_all
    btn.signal_connect(:clicked) {
      dragger.set_div( eX.text.to_f, eY.text.to_f )
    }
  end

end

if __FILE__ == $PROGRAM_NAME
  MainApp.new
  Gtk.main
end