#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Paned Widget Example
#
# The Gtk::HPaned and Gtk::VPaned Widgets divide their content area into 
# two panes with a divider in between that the user can adjust. A 
# separate child is placed into each pane.
#
# There are a number of options that can be set for each pane.
#
# This example code here contains both a horizontal (HPaned) and a vertical
# (VPaned) widget together, and allows you to adjust the options for each
# side of each of these widgets.
# =========================================================================== #
# === Gtk::PanedWidgetExample
# $GTK_PARADISE/examples/gtk3/010_paned_widget_example.rb
# =========================================================================== #
require 'gtk3'

module Gtk

class PanedWidgetExample < ::Gtk::Box

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize
    super(:vertical)
    reset
    self.border_width = 0
    vbox = ::Gtk::Box.new(:vertical, 0)
    add(vbox)
    # ======================================================================= #
    # Add a VPaned widget next:
    # ======================================================================= #
    vpaned = ::Gtk::Paned.new(:vertical)
    vbox.pack_start(vpaned, expand: true, fill: true, padding: 0)
    vpaned.border_width = 5
    # ======================================================================= #
    # Add a HPaned widget next:
    # ======================================================================= #
    hpaned = ::Gtk::Paned.new(:horizontal)
    vpaned.add1(hpaned)
    vpaned.set_name('padded_pane')
    frame = ::Gtk::Frame.new
    frame.shadow_type = :in
    frame.set_size_request(60, 60)
    hpaned.add1(frame)
    # ======================================================================= #
    # Add a simple button next, into a Gtk::Frame.
    # ======================================================================= #
    button = ::Gtk::Button.new(
      label: '_Hi there, I want to be dragged!', use_underline: true
    )
    frame.add(button)
    frame = ::Gtk::Frame.new
    frame.shadow_type = :in
    frame.set_size_request(80, 60)
    hpaned.add2(frame)
    frame = ::Gtk::Frame.new
    frame.shadow_type = :in
    frame.set_size_request(60, 80)
    vpaned.add2(frame)
    # ======================================================================= #
    # Now create toggle buttons to control sizing. Three labels will be
    # supplied as well.
    # ======================================================================= #
    vbox.pack_start(create_pane_options(hpaned,
      'Horizontal', 'Left', 'Right'),
      expand: false, fill: false, padding: 0
    )
    vbox.pack_start(create_pane_options(vpaned,
      'Vertical', 'Top', 'Bottom'),
      expand: false, fill: false, padding: 0
    )
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    setup_CSS_rules
  end

  # ========================================================================= #
  # === setup_CSS_rules
  # ========================================================================= #
  def setup_CSS_rules
    css_provider = Gtk::CssProvider.new
    _ = '#padded_pane paned separator {
      background-size:   20px;
      min-width:         20px;
      background-color:  royalblue;
    }'
    css_provider.load(data: _)
    ::Gtk::StyleContext.add_provider_for_screen(
      Gdk::Screen.default,
      css_provider,
      ::Gtk::StyleProvider::PRIORITY_APPLICATION
    )
  end

  # ========================================================================= #
  # === create_pane_options
  #
  # This method will embed widgets into a Gtk::Frame.
  # ========================================================================= #
  def create_pane_options(paned, frame_label, label1, label2)
    frame = ::Gtk::Frame.new(frame_label)
    frame.border_width = 4

    table = ::Gtk::Table.new(3, 2, true)
    frame.add(table)

    label = ::Gtk::Label.new(label1)
    table.attach_defaults(label, 0, 1, 0, 1)

    check_button = ::Gtk::CheckButton.new('_Resize')
    table.attach_defaults(check_button, 0, 1, 1, 2)
    check_button.signal_connect(:toggled) {
      toggle_resize(paned.child1)
    }

    check_button = ::Gtk::CheckButton.new('_Shrink')
    table.attach_defaults(check_button, 0, 1, 2, 3)
    check_button.active = true
    check_button.signal_connect(:toggled) {
      toggle_shrink(paned.child1)
    }

    label = ::Gtk::Label.new(label2)
    table.attach_defaults(label, 1, 2, 0, 1)

    check_button = ::Gtk::CheckButton.new('_Resize')
    table.attach_defaults(check_button, 1, 2, 1, 2)
    check_button.active = true
    check_button.signal_connect(:toggled) {
      toggle_resize(paned.child2)
    }

    check_button = ::Gtk::CheckButton.new('_Shrink')
    table.attach_defaults(check_button, 1, 2, 2, 3)
    check_button.active = true
    check_button.signal_connect(:toggled) {
      toggle_shrink(paned.child2)
    }
    return frame
  end

  # ========================================================================= #
  # === toggle_resize
  # ========================================================================= #
  def toggle_resize(child)
    paned = child.parent # This is Gtk::Paned.

    is_child1 = (child == paned.child1)
    resize = if is_child1
      paned.child1_resize?
    else
      paned.child2_resize?
    end

    shrink = if is_child1
      paned.child1_shrink?
    else
      paned.child2_shrink?
    end

    child.parent.remove(child)
    if is_child1
      paned.pack1(child, !resize, shrink)
    else
      paned.pack2(child, resize: !resize, shrine: shrink)
    end
  end

  # ========================================================================= #
  # === toggle_shrink
  # ========================================================================= #
  def toggle_shrink(child)
    paned = child.parent
    is_child1 = (child == paned.child1)

    resize = if is_child1
      paned.child1_resize?
    else
      paned.child2_resize?
    end

    shrink = if is_child1
      paned.child1_shrink?
    else
      paned.child2_shrink?
    end
    child.parent.remove(child)
    if is_child1
      paned.pack1(child, resize: resize, shrink: !shrink)
    else
      paned.pack2(child, resize: resize, shrink: !shrink)
    end
  end

end; end

if __FILE__ == $PROGRAM_NAME
  window = Gtk::Window.new
  window.set_title('PanedWidgets')
  window.add(Gtk::PanedWidgetExample.new)
  window.set_size_request(1000, 500)
  window.signal_connect(:delete_event) { Gtk.main_quit }
  window.move(0, 0)
  window.show_all
  Gtk.main  
end