#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::TextBufferTags
#
# This module will create some TextBufferTags.
#
# Usage Examples:
#
#   Gtk::TextBufferTags.create_default_tags
#   Gtk::TextBufferTags.create_default_tags(@source_buffer_4)
#   @text_buffer = Gtk::TextBufferTags.create_my_tags(@text_buffer)
#
# =========================================================================== #
# require 'gtk_paradise/toplevel_methods/text_buffer_tags.rb'
# Gtk::TextBufferTags.create_default_tags()
# =========================================================================== #
module Gtk

module TextBufferTags # === Gtk::TextBufferTags

  # ========================================================================= #
  # === Gtk::TextBufferTags.h1_tag
  # ========================================================================= #
  def self.h1_tag(i)
    i.create_tag('heading1', { 'font' => 'Sans 22'})
  end

  # ========================================================================= #
  # === Gtk::TextBufferTags.h2_tag
  # ========================================================================= #
  def self.h2_tag(i)
    i.create_tag('heading2', { 'font' => 'Sans 20'})
  end

  # ========================================================================= #
  # === Gtk::TextBufferTags.create_default_tags
  #
  # The first argument must be a text-buffer.
  # ========================================================================= #
  def self.create_default_tags(
      i = @text_buffer
    )
    unless i.respond_to? :create_tag
      puts "The input class, #{i.class} does "\
           "not respond to :create_tag method."
    else # This is the default clause.
      # =================================================================== #
      # Create some headings first:
      # =================================================================== #
      Gtk::TextBufferTags.h1_tag(i) # i.create_tag('heading1', { 'font' => 'Sans 22'})
      Gtk::TextBufferTags.h2_tag(i) 
      i.create_tag('heading3', { 'font' => 'Sans 18'}) 
      i.create_tag('heading4', { 'font' => 'Sans 16'}) 
      i.create_tag('heading5', { 'font' => 'Sans 14'}) 
      i.create_tag('heading6', { 'font' => 'Sans 12'})
      # =================================================================== #
      # === Bold
      # =================================================================== #
      i.create_tag('b',        { weight: 600 })
      i.create_tag('bold',     { weight: 600 })
      # =================================================================== #
      # === Italic
      #
      # i.create_tag('bold',   { 'weight' => :bold })
      # italic:
      # i.create_tag('i',      { :weight => :italic })
      # =================================================================== #
      i.create_tag('italic',   { style: :italic })
      # oblique (kursiv):
      # i.create_tag('o',      { 'weight' => :oblique })
      # i.create_tag('kursiv', { 'weight' => :oblique })
      # small caps:
      # i.create_tag('sub',    { 'weight' => :small_caps })
      # Comments for various languages
      i.create_tag('comment',      { foreground: 'darkgray' })
      i.create_tag('ruby-comment', { foreground: 'darkgray' })
      # =================================================================== #
      # Some colours - COLOUR TAG.
      # =================================================================== #
      #
      # =================================================================== #
      # red:
      # =================================================================== #
      i.create_tag('red', { foreground: 'red' })
      # =================================================================== #
      # darkred:
      # =================================================================== #
      i.create_tag('darkred', { foreground: 'darkred' })
      # =================================================================== #
      # green:
      # =================================================================== #
      i.create_tag('green', { foreground: 'green'})
      # =================================================================== #
      # darkgreen:
      # =================================================================== #
      i.create_tag('darkgreen', { foreground: 'darkgreen'})
      # =================================================================== #
      # lightgreen:
      # =================================================================== #
      i.create_tag('lightgreen', { foreground: 'lightgreen'})
      # =================================================================== #
      # blue:
      # =================================================================== #
      i.create_tag('blue',      { foreground: 'blue'})
      # =================================================================== #
      # steelblue:
      # =================================================================== #
      i.create_tag('steelblue', { foreground: 'lightblue'})
      # =================================================================== #
      # seagreen:
      # =================================================================== #
      i.create_tag('seagreen', { foreground: 'seagreen'})
      # =================================================================== #
      # lightblue:
      # =================================================================== #
      i.create_tag('lightblue', { foreground: 'lightblue'})
      # =================================================================== #
      # darkblue:
      # =================================================================== #
      i.create_tag('darkblue', { foreground: 'darkblue'})
      # =================================================================== #
      # slateblue:
      # =================================================================== #
      i.create_tag('slateblue', { foreground: 'slateblue'})
      # =================================================================== #
      # tomato:
      # =================================================================== #
      i.create_tag('tomato', { foreground: 'tomato'})
      # =================================================================== #
      # violet:
      # =================================================================== #
      i.create_tag('violet', { foreground: 'violet'})
      # =================================================================== #
      # yellow:
      # =================================================================== #
      i.create_tag('yellow', { foreground: 'yellow'})
      # =================================================================== #
      # === Alignments / Justifications
      # =================================================================== #
      i.create_tag('right_side', { justification: :right })

      # =================================================================== #
      # === Background colours
      #
      # Some background colours are defined next.
      # =================================================================== #

      # =================================================================== #
      # bg_red:
      # =================================================================== #
      i.create_tag('bg_red', { background: 'red'})
      # =================================================================== #
      # bg_black:
      # =================================================================== #
      i.create_tag('bg_black', { background: 'black'})
      # =================================================================== #
      # bg_blanchedalmond:
      # =================================================================== #
      i.create_tag('bg_blanchedalmond', { background: 'blanchedalmond'})
      # =================================================================== #
      # bg_honeydew:
      # =================================================================== #
      i.create_tag('bg_honeydew', { background: 'honeydew'})
    end
    return i # Finally return the text-buffer tags here.
  end

  # ========================================================================= #
  # === create_my_tags
  # ========================================================================= #
  def create_my_tags(i = @text_buffer)
    Gtk::TextBufferTags.create_my_tags
  end

end

# =========================================================================== #
# === Gtk::Tokenizer
#
# Will fontify things. Can be reused to colourize Ruby code.
#
# To use this class, try something like:
#
#   @tokenizer = Gtk::Tokenizer.new
#
# =========================================================================== #
# require 'gtk_paradise/toplevel_methods/tokenizer_and_text_buffer_tags.rb'
# tokenizer = Gtk::Tokenizer.new
# =========================================================================== #
class Tokenizer # === Gtk::Tokenizer

  begin
    require 'roebe/toplevel_methods/keywords.rb'
    # ======================================================================= #
    # === RUBY_RESERVED_WORDS
    # ======================================================================= #
    RUBY_RESERVED_WORDS = Roebe.return_ruby_keywords
    # ======================================================================= #
    # === RUBY_RESERVED_WORDS_PATTERN
    # ======================================================================= #
    RUBY_RESERVED_WORDS_PATTERN = Regexp.compile(/(^|\s+)(#{RUBY_RESERVED_WORDS.collect do |pattern| 
      Regexp.quote(pattern) 
    end.join('|')})(\s+|$)/)
  rescue LoadError; end

  # ========================================================================= #
  # === handle_string
  #
  # This piece of code simply works through a given string. 
  # ========================================================================= #
  def handle_string(i)
    case i
    when /".+?"/,
         /'.+?'/
      tag = :string
    # ======================================================================= #
    # === :comment
    # ======================================================================= #
    when /#.*$/,
         /^=begin/,
         /^=end/  # ,/^\/\/.*$/
      tag = :comment
    # ======================================================================= #
    # === :parens
    #
    # The first is for empty () only IF a word is before it
    # ======================================================================= #   
    when /\(\)/ #/\w(\(\))$/ 
      tag = :parens    
    # ======================================================================= #
    # === :const
    # ======================================================================= #
    when /[A-Z][A-Za-z0-9_]+/
      tag = :const
    # ======================================================================= #
    # === :reserved
    # ======================================================================= #
    when RUBY_RESERVED_WORDS_PATTERN
      tag = :reserved
    else
      tag = nil 
    end
    return tag, $LAST_MATCH_INFO
  end

  # ========================================================================= #  
  # === tokenize
  # ========================================================================= #
  def tokenize(
      str   = 'hello world',
      index = 0
    )
    until str.empty?
      tag = nil
      # ==================================================================== #
      # Check given string and assign tags to it whenever you encounter
      # a special meaning, such as a :string or a :comment
      # Btw in ruby only # comment exist
      # ==================================================================== #
      tag, $LAST_MATCH_INFO = handle_string(str)
      # ==================================================================== #
      # We handled the tags, now its time to tokenize them again?
      # ==================================================================== #
      if tag
        # pre_match is zb:
        # =22
        # e "Pre_match was:"+$LAST_MATCH_INFO.pre_match unless $LAST_MATCH_INFO.pre_match.empty?
        #
        tokenize($LAST_MATCH_INFO.pre_match, index) { |*args|
          yield(*args)
        }
        start_pos = index + $LAST_MATCH_INFO.begin(0)
        end_pos   = index + $LAST_MATCH_INFO.end(0)      
        yield(tag, start_pos, end_pos)
        index += (str.length - $LAST_MATCH_INFO.post_match.length)
        str = $LAST_MATCH_INFO.post_match
      else
        index += str.length
        str = ''.dup
      end
    end
  end

end; end

if __FILE__ == $PROGRAM_NAME
  require 'gtk3'
  require 'gtk_paradise/run'
  r = Gtk.run
  r.use_this_font = :hack_25
  @text_buffer = Gtk.text_buffer
  Gtk::TextBufferTags.create_my_tags(@text_buffer)

  iter = @text_buffer.start_iter
  @text_buffer.insert(iter, "This is an editor.\n")
  iter.offset = 25

  # tag = @text_buffer.tag_table.lookup(:h2)
  # @text_buffer.insert(iter, 'A hyperlink: ', tags: [tag])

  @text_view  = Gtk::TextView.new(@text_buffer)
  r << @text_view
  r.run
  e
  e 'Next testing Gtk::Tokenizer:'
  e
  _ = Gtk::Tokenizer.new
  _.tokenize
  e
end