#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::CatMerge
#
# cat_merge.rb simply merges some files using cat.
# 
# As in:
#
#   cat a.vob b.vob > c.vob
#
# =========================================================================== #
require 'gtk_paradise/require_gtk2'

module Gtk

class CatMerge < ::Gtk::VBox

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  begin
    require 'colours'
    include ::Colours
  rescue LoadError; end

  require 'fileutils'
  require 'yaml'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 820

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 650

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      parent_widget,
      run_already = true
    )
    super()
    reset
    @parent_widget = parent_widget
    get_files
    run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    infer_the_namespace?
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    increase_font_size
  end

  # ========================================================================= #
  # === create_entries
  # ========================================================================= #
  def create_entries
    @entry_cmdline = gtk_entry
  end

  # ========================================================================= #
  # === get_files
  #
  # This method will obtain all files from the current directory.
  # ========================================================================= #
  def get_files
    @files = Dir["#{Dir.pwd}/*"]
    @files.reject! { |entry|
      if entry.downcase.include?('.vob')
        false
      else
        true
      end
    }
  end

  # ========================================================================= #
  # === create_combo_box
  # ========================================================================= #
  def create_combo_box
    @combo_box = gtk_combo_box
    @combo_box.wrap_width = 1
    @files.each { |the_file|
      @combo_box.append_text(the_file)
    }
    @combo_box.active = 0
  end

  # ========================================================================= #
  # === create_buttons
  # ========================================================================= #
  def create_buttons
    @button_merge = gtk_coloured_button('_Merge_slateblue')
    @button_merge.on_clicked { do_merge_files }
    # ======================================================================= #
    # Add a tooltip to the merge button next. The leading and trailing ' '
    # are just small visual improvements. :)
    # ======================================================================= #
    tooltip_for_button_merge = ' Press this button to merge together some files. '
    @button_merge.tooltip = tooltip_for_button_merge
    @button_quit = gtk_coloured_button('_Quit_slateblue',true)
  end
  
  # ========================================================================= #
  # === do_merge_files
  #
  # This is the action that will merge the files together.
  # ========================================================================= #
  def do_merge_files
    e
    e 'Now merging files:'
    e
    the_cmd = 'cat '.dup
    the_cmd << @files.join(' ')
    the_cmd << ' > '
    the_cmd << '_MERGED_FILES_'
    the_cmd << @files.first
    if @parent_widget.filename?
      the_cmd << " #{@parent_widget.filename?}"
    end
    e
    e the_cmd
    e
    @entry_cmdline.text = the_cmd
    # system(the_cmd)
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    toolbar = gtk_toolbar
    toolbar.add_open_file_entry(@parent_widget)
    pack_start(toolbar, false, false, 0)
    vbox_with_two_buttons = gtk_vbox
    vbox_with_two_buttons.add(@combo_box)
    vbox_with_two_buttons.pack_start(@entry_cmdline,false,true)
    # The two buttons, merge and quit, will be added next:
    hbox = gtk_hbox
    hbox.add(@button_merge)
    hbox.add(@button_quit)
    vbox_with_two_buttons.add(hbox).show_all
    add(vbox_with_two_buttons)
    show_all
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_entries
    create_buttons
    create_combo_box
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_skeleton
    connect_skeleton
  end

end; end

if __FILE__ == $PROGRAM_NAME
  require 'gtk_paradise/run'
  r = Gtk.run(
    Gtk::CatMerge::WIDTH,
    Gtk::CatMerge::HEIGHT,
    :tabble,
    'CatMerge',
    1
  )
  x = Gtk::CatMerge.new(r)
  r << x
  r.show_all
  r.modify_background(:normal, :grey)
  r.enable_simple_exit
  r.top_left_then_run
end # $RGTK/tools/cat_merge.rb