#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Dictionary.new
#
# x/GUI/GTK/TOOLS/dictionary.rb
# =========================================================================== #
require 'gtk_paradise/require_gtk2'

module Gtk

class Dictionary < ::Gtk::Frame

  begin
    require 'colours'
    include Colours
  rescue LoadErrors; end

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 280

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = 'Liberation Mono 20' # Hack 20.

  begin
    require 'yaml'
    require 'dictionaries'
    # ======================================================================= #
    # === ENGLISH_DICTIONARY
    # ======================================================================= #
    _ = Dictionaries.file_english
    if File.exist? _
      ENGLISH_DICTIONARY = YAML.load_file(_)
    end
  rescue LoadError; end

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(run_already = true)
    super(' Dictionary ')
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                  (reset tag)
  # ========================================================================= #
  def reset
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    # ======================================================================= #
    # === @search_string
    # ======================================================================= #
    @search_string = ''.dup
    # ======================================================================= #
    # === @use_this_browser
    # ======================================================================= #
    @use_this_browser = :palemoon # :firefox
    set_default_font    
  end

  # ========================================================================= #
  # === set_default_font
  # ========================================================================= #
  def set_default_font
    ::Gtk::Settings.default.gtk_font_name = USE_THIS_FONT
  end

  # ========================================================================= #
  # === modify_search_string
  #
  # This builds up our search string to use.
  #
  # Full example:
  #
  # firefox -new-tab "http://dict.leo.org/ende?lp=ende&lang=de&searchLoc=0&cmpType=relaxed&sectHdr=on&spellToler=on&search=&relink=on"
  # ========================================================================= #
  def modify_search_string
    _ = '"http://dict.leo.org/'.dup
    _ << 'ende?search='
    _ << @entry_search_term.text
    _ << '"'
    @search_string = _
  end

  # ========================================================================= #
  # === do_search
  # ========================================================================= #
  def do_search
    modify_search_string
    invoke_sys_cmd
  end

  # ========================================================================= #
  # === create_gtk_entry_completion
  # ========================================================================= #
  def create_gtk_entry_completion
    completion = gtk_entry_completion
    @entry_search_term.completion = completion
    @list_store = ::Gtk::ListStore.new(String)
    array = ENGLISH_DICTIONARY.keys # %w( GNOME total totally )
    array.each { |word|
      iter = @list_store.append
      iter[0] = word
    }
    completion.model = @list_store
    completion.text_column = 0
  end

  # ========================================================================= #
  # === invoke_sys_cmd
  #
  # triggers our sys cmd to our dcop object.
  # ========================================================================= #
  def invoke_sys_cmd
    modify_search_string # update it again
    ewarn @search_string
    Thread.new {
      system("#{@use_this_browser} #{@search_string}")
    }
  end

  # ========================================================================= #
  # === create_entries                            (entry tag, entries tag)
  #
  # This method will create all gtk-entries of that widget.
  # ========================================================================= #
  def create_entries
    @entry_search_term = gtk_entry
    @entry_search_term.can_default = true
    @entry_search_term.set_activates_default(true)
    @entry_search_term.on_key_press_event { |w, event|
      case Gdk::Keyval.to_name(event.keyval)
      when 'Return','KP_Enter' # React to Return and KP-Enter key evetns.
        do_search
      end
    }
    @entry_translation_of_the_word = gtk_entry
    @entry_translation_of_the_word.set_read_only
    create_gtk_entry_completion
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    label = gtk_bold_label(
      'Translate the following german word'
    )
    mini_vbox = paned_vbox(label, @entry_search_term)
    mini_vbox << @entry_translation_of_the_word
    button_pane = paned_vbox(@button_search, @button_translate)
    hbox = paned_hbox(mini_vbox, button_pane)
    add(hbox)
    show_all
  end

  # ========================================================================= #
  # === button_translate_clicked
  #
  # This is the action for when the user hits the button-translate.
  #
  # The intent, by the user, is to convert this into german.
  # ========================================================================= #
  def button_translate_clicked
    @entry_translation_of_the_word.set_text(
      Dictionaries.translate_this_from_english_to_german(
        @entry_search_term.text
      )
    )
  end

  # ========================================================================= #
  # === create_buttons                           (buttons tag, button tag)
  # ========================================================================= #
  def create_buttons
    # Both buttons will be bold.
    @button_search = gtk_bold_button('_Search') # Search tag.
    @button_search.on_clicked { |w,e|
      do_search unless @entry_search_term.text.empty?
    }
    @button_translate = gtk_bold_button('_Translate')
    @button_translate.on_clicked { |w,e|
      button_translate_clicked
    }
    [
      @button_search,
      @button_translate
    ].each {|this_button|
      this_button.modify_background(:active,   :coral)
      this_button.modify_background(:normal,   :cornsilk)
      this_button.modify_background(:prelight, :aliceblue)
    }
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_entries
    create_buttons
    modify_search_string
    connect_skeleton
  end

  # ========================================================================= #
  # === GtkDictionary.height?
  # ========================================================================= #
  def self.height?
    280
  end

  # ========================================================================= #
  # === GtkDictionary.width?
  # ========================================================================= #
  def self.width?
    WIDTH
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    5
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    '/home/x/DATA/images/PC/OS/LINUX/GOOGLE_MAPS.jpg'
  end

end; end

if __FILE__ == $PROGRAM_NAME
  require 'gtk_paradise/run.rb'
  x = Gtk::Dictionary.new
  r = Gtk.run
  r.height  = Gtk::Dictionary.height?
  r.width   = Gtk::Dictionary.width?
  r.name    = 'Dictionary'
  r.modify_background(:normal, :gray)
  r << x
  r.top_left
  r.enable_quick_exit_then_run
end # $RUBY_GTK/tools/dictionary.rb