#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::FileOpener
#
# This is a default widget for GTK Applications that can be started when
# a file is opened.
#
# It will also try to make use of the gem called BeautifulUrl, if
# available and installed on the computer system at hand.
# =========================================================================== #
# require 'gtk_paradise/classes/file_opener.rb
# Gtk.file_opener
# =========================================================================== #
require 'gtk_paradise/require_gtk2'

module Gtk

class FileOpener < ::Gtk::VBox # === Gtk::FileOpener

  begin
    require 'colours'
    include ::Colours::E
  rescue LoadError; end

  begin # Silent require of the beautiful_url project.
    require 'beautiful_url'
  rescue LoadError; end

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === ARRAY_ADD_THESE_SHORTCUT_FOLDERS
  #
  # Specify which shortcut-folders we ought to use.
  # ========================================================================= #
  ARRAY_ADD_THESE_SHORTCUT_FOLDERS =
  [
    ENV['DEPOT_PKG'],
    ENV['MY_TEMP'],
    ENV['MY_SONGS'],
    ENV['RUBY_SRC'],
    Dir.pwd,
    'C:\ingrid'
  ]

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      run_already = true
    )
    super()
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    # ======================================================================= #
    # === @title
    # ======================================================================= #
    @title = 'File Opener'
    # ======================================================================= #
    # === @use_this_editor
    # ======================================================================= #
    @use_this_editor = 'bluefish'
  end

  # ========================================================================= #
  # === use_this_editor?
  # ========================================================================= #
  def use_this_editor?
    @use_this_editor
  end; alias use_which_editor? use_this_editor? # === use_which_editor?

  # ========================================================================= #
  # === create_status_bar
  # ========================================================================= #
  def create_status_bar
    @status_bar = gtk_statusbar
    @status_bar.has_resize_grip = true
    @status_bar_context_id = @status_bar.get_context_id('FileOpener')
  end

  # ========================================================================= #
  # === create_gtk_entries                           (entry tag, entries tag)
  # ========================================================================= #
  def create_gtk_entries
    @entry = gtk_entry
    @entry.set_max_length 80
    @entry.set_activates_default(true)
    @entry.set_focus(true) 
    @entry.on_activate {|widget| open_this_file }
    @entry.on_button_press_event {|widget, event| 
      @entry.set_focus(true) 
      @entry.on_click_select_all 
    }
    @entry_for_the_editor_in_use = gtk_entry(use_which_editor?)
    @entry_for_the_editor_in_use.on_click_select_all
  end

  # ========================================================================= #
  # === return_the_value_in_the_gtk_entry_keeping_track_of_the_editor_in_use
  # ========================================================================= #
  def return_the_value_in_the_gtk_entry_keeping_track_of_the_editor_in_use
    @entry_for_the_editor_in_use.text
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    add(
      create_paned_hbox(
        @entry,
        @button_open_this_sanitized_file
      )
    )
    minimal(@entry_for_the_editor_in_use, 0)
    add(@status_bar)
  end

  # ========================================================================= #
  # === create_gtk_buttons
  # ========================================================================= #
  def create_gtk_buttons
    @button_open_this_sanitized_file = gtk_button('_Sanitized File Opening')
    @button_open_this_sanitized_file.modify_background( :active,   :coral )
    @button_open_this_sanitized_file.modify_background( :normal,   :cornsilk )
    @button_open_this_sanitized_file.modify_background( :prelight, :aliceblue )
    @button_open_this_sanitized_file.can_default = true
    # ======================================================================= #
    # Next, determine what to do when this button is clicked:
    # ======================================================================= #
    @button_open_this_sanitized_file.on_clicked {
      @entry.set_focus(true)
      open_this_file
    } # left mouse click
    @button_open_this_sanitized_file.signal_connect(:event) {|widget, event| # right mouse click event
      case event.event_type.name
      when 'GDK_BUTTON_PRESS'
        create_file_chooser if event.button == 3
      end
    }
    @button_open_this_sanitized_file.fancy_tooltips = 
      'This will <b>open the file in your editor</b>. '
      'The file path will be displayed on the bottom.'
  end

  # ========================================================================= #
  # === open_this_file
  #
  # This opens the file in bluefish editor, and also modifies our status bar.
  # This also sets @url.
  #
  # For this to work, an argument has to be given.
  # ========================================================================= #
  def open_this_file(
      i                = @entry.text,
      with_this_editor = return_the_value_in_the_gtk_entry_keeping_track_of_the_editor_in_use
    )
    _ = BeautifulUrl[i]
    _ = _.first if _.is_a? Array
    _ = _.dup if _.frozen?
    if _.start_with? 'http://localhost/DATA'
      _.sub!(
        /^http:\/\/localhost\/DATA/, ENV['MY_DATA']
      )
    end
    @url = _
    if File.exist? @url
      @status_bar.push(@status_bar_context_id, @url)
      esystem(with_this_editor+' '+@url) # Open in the editor.
    else
      _ = "File `#{@url}` does not exist."
      @status_bar.push(@status_bar_context_id, _)
      e _
    end
  end

  # ========================================================================= #
  # === create_file_chooser
  # ========================================================================= #
  def create_file_chooser
    # ======================================================================= #
    # Assign to the @file_chooser_dialog variable next.
    # ======================================================================= #
    @file_chooser_dialog = FileChooserDialog.new('Open File', self, 
      Gtk::FileChooser::ACTION_OPEN, nil,
     [Gtk::Stock::CANCEL, Gtk::Dialog::RESPONSE_CANCEL],
     [Gtk::Stock::OPEN,   Gtk::Dialog::RESPONSE_ACCEPT]
    )
    @file_chooser_dialog.current_folder = Dir.pwd
    ARRAY_ADD_THESE_SHORTCUT_FOLDERS.each {|folder|
      begin
        @file_chooser_dialog.add_shortcut_folder(folder)
      rescue Exception => error
        pp error if ::Gtk.is_on_roebe?
      end
    }
    @file_chooser_dialog.show_hidden = true # Do show hidden files.
    @extra_button = gtk_button('_Extra button') 
    @extra_button.on_clicked { e ' > Extra button was clicked' }
    @file_chooser_dialog.extra_widget = @extra_button
    case @file_chooser_dialog.run
    when Gtk::Dialog::RESPONSE_ACCEPT 
      filename = @file_chooser_dialog.filename
      open_this_file(filename)
      @entry.set_text(filename)
      e filename
    end
    @file_chooser_dialog.destroy
  end; alias do_pick_file create_file_chooser # === do_pick_file

  # ========================================================================= #
  # === file?
  # ========================================================================= #
  def file?
    @entry.text
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    4
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    :tabble
  end

  # ========================================================================= #
  # === width?
  # ========================================================================= #
  def width?
    300
  end

  # ========================================================================= #
  # === height?
  # ========================================================================= #
  def height?
    500
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_gtk_entries
    create_gtk_buttons
    create_status_bar
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_skeleton
    connect_skeleton
  end

end

# =========================================================================== #
# === Gtk.file_opener
#
# Easier toplevel-method for opening a new file.
# =========================================================================== #
def self.file_opener
  Gtk::FileOpener.new
end

end

if __FILE__ == $PROGRAM_NAME
  require 'gtk_paradise/run.rb'
  x = Gtk::FileOpener.new
  r = Gtk.run
  r << x
  r.modify_background :normal, :grey
  r.enable_quick_exit_then_top_left_then_run
end # gtkfileopener