#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::DirectoryContentWidget
#
# This widget will show the content of a given directory, that is
# whether there are directories, files and what not.
#
# The idea is to enhance this widget so that we can simply re-use
# it in many different projects as-is.
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/directory_content_widget/directory_content_widget.rb'
# Gtk::DirectoryContentWidget.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class DirectoryContentWidget < ::Gtk::BaseModuleBox # === Gtk::DirectoryContentWidget 

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Directory Content'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '70% or minimum 1500px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '60% or minimum 520px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = 'Noto Mono 18' # :hack_17 # :dejavu_condensed_17

  # ========================================================================= #
  # === USE_THIS_FONT_FOR_THE_SCROLLED_WINDOW
  # ========================================================================= #
  USE_THIS_FONT_FOR_THE_SCROLLED_WINDOW = 'Noto Mono 16'

  # ========================================================================= #
  # === Gtk::DirectoryContentWidget::HASH_DESIGNATED_KEY_COMBINATIONS
  # ========================================================================= #
  HASH_DESIGNATED_KEY_COMBINATIONS = {
    'alt + ←' => 'down_by_one',
    'alt + w' => 'down_by_one',
    'alt + 1' => 'activate_list_at_position(0)',
    'alt + 2' => 'activate_list_at_position(1)',
    'alt + 3' => 'activate_list_at_position(2)',
    'alt + 4' => 'activate_list_at_position(3)',
    'alt + 5' => 'activate_list_at_position(4)',
    'alt + 6' => 'activate_list_at_position(5)',
    'alt + 7' => 'activate_list_at_position(6)',
    'alt + 8' => 'activate_list_at_position(7)',
    'alt + 9' => 'activate_list_at_position(8)'
  }

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      optional_use_this_as_the_start_directory = :return_pwd,
      run_already                              = true,
      &block
    )
    super(:vertical)
    reset
    set_use_this_directory(
      optional_use_this_as_the_start_directory
    )
    set_start_directory(use_this_directory?.dup) # Copy it on startup of this widget.
    raw_cd(@user_configuration[:start_directory])
    # ======================================================================= #
    # === Handle blocks given next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :do_not_use_widget_allowing_the_user_to_change_the_local_directory
      #
      # This Symbol will omit the bottom-widget that is normally a part
      # of this class.
      # ===================================================================== #
      when :do_not_use_widget_allowing_the_user_to_change_the_local_directory,
           :omit_the_bottom_widget
        @show_widget_allowing_the_user_to_change_the_local_directory = false
      # ===================================================================== #
      # === :fancy_CSS
      # ===================================================================== #
      when :fancy_CSS,
           :do_customize_the_CSS_rules
        do_customize_the_CSS_rules
      end
    end
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    infer_the_namespace
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ====================================================================== #
    # === @context_menu
    #
    # Define the context-menu that we want to enable for this widget.
    # ====================================================================== #
    @context_menu = create_context_menu(self) {{
      # n_entries: 12,
      actions: {
        'Quit':               :exit_the_application,
        'Create a directory': :interactively_create_directory
      }
    }}
    @context_menu.css_class(
      'bblack3_and_padding'
    )
    # ======================================================================= #
    # === @show_widget_allowing_the_user_to_change_the_local_directory
    #
    # If this variable is set to true then a bottom-widget will be used 
    # that allows the user to quickly change the directory.
    # ======================================================================= #
    @show_widget_allowing_the_user_to_change_the_local_directory = true
    infer_the_size_automatically
    # ======================================================================= #
    # === @user_configuration
    #
    # This is deliberately different from @internal_hash because it might
    # otherwise conflict with the same named variable that is part of
    # Gtk::BaseModule.
    # ======================================================================= #
    @user_configuration = {}
    # ======================================================================= #
    # === :use_this_browser
    # ======================================================================= #
    @user_configuration[:use_this_browser] = :palemoon
    # ======================================================================= #
    # === :start_directory    
    # ======================================================================= #
    @user_configuration[:start_directory] = nil
    # ======================================================================= #
    # === :use_this_editor
    # ======================================================================= #
    @user_configuration[:use_this_editor] = :bluefish
    # ======================================================================= #
    # === :use_this_multimedia_player
    # ======================================================================= #
    @user_configuration[:use_this_multimedia_player] = :mpv
    # ======================================================================= #
    # === :use_this_pdf_viewer
    # ======================================================================= #
    @user_configuration[:use_this_pdf_viewer] = :evince
    # ======================================================================= #
    # === :use_this_directory
    #
    # This variable denotes the start-directory. It will either be the
    # current working directory, or the first argument in use.
    # ======================================================================= #
    @user_configuration[:use_this_directory] = return_pwd
    # ======================================================================= #
    # === :last_selection
    # ======================================================================= #
    @user_configuration[:last_selection] = nil
    # ======================================================================= #
    # === :customize_the_CSS_rules
    #
    # This variable keeps track as to whether we will modify the CSS rules
    # for the tree-view (by default) or not.
    # ======================================================================= #
    @user_configuration[:customize_the_CSS_rules] = false
    handle_CSS
  end

  # ========================================================================= #
  # === use_which_pdf_viewer?
  # ========================================================================= #
  def use_which_pdf_viewer?
    @user_configuration[:use_this_pdf_viewer]
  end

  # ========================================================================= #
  # === use_which_multimedia_player?
  # ========================================================================= #
  def use_which_multimedia_player?
    @user_configuration[:use_this_multimedia_player]
  end

  # ========================================================================= #
  # === use_which_editor?
  # ========================================================================= #
  def use_which_editor?
    @user_configuration[:use_this_editor]
  end

  # ========================================================================= #
  # === use_which_browser?
  # ========================================================================= #
  def use_which_browser?
    @user_configuration[:use_this_browser]
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    0
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === handle_this_file_or_symlink
  #
  # This method can be used to do specific actions, such as playing
  # a .mp3 song and so forth.
  # ========================================================================= #
  def handle_this_file_or_symlink(
      i = last_selection?
    )
    case i
    # ======================================================================= #
    # === Images
    # ======================================================================= #
    when /\.png$/i,
         /\.jpg$/i
      Thread.new {
        esystem "#{use_which_browser?} #{i}"
      }
    # ======================================================================= #
    # === .md
    #
    # The following file-types will be opened via the bluefish editor.
    #
    # At some later point I will make this more flexible.
    # ======================================================================= #
    when /\.md$/i,
         /\.yml$/i,
         /\.rb$/i,
         /\.py$/i,
         /\.txt$/i,
         /\.pdb$/i,
         /\.fa$/i,
         /\.fasta$/i
      Thread.new {
        esystem "#{use_which_editor?} #{i}"
      }
      if i.end_with? '.fasta' # Handle .fasta files slightly differently.
        e 'Displaying the content of the file.'
        Gtk.display_this_file_and_then_run_it(i)
      end
    # ======================================================================= #
    # === .mp3
    #
    # Here we will put multimedia-files down.
    # ======================================================================= #
    when /\.mp3$/i,
         /\.mp4$/i,
         /\.avi$/i,
         /\.webm$/i,
         /\.mkv$/i
      Thread.new {
        esystem "#{use_which_multimedia_player?} \"#{i}\""
      }
    # ======================================================================= #
    # === .pdf
    # ======================================================================= #
    when /\.pdf$/i
      Thread.new {
        esystem "#{use_which_pdf_viewer?} \"#{i}\""
      }
    # ======================================================================= #
    # === Handle archives next
    # ======================================================================= #
    when /\.xz$/i,
         /\.zip$/i,
         /\.tar.bz2$/i
      Thread.new {
        extract_this_archive(i)
      }
    end
  end

  # ========================================================================= #
  # === set_directory_content
  # ========================================================================= #
  def set_directory_content(i)
    @directory_content = i
  end

  # ========================================================================= #
  # === use_this_directory?
  # ========================================================================= #
  def use_this_directory?
    @user_configuration[:use_this_directory]
  end; alias current_dir? use_this_directory? # === current_dir?
       alias pwd?         use_this_directory? # === pwd?

  # ========================================================================= #
  # === context_menu?
  # ========================================================================= #
  def context_menu?
    @context_menu
  end

  # ========================================================================= #
  # === interactively_create_directory
  # ========================================================================= #
  def interactively_create_directory
    e 'Input the name of the directory that is to be created:'
  end

  # ========================================================================= #
  # === set_start_directory
  # ========================================================================= #
  def set_start_directory(i)
    @user_configuration[:start_directory] = i.to_s.dup
  end; alias set_start_dir set_start_directory # === set_start_dir

  # ========================================================================= #
  # === set_use_this_directory
  #
  # Usage example:
  #
  #   set_use_this_directory(:return_pwd)
  #
  # ========================================================================= #
  def set_use_this_directory(
      i = return_pwd
    )
    if i.is_a? Array
      i = i.join(' ').strip
    end
    case i
    # ======================================================================= #
    # === :return_pwd
    # ======================================================================= #
    when :return_pwd,
         :pwd,
         ''
      i = return_pwd
    end
    if i
      @user_configuration[:use_this_directory] = i
    end
  end

  # ========================================================================= #
  # === clear
  # ========================================================================= #
  def clear
    @list_store.clear
  end

  # ========================================================================= #
  # === feed_this_data
  #
  # Currently this method can only work on Arrays.
  # ========================================================================= #
  def feed_this_data(i)
    if i.is_a? Array
      fill_up_the_main_iter_with(i)
    end
  end

  # ========================================================================= #
  # === set_pwd
  # ========================================================================= #
  def set_pwd(
      i = :return_pwd
    )
    set_use_this_directory(:return_pwd)
    update
  end

  # ========================================================================= #
  # === cd_then_update
  # ========================================================================= #
  def cd_then_update(
      i = last_selection?
    )
    cd i
    update
  end

  # ========================================================================= #
  # === activate_list_at_position
  #
  # This is currently (May 2021) incomplete.
  # ========================================================================= #
  def activate_list_at_position(i = 0)
    treeview = treeview?
    i = i.to_i # We need a number.
    path = Gtk::TreePath.new(i.to_s)
    column = treeview.columns?[0]
    edit = false
    treeview.set_cursor(path, column, edit)
    treeview.grab_focus
  end

  # ========================================================================= #
  # === extract_this_archive
  # ========================================================================= #
  def extract_this_archive(i) 
    if File.exist? i
      begin
        require 'extracter'
        Extracter.new(i)
      rescue LoadError; end
    end
  end

  # ========================================================================= #
  # === obtain_directory_content
  # ========================================================================= #
  def obtain_directory_content(
      from_this_directory = use_this_directory?
    )
    @directory_content = Dir[
      ("#{from_this_directory}/*").squeeze('/')
    ].sort.map {|entry| File.absolute_path(entry) }
  end

  # ========================================================================= #
  # === down_by_one
  # ========================================================================= #
  def down_by_one
    cd_one_directory_downwards
    update
  end; alias down down_by_one # === down

  # ========================================================================= #
  # === customize_the_CSS_rules?
  # ========================================================================= #
  def customize_the_CSS_rules?
    @user_configuration[:customize_the_CSS_rules]
  end

  # ========================================================================= #
  # === do_customize_the_CSS_rules
  # ========================================================================= #
  def do_customize_the_CSS_rules
    @user_configuration[:customize_the_CSS_rules] = true
  end

  # ========================================================================= #
  # === handle_CSS
  # ========================================================================= #
  def handle_CSS
    use_gtk_paradise_project_css_file
    # ======================================================================= #
    # We can style the treeview via CSS, but this is not always wanted,
    # so it has to be customizable.
    # ======================================================================= #
    if customize_the_CSS_rules?
      more_css '
treeview.view header button {
  color: steelblue;
}

treeview.view {
  color: lightgreen;
  background-color: black;
}

'
    end
  end

  # ========================================================================= #
  # === create_the_labels                                        (labels tag)
  #
  # This method will create the 'Directory:' label.
  # ========================================================================= #
  def create_the_labels
    # ======================================================================= #
    # === @label_directory
    # ======================================================================= #
    @label_directory = gtk_label('<b>Directory</b>: ')
    @label_directory.css_class('midnightblue')
    @label_directory.hint = 'To the right of this text '\
      'you can specify the target directory on the local filesystem.'
  end

  # ========================================================================= #
  # === do_select_everything_in_the_main_entry
  # ========================================================================= #
  def do_select_everything_in_the_main_entry
    @entry_use_this_directory.select_everything
  end

  # ========================================================================= #
  # === create_liststore
  # ========================================================================= #
  def create_liststore
    @list_store = Gtk::ListStore.new(
      String,            # Name of the file or directory.
      GdkPixbuf::Pixbuf, # This can be an icon for a file or for a directory.
      String,
      String,
      String
    )
    # ======================================================================= #
    # Sort the file-size entry a bit differently.
    # ======================================================================= #
    @list_store.set_sort_func(3) { |_model, iter1, iter2|
      iter2[3].to_s.sub(/ kb/,'').to_f <=> iter1[3].to_s.sub(/ kb/,'').to_f
    }
    # ======================================================================= #
    # Next sort the "Last Modified" entry properly.
    # ======================================================================= #
    @list_store.set_sort_func(4) { |_model, iter1, iter2|
      parse1 = ::Time.parse(iter2[4])
      parse = ::Time.parse(iter1[4]) 
      parse1 <=> parse2
    }
    update
  end

  # ========================================================================= #
  # === last_selection?
  #
  # This method will return which entry is the last one that has been
  # selected by the user.
  # ========================================================================= #
  def last_selection?
    @user_configuration[:last_selection]
  end; alias selected?  last_selection? # === selected?
       alias selection? last_selection? # === selection?

  # ========================================================================= #
  # === create_entries                               (entries tag, entry tag)
  # ========================================================================= #
  def create_entries
    # ======================================================================= #
    # This is the entry that appears on the very bottom area of this widget.
    # ======================================================================= #
    @entry_use_this_directory = entry
    @entry_use_this_directory.yellow_background
    # @entry_use_this_directory.on_click_select_everything
    @entry_use_this_directory.on_enter {
      target = @entry_use_this_directory.text.to_s
      unless target.end_with? '/'
        target = target.dup if target.frozen?
        target << '/'
        @entry_use_this_directory.set_text(target)
        @entry_use_this_directory.cursor_is_right
      end
      if File.directory? target
        cd(target)
        set_use_this_directory(target)
        sync
      end
    }
  end

  # ========================================================================= #
  # === create_the_buttons                          (buttons tag, button tag)
  # ========================================================================= #
  def create_the_buttons
    # ======================================================================= #
    # === @button_update
    # ======================================================================= #
    @button_update = bold_button('_Update')
    @button_update.hint = 
      'Click this button to update the content listing shown above.'
    @button_update.set_name('button1')
    @button_update.on_clicked {
      update
    }
    @button_cd_one_level_down = button('..')
    @button_cd_one_level_down.set_name('button1')
    @button_cd_one_level_down.on_clicked {
      cd_one_directory_downwards
      set_use_this_directory(:return_pwd)
      update
    }
  end

  # ========================================================================= #
  # === sync_directory_content_onto_the_main_liststore
  # ========================================================================= #
  def sync_directory_content_onto_the_main_liststore(
      i = @directory_content
    )
    array = []
    i.each {|entry|
      entry = entry.dup if entry.frozen?
      filesize = File.size(entry).to_s
      # ===================================================================== #
      # Determine the file type next, such as "file" or "directory".
      # ===================================================================== #
      type = File.ftype(entry)
      last_modified = File.mtime(entry)
      last_modified_as_string = last_modified.day.to_s.rjust(2,'0')+'.'+
                                last_modified.month.to_s.rjust(2,'0')+'.'+
                                last_modified.year.to_s
      shorter_filename = File.basename(entry)
      case type
      when 'directory' # A directory, such as "/tmp", should have a trailing "/".
        shorter_filename << '/'
      end
      # ===================================================================== #
      # Next build up our array that is then used to populate the liststore
      # at hand.
      # ===================================================================== #
      array << [
        shorter_filename,
        type,
        type,
        filesize,
        last_modified_as_string
      ]
    }
    fill_up_the_main_iter_with(array)
  end

  # ========================================================================= #
  # === fill_up_the_main_iter_with
  # ========================================================================= #
  def fill_up_the_main_iter_with(array)
    array.each {|shorter_filename, pixbuf_representation, type, filesize, last_modified_as_string|
      iter = @list_store.append

      case pixbuf_representation
      when 'file'
        pixbuf_representation = icon_theme_default(:document_open,   16)
      when 'directory'
        pixbuf_representation = icon_theme_default(:inode_directory, 16)
      when 'link'
        pixbuf_representation = icon_theme_default(:document_open,   16)
      else
        e 'Unhandled in the method: fill_up_the_main_iter_with() '+
          pixbuf_representation
      end

      iter.set_value(0, shorter_filename)
      iter.set_value(1, pixbuf_representation)
      iter.set_value(2, type)
      filesize = (filesize.to_f / 1024).round(1).to_s
      filesize = '%.2f' % filesize.to_f # Format the filesize next.
      filesize = filesize.rjust(9, ' ')
      iter.set_value(3, "#{filesize} kb")
      iter.set_value(4, last_modified_as_string) 
    }
  end

  # ========================================================================= #
  # === main_entry?
  # ========================================================================= #
  def main_entry?
    @entry_use_this_directory
  end

  # ========================================================================= #
  # === create_skeleton                            (create tag, skeleton tag)
  # ========================================================================= #
  def create_skeleton
    create_the_labels
    create_the_buttons
    create_entries
    create_liststore
    create_treeview
    create_scrolled_window
  end

  # ========================================================================= #
  # === determine_the_minimum_size_of_the_scrolled_window
  # ========================================================================= #
  def determine_the_minimum_size_of_the_scrolled_window
    @scrolled_window.set_size_request(450, 260)
  end

  # ========================================================================= #
  # === show_what_was_selected
  # ========================================================================= #
  def show_what_was_selected(i = last_selection?)
    unless i.empty?
      e "Selected: #{sfancy(i)}"
    end
  end

  # ========================================================================= #
  # === deselect_the_treeview
  # ========================================================================= #
  def deselect_the_treeview
    @tree_view.deselect
  end

  # ========================================================================= #
  # === create_treeview
  #
  # This method will create the main tree-view of this widget, and
  # make it useful.
  # ========================================================================= #
  def create_treeview
    # ======================================================================= #
    # === @tree_view
    # ======================================================================= #
    @tree_view = gtk_tree_view(@list_store) { :clickable_headers }
    @tree_view.enable_selection_single
    @tree_view.do_use_clickable_headers
    @tree_view.enable_drag_and_drop
    @tree_view.reorderable = true
    @tree_view.rules_hint  = true
    @tree_view.enable_search = true
    @tree_view.unselect_all # By default we don't want anything selected.
    # ======================================================================= #
    # We add the headers next: name, type, size and last_modified.
    #
    # Name is the local filename or name of the directory or symlink.
    # Type is whether it is a directory, a file and so forth.
    # Last_modified keeps track when this file or directory was last
    # modified.
    # ======================================================================= #
    name_column                 = gtk_tree_view_column('Name',          gtk_cell_renderer,        text: 0)
    image_representation_column = gtk_tree_view_column('',              gtk_cell_renderer_pixbuf, pixbuf: 1)
    type_column                 = gtk_tree_view_column('Type',          gtk_cell_renderer_centered_text, text: 2)
    type_column.center
    size_column                 = gtk_tree_view_column('Size (in kb)',  gtk_cell_renderer_right_aligned, text: 3)
    last_modified_column        = gtk_tree_view_column('Last Modified', gtk_cell_renderer, text: 4)
    @tree_view.append_columns(
      name_column,
      image_representation_column,
      type_column,
      size_column,
      last_modified_column
    )
    @tree_view.allow_resizing # This must come after the columns were added.
    columns = @tree_view.columns?
    # ======================================================================= #
    # Enable sorting for some of the entries next.
    # ======================================================================= #
    columns[0].set_sort_column_id(0) # This here sorts by name.
    columns[2].set_sort_column_id(2) # This here sorts by type, e. g. "directory" or "file".
    columns[3].set_sort_column_id(3) # This here sorts by size.
    columns[4].set_sort_column_id(4) # This here sorts by "Last Modified".
    # columns[1].cell_renderers?[0].center
    # ======================================================================= #
    # Next act on single mouse-button-click events.
    # ======================================================================= #
    @tree_view.on_button_release_event { |widget, event|
      selection = widget.selected?.to_s
      @user_configuration[:last_selection] = selection
      show_what_was_selected
    }
    @tree_view.on_enter { |widget, event|
      set_last_selection(File.absolute_path(@tree_view.selected?.to_s))
      e "Selection: #{@user_configuration[:last_selection]}"
      if last_selection?
        if File.directory? @user_configuration[:last_selection]
          cd_then_update(@user_configuration[:last_selection])
        else
          handle_this_file_or_symlink(@user_configuration[:last_selection])
        end
      end
    }
    @tree_view.on_button_press_event { |widget, event|
      case event.button
      when 1 # left-mouse button click event.
        if event.event_type == Gdk::EventType::BUTTON2_PRESS
          # ================================================================= #
          # === :last_selection
          #
          # This here is the case for a double-click event. We will keep
          # track of the last selection via an instance variable.
          # ================================================================= #
          # e 'Even a double-click event!'
          @user_configuration[:last_selection] = File.absolute_path(
            @tree_view.return_the_selection.to_s
          )
          show_what_was_selected
          if File.directory? @user_configuration[:last_selection]
            e 'Changing into the directory '+
              sdir(@user_configuration[:last_selection])+
              ' next.'
            cd_then_update(@user_configuration[:last_selection])
          else
            handle_this_file_or_symlink(@user_configuration[:last_selection])
          end
        else # else this is a simple left-mouse button click.
          # @user_configuration[:last_selection] = @tree_view.return_the_selection.to_s
        end
      when 2
        # e 'middle mouse-button was clicked.'
      when 3
        # =================================================================== #
        # Right-mouse button was clicked.
        # =================================================================== #
        deselect_the_treeview
      end
    }
  end

  # ========================================================================= #
  # === cd_one_directory_downwards
  # ========================================================================= #
  def cd_one_directory_downwards
    cd File.absolute_path(File.dirname(return_pwd))
  end

  # ========================================================================= #
  # === update                                                   (update tag)
  # ========================================================================= #
  def update
    clear
    obtain_directory_content
    sync_directory_content_onto_the_main_liststore
    @entry_use_this_directory.set_text(use_this_directory?.to_s)
  end; alias sync update # === sync

  # ========================================================================= #
  # === change_directory                                             (cd tag)
  #
  # This class needs its own implementation of cd-functionality.
  # ========================================================================= #
  def change_directory(i)
    if File.directory? i
      raw_cd(i)
      set_use_this_directory(File.absolute_path(i))
    end
  end; alias cd change_directory # === cd

  # ========================================================================= #
  # === set_last_selection
  # ========================================================================= #
  def set_last_selection(i)
    @user_configuration[:last_selection] = i
  end; alias last_selection= set_last_selection # === last_selection=

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    top_box = gtk_hbox
    top_box.maximal(@scrolled_window)
    lower_box = gtk_hbox
    if @show_widget_allowing_the_user_to_change_the_local_directory
      mini_hbox = gtk_hbox
      small_icon_on_the_left_side = image_directory # gtk_label(:directory)
      @event_box_for_the_small_icon_on_the_left_side = gtk_event_box(
        small_icon_on_the_left_side
      )
      mini_hbox.minimal(@event_box_for_the_small_icon_on_the_left_side, 4)
      @event_box_for_the_small_icon_on_the_left_side.on_clicked {
        do_select_everything_in_the_main_entry
      }
      mini_hbox.minimal(@label_directory)
      mini_hbox.maximal(@entry_use_this_directory)
      mini_hbox.minimal(@button_cd_one_level_down, 2)
      lower_box.maximal(mini_hbox)
      # ===================================================================== #
      # Since as of August 2022, the update-button is also omitted.
      # ===================================================================== #
      lower_box.minimal(@button_update)
      draggable_pane = gtk_drag_left_to_right(
        top_box,
        lower_box
      )
      draggable_pane.position = 900 # The higher the number, the lower the bottom pane will be.
      maximal(draggable_pane)
    else
      maximal(top_box)
    end
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
    Thread.new {
      sleep 0.01
      main_entry?.deselect
    }
  end

  # ========================================================================= #
  # === create_scrolled_window                                 (scrolled tag)
  # ========================================================================= #
  def create_scrolled_window
    # ======================================================================= #
    # === @scrolled_window
    # ======================================================================= #
    @scrolled_window = gtk_scrolled_window(treeview?) { :always }
    @scrolled_window.pad8px
    @scrolled_window.bblack1
    @scrolled_window.use_this_font = USE_THIS_FONT_FOR_THE_SCROLLED_WINDOW
    determine_the_minimum_size_of_the_scrolled_window
  end

  # ========================================================================= #
  # === treeview?
  # ========================================================================= #
  def treeview?
    @tree_view
  end

  # ========================================================================= #
  # === use_this_font=
  # ========================================================================= #
  def use_this_font=(i)
    # Sync it towards other widgets, including the entries.
    @tree_view.use_this_font = i
    @entry_use_this_directory.use_this_font = i
    self.set_font(i)
  end

  # ========================================================================= #
  # === Gtk::DirectoryContentWidget.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::DirectoryContentWidget.new(i)
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.add_accel_group(@accel_group = ::Gtk::AccelGroup.new)
    r.enable_context_menu
    _.set_parent_widget(r)
    r.enable_quick_exit
    r.signal_connect(:button_press_event) { |widget, event|
      if ::Gtk.right_mouse_click?(event) # right mouse click event.
        _.context_menu?.popup_based_on_this_event(event)
      end
    }
    # ====================================================================== #
    # Enable additional key-combinations next:
    # ====================================================================== #
    _.enable_these_key_combinations(
      ::Gtk::DirectoryContentWidget::HASH_DESIGNATED_KEY_COMBINATIONS
    )
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::DirectoryContentWidget.run
end