#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::FontSizeWidget
#
# This widget is essentially a scrolled window that contains some entries
# for picking a specific font. That way it can be embedded into, for
# instance, Gtk::Editor.
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/font_size_widget/font_size_widget.rb'
# Gtk::FontSizeWidget.new
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class FontSizeWidget < ::Gtk::BaseModuleBox # === Gtk::FontSizeWidget 

  require 'gtk_paradise/widgets/gtk3/font_size_widget/action_button.rb'

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Font Size Widget'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '10% or minimum 120px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 18

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_15

  # ========================================================================= #
  # === ADD_N_BUTTONS
  #
  # If you display this in 3-buttons-per-row then this should be divisable
  # by 3 as well - otherwise you will have empty spots in the displayed
  # layout. This explains why the chosen numbed was 21 here, as the
  # default.
  # ========================================================================= #
  ADD_N_BUTTONS = 18

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      if yielded.is_a? Hash
        # =================================================================== #
        # === :display_n_font_buttons_per_line
        #
        # This allows us to overrule how many buttons shall be displayed
        # on the FontWidget.
        # =================================================================== #
        if yielded.has_key? :display_n_font_buttons_per_line
          _ = yielded[:display_n_font_buttons_per_line]
          set_display_n_font_buttons_per_line(_)
        end
        # =================================================================== #
        # === :use_this_font
        #
        # This method requires that an instance variable has been
        # properly set.
        #
        # Example:
        #
        #   use_this_font: USE_THIS_SMALLER_FONT
        #
        # =================================================================== #
        if yielded.has_key? :use_this_font
          @use_this_font = yielded[:use_this_font]
        end
        # =================================================================== #
        # === :enable_debug
        # =================================================================== #
        if yielded.has_key? :enable_debug
          if (yielded[:enable_debug] == true)
            do_enable_debug
          end
        end
      end
    end
    create_skeleton # Must come after we handle the commandline arguments above.
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
    reset_font_size_and_font_family
    # ======================================================================= #
    # === @parent_widget
    # ======================================================================= #
    @parent_widget = nil
    # ======================================================================= #
    # === @array_buttons
    #
    # Keep track of the buttons defined and in use in the following Array.
    # ======================================================================= #
    @array_buttons = []
    # ======================================================================= #
    # === @shall_we_use_a_user_input_widget_on_top
    #
    # If the following instance variable is true then a gtk-entry widget
    # will be put on "top" of the other boxes.
    # ======================================================================= #
    @shall_we_use_a_user_input_widget_on_top = true
    # ======================================================================= #
    # === @entry_user_input
    # ======================================================================= #
    @entry_user_input = false
    # ======================================================================= #
    # === @display_n_font_buttons_per_line
    #
    # The default value for this setting is 1. If it is higher than 1
    # then we will use more entries per line.
    # ======================================================================= #
    @display_n_font_buttons_per_line = 1
  end

  # ========================================================================= #
  # === set_display_n_font_buttons_per_line
  # ========================================================================= #
  def set_display_n_font_buttons_per_line(i = 1)
    @display_n_font_buttons_per_line = i
  end

  # ========================================================================= #
  # === set_parent_widget
  # ========================================================================= #
  def set_parent_widget(i)
    @parent_widget = i
  end

  # ========================================================================= #
  # === use_this_font_family?
  # ========================================================================= #
  def use_this_font_family?
    @use_this_font_family
  end; alias font1? use_this_font_family? # === font1?

  # ========================================================================= #
  # === use_this_font_size?
  # ========================================================================= #
  def use_this_font_size?
    @use_this_font_size
  end; alias font2? use_this_font_size? # === font2?

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === combined_font1_and_font2
  # ========================================================================= #
  def combined_font1_and_font2
    font1?.to_s+' '+font2?.to_s
  end

  # ========================================================================= #
  # === do_increment_the_font_size_by_plus_one_then_assign_to_the_proper_variable
  # ========================================================================= #
  def do_increment_the_font_size_by_plus_one_then_assign_to_the_proper_variable
    @use_this_font_size = (@use_this_font_size.to_i+1).to_s
  end

  # ========================================================================= #
  # === scrolled_window?
  # ========================================================================= #
  def scrolled_window?
    @scrolled_window
  end

  # ========================================================================= #
  # === return_the_assumed_tooltip
  # ========================================================================= #
  def return_the_assumed_tooltip
    'Change to use the '+font1?.to_s.sub(/Font /,'')+
    ' font, '+font2?.to_s+'px.'
  end

  # ========================================================================= #
  # === add_n_buttons_in_total
  # ========================================================================= #
  def add_n_buttons_in_total
    ADD_N_BUTTONS
  end

  # ========================================================================= #
  # === create_the_grid
  # ========================================================================= #
  def create_the_grid
    # ======================================================================= #
    # === @grid
    #
    # We will style it a little bit as well.
    # ======================================================================= #
    @grid = return_default_grid
    @grid.reset_the_counter
    @grid.pad4px
    @grid.mar2px
    @grid.hcenter
  end

  # ========================================================================= #
  # === create_the_vbox                                            (vbox tag)
  # ========================================================================= #
  def create_the_vbox
    # ======================================================================= #
    # === @vbox
    # ======================================================================= #
    @vbox = create_vbox
    @vbox.hcenter
  end

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
    create_the_vbox
    create_the_grid
    create_the_scrolled_window
    button_number = 0
    add_n_buttons_in_total.times {
      button_number += 1 # Keep track of the button number.
      use_this_text = combined_font1_and_font2
      button = ::Gtk::FontSizeWidget::ActionButton.new(use_this_text)
      button.bblack1
      button.hint = return_the_assumed_tooltip
      button.text_displayed_on_the_button = use_this_text
      button.on_clicked {
        do_handle_the_click_action(button.text?)
      }
      @array_buttons << button # Keep track of which buttons are in use.
      # ===================================================================== #
      # Always add it first - this will always be correct.
      # ===================================================================== #
      @grid.middle(button)
      # ===================================================================== #
      # Next, we must determine whether a new row has to be added.
      # This is only the case when:
      #
      #   button_number % @display_n_font_buttons_per_line == 0
      # 
      # ===================================================================== #
      if (button_number % @display_n_font_buttons_per_line) == 0
        @grid.add_new_row
      end
      # ===================================================================== #
      # Next increment it here:
      # ===================================================================== #
      do_increment_the_font_size_by_plus_one_then_assign_to_the_proper_variable
    }
    use_and_display_this_new_font(@use_this_font)
    set_vbox_width_and_height
  end

  # ========================================================================= #
  # === shall_we_use_a_user_input_widget_on_top?
  # ========================================================================= #
  def shall_we_use_a_user_input_widget_on_top?
    @shall_we_use_a_user_input_widget_on_top
  end

  # ========================================================================= #
  # === return_user_input_widget
  # ========================================================================= #
  def return_user_input_widget
    @entry_user_input = gtk_entry
    # ======================================================================= #
    # Next add the completion object:
    # ======================================================================= #
    entry_completion = gtk_entry_completion
    @entry_user_input.completion = entry_completion
    model = string_list_store # This is a ListStore model.
    model.attach_this_array(return_available_fonts)
    entry_completion.model = model
    entry_completion.first_text_column
    
    @entry_user_input.remove_background
    @entry_user_input.bblack2
    @entry_user_input.center
    @entry_user_input.on_enter {
      try_to_use_this_font(
        @entry_user_input.text?
      )
    }
    return @entry_user_input
  end

  # ========================================================================= #
  # === do_handle_the_click_action                                (click tag)
  #
  # This method will determine which action will happen when the button
  # is clicked. The main use for this is to sync the change back to
  # the parent widget.
  # ========================================================================= #
  def do_handle_the_click_action(
      text_displayed_on_the_button = nil
    )
    if @shall_we_use_a_user_input_widget_on_top and @entry_user_input
      new_text = @entry_user_input.text?
      unless text_displayed_on_the_button == new_text
        @entry_user_input.set_text(text_displayed_on_the_button)
      end
    end
    if @parent_widget
      @parent_widget.sync_change_to_use_this_font(text_displayed_on_the_button)
    end
    if debug?
      e 'Debugging: the text to be sent is: '+
        steelblue(text_displayed_on_the_button.to_s)
    end
  end

  # ========================================================================= #
  # === css_class_for_the_entry
  #
  # This method can be used to specifically set a CSS class for use on
  # that gtk-entry.
  # ========================================================================= #
  def css_class_for_the_entry(i)
    @entry_user_input.css_class(i)
  end

  # ========================================================================= #
  # === try_to_use_this_font
  #
  # This method is - at the least for now - the one that will be called
  # only by the gtk-entry, when the "on enter" combination is started.
  # ========================================================================= #
  def try_to_use_this_font(this_font)
    use_and_display_this_new_font(this_font)
  end

  # ========================================================================= #
  # === return_available_fonts
  #
  # This method will try to return all available fonts, on a Linux system.
  # ========================================================================= #
  def return_available_fonts
    array = `fc-list`.split("\n")
    new_array = array.select! {|entry|
      entry.include?(': ') and
      (entry.include?('share/fonts/TTF') or entry.include?('share/fonts/ttf'))
    }.map {|inner_entry|
      inner_entry = inner_entry.tr('_-',' ')
      # ========================================================================= #
      # That entry may look like this:
      #
      #   /usr/share/fonts/TTF/DejaVuSans-Bold.ttf: DejaVu Sans:style=Bold
      #
      # ========================================================================= #
      if inner_entry.include?('.ttf: ')
        splitted = inner_entry.split('.ttf: ')
      elsif inner_entry.include?('.TTF: ')
        splitted = inner_entry.split('.TTF: ')
      end
      if splitted and splitted.respond_to? :first
        first = splitted.first
        first.sub(
          Regexp.quote('/usr/share/fonts/TTF/'),
          ''
        ).sub(
          Regexp.quote('/usr/share/fonts/ttf/'),
          ''
        )
      end
    }
    new_array.compact! # Must remove nil-entries, because the .sort below would fail otherwise.
    return new_array.sort
  end

  # ========================================================================= #
  # === use_and_display_this_new_font
  # ========================================================================= #
  def use_and_display_this_new_font(
      i = 'Andale Mono 25'
    )
    # ======================================================================= #
    # === Handle Symbols first
    # ======================================================================= #
    if i.is_a? Symbol
      i = i.to_s.tr('_',' ')
      i[0,1] = i[0,1].upcase
    end
    splitted = i.to_s.split(' ')
    reset_font_size_and_font_family
    @use_this_font_size   = splitted.pop
    @use_this_font_family = splitted.join(' ')
    ADD_N_BUTTONS.times {|index|
      use_this_text = combined_font1_and_font2
      @array_buttons[index].set_text(use_this_text)
      @array_buttons[index].hint = return_the_assumed_tooltip
      @text_displayed_on_the_button = use_this_text
      @array_buttons[index].set_new_text(use_this_text)
      do_increment_the_font_size_by_plus_one_then_assign_to_the_proper_variable
    }
  end

  # ========================================================================= #
  # === reset_font_size_and_font_family
  # ========================================================================= #
  def reset_font_size_and_font_family
    # ======================================================================= #
    # === @use_this_font_family
    # ======================================================================= #
    @use_this_font_family = 'Font Hack'
    # ======================================================================= #
    # === @use_this_font_size
    # ======================================================================= #
    @use_this_font_size   = '12'
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    if shall_we_use_a_user_input_widget_on_top?
      minimal(return_user_input_widget, 2)
    end
    @vbox.maximal(@grid,      0)
    maximal(@scrolled_window, 0)
  end

  # ========================================================================= #
  # === create_the_scrolled_window
  # ========================================================================= #
  def create_the_scrolled_window
    @scrolled_window = gtk_scrolled_window(@vbox) { :only_up_and_down }
    @scrolled_window.width_height(50, 44)
  end

  # ========================================================================= #
  # === set_vbox_width_and_height
  # ========================================================================= #
  def set_vbox_width_and_height(
      width  = 200,
      height =  40
    )
    @vbox.width_height(width, height)
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::FontSizeWidget.run
  # ========================================================================= #
  def self.run(
      i = ARGV,
      &block
    )
    _ = ::Gtk::FontSizeWidget.new(i, &block)
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  shall_we_debug_this_file = false # If you want to debug this widget, set this variable to true.
  shall_we_debug_this_file = true if Gtk.is_on_roebe?
  Gtk::FontSizeWidget.run {{
    use_this_font: 'Andale Mono 35',
    enable_debug: shall_we_debug_this_file
  }}
end