#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Keyboard
# =========================================================================== #
# require 'gtk_paradise/gui/gtk3/gtk3_standalone/gtk3_standalone.rb'
# Gtk::Keyboard.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'


module Gtk

class Keyboard < ::Gtk::Box # === Gtk::Keyboard 

  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'A keyboard'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '55% or minimum 500px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '55% or minimum 400px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_20

  # ========================================================================= #
  # === FIRST_ROW_OF_THE_KEYBOARD
  # ========================================================================= #
  FIRST_ROW_OF_THE_KEYBOARD = %w(
    Esc
    F1
    F2
    F3
    F4
    F5
    F6
    F7
    F8
    F9
    F10
    F11
    F12
    Insect
    printScr
    Delete
  )

  # ========================================================================= #
  # === SECOND_ROW_OF_THE_KEYBOARD
  # ========================================================================= #
  SECOND_ROW_OF_THE_KEYBOARD = %w(
    ~
    1
    2
    3
    4
    5
    6
    7
    8
    9
    0
    -
    =
    Backspace
    Home
  )

  # ========================================================================= #
  # === THIRD_ROW_OF_THE_KEYBOARD
  # ========================================================================= #
  THIRD_ROW_OF_THE_KEYBOARD = %w(
    Tab
    Q
    W
    E
    R
    T
    Y
    U
    I
    O
    P
    {
    }
    \\
  )+['Pg Up']

  # ========================================================================= #
  # === FOURTH_ROW_OF_THE_KEYBOARD
  # ========================================================================= #
  FOURTH_ROW_OF_THE_KEYBOARD = ['Caps Lock']+%w(
    A
    S
    D
    F
    G
    H
    I
    J
    K
    L
    ;
    #
    Enter
  )+['Pg Dn']
 
  # ========================================================================= #
  # === FIFTH_ROW_OF_THE_KEYBOARD
  # ========================================================================= #
  FIFTH_ROW_OF_THE_KEYBOARD = %w(
    Shift
    Z
    X
    C
    V
    B
    N
    M
    ,
    .
    ?
    Shift
    ↑
    End
  )

  # ========================================================================= #
  # === SIXTH_ROW_OF_THE_KEYBOARD
  # ========================================================================= #
  SIXTH_ROW_OF_THE_KEYBOARD = %w(
    Ctrl
    Fn
    Win
    Alt
    space
    Alt
    Fn
    Ctrl
    ←
    ↓
    →
  )

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file
    more_CSS_then_apply_it '

.CSS_rules_for_each_button {

  color: black;
  font-weight: 500;
  border: none;
  background: rgb(
    221,
    208,
    236
  );
  box-shadow: 1px 1px 2px #fd0c99,
              1px 1px 2px #ff00dd;

}
.class_for_the_grid {
  padding: 4px;
  background: rgb(
    221,
    208,
    236
  );
  border-radius: 6px 0px 6px 6px;
}

' # width:  55px;
    infer_the_size_automatically
    @use_text_view_on_the_bottom = true
    # ======================================================================= #
    # === @text_view
    # ======================================================================= #
    @text_view = nil
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    8
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    4
  end

  # ========================================================================= #
  # === create_skeleton                            (create tag, skeleton tag)
  # ========================================================================= #
  def create_skeleton
    create_the_grid
  end

  # ========================================================================= #
  # === append_this_row
  # ========================================================================= #
  def append_this_row(i)
    i.flatten.each {|this_button|
      button_to_add = custom_button(this_button)
      case this_button
      # ===================================================================== #
      # Next add keys that are of two-width space. 
      # ===================================================================== #
      when 'Backspace',
           'Tab',
           'Enter',
           'Shift'
        @grid.two_width(button_to_add)
      # ===================================================================== #
      # Next add the space key. 
      # ===================================================================== #
      when 'Space','space'
        @grid.six_width(button_to_add)
      else
        @grid.left(button_to_add)
      end
    }
    @grid.new_row
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    outer_vbox = create_vbox
    outer_vbox.minimal(@grid)
    outer_vbox.hcenter
    if @use_text_view_on_the_bottom
      outer_vbox.minimal(return_text_view_on_the_bottom, 8)
    end
    minimal(outer_vbox, 4)
  end

  # ========================================================================= #
  # === return_text_view_on_the_bottom
  # ========================================================================= #
  def return_text_view_on_the_bottom
    text_buffer = create_text_buffer
    @text_view = create_text_view(text_buffer)
    scrolled_window = create_scrolled_window(@text_view)
    scrolled_window.bblack1
    scrolled_window.pad8px
    scrolled_window.width_height(500, 300)
    return scrolled_window
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === custom_button
  # ========================================================================= #
  def custom_button(i)
    i = i.to_s.dup
    _ = button
    _.clear_background
    _.set_text(i)
    _.lightgreen
    _.css_class('CSS_rules_for_each_button')
    _.on_clicked {
      work_on_a_copy = i.dup
      case work_on_a_copy
      when /^delete$/i
        if @text_view
          @text_view.set_text(
            @text_view.text?.chop
          )
          work_on_a_copy = nil
        end
      when 'space','Space'
        work_on_a_copy = ' '
      when 'Enter'
        work_on_a_copy = "\n"
      end
      if @text_view
        @text_view.add_this_text(work_on_a_copy) if work_on_a_copy # We won't output the text in this case.
      else
        e work_on_a_copy
      end
    }
    return _
  end

  # ========================================================================= #
  # === create_the_grid                                            (grid tag)
  # ========================================================================= #
  def create_the_grid
    @grid = default_grid
    @grid.clear_background
    @grid.set_row_spacing(4)
    @grid.set_column_spacing(4)
    @grid.css_class('class_for_the_grid')
    @grid.css_class('bblack1')
    [
      FIRST_ROW_OF_THE_KEYBOARD,
      SECOND_ROW_OF_THE_KEYBOARD,
      THIRD_ROW_OF_THE_KEYBOARD,
      FOURTH_ROW_OF_THE_KEYBOARD,
      FIFTH_ROW_OF_THE_KEYBOARD,
      SIXTH_ROW_OF_THE_KEYBOARD
    ].each {|entry| append_this_row(entry) }
  end

  # ========================================================================= #
  # === Gtk::Keyboard.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::Keyboard.new(i)
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.enable_quick_exit
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::Keyboard.run
end