#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::NotebookTabWithIcon
#
# This class represents a widget that can be put into notebooks, which
# have a little icon, typically on their left side on top.
#
# As example, you can pick among the following to do a tab:
#
#   Gtk::NotebookTabWithIcon.new # picks default
#   Gtk::NotebookTabWithIcon.new('rand')
#   Gtk::NotebookTabWithIcon.new('some/path/file.png')
#
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/notebook_tab_with_icon/notebook_tab_with_icon.rb'
# ::Gtk::NotebookTabWithIcon.new(:tabble, 'Here is the original label.', &block)
# Gtk::NotebookTabWithIcon.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class NotebookTabWithIcon < ::Gtk::BaseModuleBox # === Gtk::NotebookTabWithIcon 

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Notebook-Tab with icon'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '4% or minimum 60px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '4% or minimum 35px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_12

  # ========================================================================= #
  # === initialize
  #  
  # Four input-arguments can be provided. Their meaning is:
  #
  #   (1) location_of_the_image # The location to the image that you wish to use.
  #   (2) label_title  # The title of the label, appearing next to the image
  #   (3) text_colour  # The colour of the label
  #   (4) can_focus    # whether the user can select (focus) on it
  #
  # ========================================================================= #
  def initialize(
      location_of_the_image = :small_shield,
      label_title           = '',
      text_colour           = 'black',
      can_focus             = false,
      run_already           = true,
      &block
    )
    super(:horizontal)
    reset
    set_label_title(label_title)
    set_text_colour(text_colour)
    set_location_of_the_image(
      location_of_the_image
    )
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === Handle hashes next
      # ===================================================================== #
      if yielded.is_a? Hash
        # =================================================================== #
        # === :text_colour
        # =================================================================== #
        if yielded.has_key? :text_colour
          set_text_colour(yielded[:text_colour])
        end
        # =================================================================== #
        # === :use_close_image
        # =================================================================== #
        if yielded.has_key? :use_close_image
          use_close_image if yielded[:use_close_image]
        end
        # =================================================================== #
        # === :placement
        # =================================================================== #
        if yielded.has_key? :placement
          @placement = yielded.delete(:placement)
        end
      elsif yielded.is_a? Symbol
        case yielded
        # =================================================================== #
        # === :right_side
        # =================================================================== #
        when :right_side
          set_placement(:right)
        end
      end
    end
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file 
    infer_the_size_automatically
    # ======================================================================= #
    # === @placement
    # ======================================================================= #
    set_placement(:left)
    # ======================================================================= #
    # === @image
    # ======================================================================= #
    @image = nil
    # ======================================================================= #
    # === @background_colour
    # ======================================================================= #
    @background_colour = :very_light_grey # :lightgrey # :white
  end

  # ========================================================================= #
  # === set_label_title
  # ========================================================================= #
  def set_label_title(i)
    @label_title = i
  end

  # ========================================================================= #
  # === label_title?
  # ========================================================================= #
  def label_title?
    @label_title
  end; alias label_text? label_title? # === label_text?

  # ========================================================================= #
  # === set_placement
  # ========================================================================= #
  def set_placement(i = :right)
    @placement = i.to_sym
  end

  # ========================================================================= #
  # === placement?
  # ========================================================================= #
  def placement?
    @placement
  end

  # ========================================================================= #
  # === create_the_image
  #
  # This will assign to the variable called @image.
  # ========================================================================= #
  def create_the_image
    @image = create_image
    @image.set_padding(3, 1)
  end

  # ========================================================================= #
  # === change_location_of_the_image_and_icon
  # ========================================================================= #
  def change_location_of_the_image_and_icon(
      i = @location_of_the_image
    )
    change_location_of_the_image(i)
    change_icon(i)
  end

  # ========================================================================= #
  # === set_text_colour
  # ========================================================================= #
  def set_text_colour(i)
    @text_colour = i
  end

  # ========================================================================= #
  # === set_location_of_the_image
  # ========================================================================= #
  def set_location_of_the_image(i = :default)
    case i
    # ======================================================================= #
    # === :tabble
    # ======================================================================= #
    when :tabble,
         :default
      i = '/home/x/data/images/linux/TABBLE.png'
    # ======================================================================= #
    # === small_shield
    # ======================================================================= #
    when :small_shield
      i = project_base_directory?+
          'images/misc/small_shield.png'
    end
    @location_of_the_image = i # gi $MY_USERS/x/DATA/IMG/RPG/STD/SMALL_SHIELD.png
  end

  # ========================================================================= #
  # === update_icon
  #
  # With this you can update the icon. It will always point to @location_of_the_image
  # ========================================================================= #
  def update_icon(
      i = @location_of_the_image
    )
    if File.exist? i
      @image.set_file(i)
    else
      puts "No file exists at `#{i}`."
    end
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    4
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === set_tab_pos
  # ========================================================================= #
  def set_tab_pos(i)
    super(i)
  end

  # ========================================================================= #
  # === create_the_grid
  # ========================================================================= #
  def create_the_grid
    # ======================================================================= #
    # === @grid
    # ======================================================================= #
    @grid = default_grid
  end

  # ========================================================================= #
  # === use_close_image
  # ========================================================================= #
  def use_close_image
    change_location_of_the_image(:close_image)
  end

  # ========================================================================= #
  # === change_location_of_the_image
  #
  # With this you can manipulate @location_of_the_image
  # ========================================================================= #
  def change_location_of_the_image(
      this_location = @location_of_the_image
    )
    img = '/home/x/data/images/'
    case this_location.to_s
    when 'rand','r'
      # cd $RUBY_GTK/GAM/MINESWEEPER/images/symbols/
      img_base = ENV['RUBY_GTK'].to_s+'/games/MINESWEEPER/images/symbols/'
      case rand(7)
      when 0 then set_location_of_the_image(img_base+'bang.png')
      when 1 then set_location_of_the_image(img_base+'flag.png')
      when 2 then set_location_of_the_image(img_base+'flag-question.png')
      when 3 then set_location_of_the_image(img_base+'mine.png')
      when 4 then set_location_of_the_image(
                    '/home/x/programming/ruby/src/cyberweb/lib/cyberweb/images/standard_images/TELEFON.png'
                  )
      when 5 then change_location_of_the_image('bopen')
      when 6 then change_location_of_the_image('bclose')
      end
    # ======================================================================= #
    # === shield
    #
    # shield icon
    # ======================================================================= #
    when 'shield',
         's'
      set_location_of_the_image(project_base_directory?+'images/misc/small_shield.png')
    # ======================================================================= #
    # === close_image
    # ======================================================================= #
    when /^-?-?close(_|-)?image$/,
         /^-?-?close(_|-)?picture$/
      set_location_of_the_image(project_base_directory?+'images/icons/close_image.png')
    # ======================================================================= #
    # === alert
    #
    # Ausrufungszeichen
    # ======================================================================= #
    when 'alert',
         'ausrufungszeichen'
      set_location_of_the_image(img+'/STD/AUSRUFUNGSZEICHEN.png')
    # ======================================================================= #
    # === book_open
    # ======================================================================= #
    when /book_open/,
         /bopen/
      set_location_of_the_image(img+'/STD/BOOK_OPENED.png')
    # ======================================================================= #
    # === book_close
    # ======================================================================= #
    when /book_close/,
         /bclose/
      set_location_of_the_image(img+'/STD/BOOK_CLOSED.png')
    end
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    # ======================================================================= #
    # The placement can be towards the left, or towards the right. The
    # following code will properly distinguish between these two
    # possibilities.
    # ======================================================================= #
    case placement?
    # ======================================================================= #
    # === :left
    # ======================================================================= #
    when :left
      @grid.left(return_the_event_box)
      @grid.right(modify_label(@label_title, @text_colour))
    # ======================================================================= #
    # === :right
    # ======================================================================= #
    when :right
      @grid.left(modify_label(@label_title, @text_colour))
      @grid.right(return_the_event_box)
    else
      e 'Unknown placement.'
    end
    minimal(@grid)
  end

  # ========================================================================= #
  # === set_index
  # ========================================================================= #
  def set_index(index = 0)
    @index = index
  end

  # ========================================================================= #
  # === index?
  # ========================================================================= #
  def index?
    @index
  end

  # ========================================================================= #
  # === return_the_event_box
  #
  # When the user clicks on the event box containing our image (which may
  # be a close button image, aka a 'x') then we will quit the part there.
  # ========================================================================= #
  def return_the_event_box
    hbox = create_hbox
    hbox.css_class('BG_'+@background_colour.to_s)
    hbox.minimal(@image, 1)
    event_box = create_event_box(hbox)
    event_box.signal_connect(:event) {|widget, event|
      case event.to_s # This is <Gdk::EventType enter-notify>.
      when /EventCrossing/
        case event.name?
        when 'GDK_ENTER_NOTIFY' # on-enter event.
          hbox.remove_css_class('BG_'+@background_colour.to_s)
          hbox.css_class('BG_lightskyblue')
        when 'GDK_LEAVE_NOTIFY' # on-leave event.
          hbox.remove_css_class('BG_lightskyblue')
          hbox.css_class('BG_'+@background_colour.to_s)
        end
      end
    }
    event_box.on_clicked {
      do_perform_either_the_destroy_the_widget_action_or_call_a_parent_method
    }
    return event_box
  end

  # ========================================================================= #
  # === do_perform_either_the_destroy_the_widget_action_or_call_a_parent_method
  # ========================================================================= #
  def do_perform_either_the_destroy_the_widget_action_or_call_a_parent_method
    if parent_widget?
      parent_widget?.do_action_associated_with_the_icon_pressed(self)
    else
      destroy # Destroy the widget here.
    end
  end

  # ========================================================================= #
  # === create_skeleton                            (create tag, skeleton tag)
  # ========================================================================= #
  def create_skeleton
    create_the_image
    create_the_grid
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
    change_location_of_the_image
    update_icon
    self.can_focus = false
    show_all
  end

  # ========================================================================= #
  # === Gtk::NotebookTabWithIcon.widget
  #
  # This method will return a new instance of Gtk::NotebookTabWithIcon.
  # ========================================================================= #
  def self.widget(
      use_this_title = '',
      use_this_icon  = :tabble,
      &block
    )
    if use_this_title.is_a? Array
      use_this_title = use_this_title.join(' ')
    end
    case use_this_title
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      use_this_title = 'Here is the original label.'
    end
    _ = ::Gtk::NotebookTabWithIcon.new(
      use_this_icon,
      use_this_title,
      &block
    )
    _.make_bold
    return _
  end

  # ========================================================================= #
  # === Gtk::NotebookTabWithIcon.run
  # ========================================================================= #
  def self.run(
      i = ARGV,
      &block
    )
    require 'gtk_paradise/run'
    _ = widget(i, :tabble, &block)
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.enable_quick_exit
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  # ========================================================================= #
  # We can pass in a different colour, such as :royalblue.
  # ========================================================================= #
  Gtk::NotebookTabWithIcon.run(:default) {{
    text_colour:     :royalblue, # @text_colour = text_colour
    use_close_image: true,
    placement:       :right
  }}
end