#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::RegexMatcher
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/regex_matcher/regex_matcher.rb'
# Gtk::RegexMatcher.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class RegexMatcher < ::Gtk::BaseModuleBox # === Gtk::RegexMatcher 

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Regex Matcher'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '95% or minimum 1480px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '55% or minimum 600px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :hack_22

  # ========================================================================= #
  # === SMALLER_FONT
  # ========================================================================= #
  SMALLER_FONT = :hack_18

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical)
    reset
    horizontal_center
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    handle_CSS_rules
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === handle_CSS_rules                                            (CSS tag)
  # ========================================================================= #
  def handle_CSS_rules
    use_gtk_paradise_project_css_file
    more_CSS '
    .big_grey_border {
      border: 8px solid grey;
      border-radius: 10px;
    }

    * {
      caret-color: gold;
    }

    .main_textview {
      caret-color: black;
    }
    '
  end

  # ========================================================================= #
  # === smaller_font?
  # ========================================================================= #
  def smaller_font?
    SMALLER_FONT
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === create_skeleton                            (create tag, skeleton tag)
  # ========================================================================= #
  def create_skeleton
    create_the_entries
    create_the_buttons
    create_the_grid
    create_the_combobox
  end

  # ========================================================================= #
  # === create_the_combobox
  # ========================================================================= #
  def create_the_combobox
    @combo_box = combo_box
    [
      '\d*', '\s*', '\w*',
      '\D*', '\S*', '\W*'
    ].each { |val| 
      @combo_box.append_text(val) 
    }
    @combo_box.active = 0
  end; alias create_combo_box create_the_combobox # === create_combo_box

  # ========================================================================= #
  # === combo_box?
  # ========================================================================= #
  def combo_box?
    @combo_box
  end; alias combobox? combo_box? # === combobox?

  # ========================================================================= #
  # === create_the_grid                                            (grid tag)
  # ========================================================================= #
  def create_the_grid
    @grid = default_grid
    @grid.default_spacing(5)
  end

  # ========================================================================= #
  # === create_the_entries                           (entries tag, entry tag)
  # ========================================================================= #
  def create_the_entries
    # ======================================================================= #
    # === @entry_regex
    # ======================================================================= #
    @entry_regex = entry
    @entry_regex.set_max_length(120)
    @entry_regex.width_height(500, 40)
    @entry_regex.use_this_font = :hack_18
    @entry_regex.pad8px
    @entry_regex.css_class('BG_Black')
    @entry_regex.css_class('white')
    # @entry_regex.signal_connect(:button_press_event) { focus_entry(1) }
    # ======================================================================= #
    # === @entry_regex_options
    # ======================================================================= #
    @entry_regex_options = entry
    @entry_regex_options.set_max_length(10)
    @entry_regex_options.width_height(90, 40)
    @entry_regex_options.use_this_font = :hack_18
    @entry_regex_options.pad8px
    @entry_regex_options.css_class('BG_Black')
    @entry_regex_options.css_class('white')
  end

  # ========================================================================= #
  # === focus_entry
  #
  # This method allows you to focus on an entry.
  # ========================================================================= #
  def focus_entry(where)
    where = where.to_s
    case where
    when '1'
      @entry_regex.set_focus(true) 
    when '2'
      entry2?.set_focus(true)
    when '3'
      entry3?.set_focus(true)
    else # Else this should never happen.
      e "The (unhandled) argument was: "\
        "#{sfancy(where)}"
    end
  end

  # ========================================================================= #
  # === create_the_buttons                                      (buttons tag)
  # ========================================================================= #
  def create_the_buttons
    # ======================================================================= #
    # === @button_quit
    # ======================================================================= #
    @button_quit = button('_Quit')
    @button_quit.clear_background
    @button_quit.on_clicked { Gtk.main_quit }
    @button_quit.bblack2
    @button_quit.set_size_request(90, 40)
    @button_quit.on_hover(:lightblue)
    @button_quit.use_this_font = :hack_15
    @button_quit.disallow_resizing
    # ======================================================================= #
    # === @button_test_regex
    # ======================================================================= #
    @button_test_regex = button('_Test Regex Now')
    @button_test_regex.clear_background
    @button_test_regex.bblack2
    @button_test_regex.on_hover(:lightblue)
    @button_test_regex.use_this_font = :hack_15
    @button_test_regex.on_clicked { do_test_the_regex }
    # ======================================================================= #
    # === @button_add_special_symbol
    # ======================================================================= #
    @button_add_special_symbol = button('_Add special regex-symbol')
    @button_add_special_symbol.clear_background
    @button_add_special_symbol.bblack2
    @button_add_special_symbol.on_hover(:lightblue)
    @button_add_special_symbol.use_this_font = :hack_15
    @button_add_special_symbol.on_clicked { do_add_a_special_regex_symbol }
  end

  # ========================================================================= #
  # === return_the_regex_quick_reference_text
  # ========================================================================= #
  def return_the_regex_quick_reference_text
    regex_quick_reference_text = selectable_text('Regex quick reference')
    regex_quick_reference_text.clear_background
    regex_quick_reference_text.use_this_font = :hack_18
    regex_quick_reference_text.make_bold
    regex_quick_reference_text.css_class('steelblue')
    # regex_quick_reference_text.css_class('BG_Black')
    regex_quick_reference_text.align_to_the_left
    hbox1 = hbox.minimal(regex_quick_reference_text, 4)
    return hbox1
  end

  # ========================================================================= #
  # === entry_regex?
  # ========================================================================= #
  def entry_regex?
    @entry_regex
  end; alias entry1? entry_regex? # === entry1?

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    your_regular_expression_text = left_aligned_text('Your regular expression:')
    your_regular_expression_text.make_selectable
    your_regular_expression_text.use_this_font = smaller_font?
    @grid.left(
      your_regular_expression_text
    )
    @grid.new_line
    hbox1 = hbox
    hbox1.minimal(text('/'),            1)
    hbox1.maximal(@entry_regex,         1)
    hbox1.minimal(text('/'),            1)
    hbox1.minimal(@entry_regex_options, 1)
    hbox1.minimal(combobox?,            1)
    @grid.full_line(hbox1)
    @grid.new_line
    your_test_string_text = left_aligned_text(
      'Your test string:'
    )
    your_test_string_text.make_selectable
    your_test_string_text.use_this_font = smaller_font?
    @grid.left(
      your_test_string_text
    )
    @grid.new_line
    # ======================================================================= #
    # === @textview_for_the_string
    # ======================================================================= #
    @textview_for_the_string = textview
    @textview_for_the_string.clear_background
    @textview_for_the_string.bblack2
    @textview_for_the_string.width_height(500, 400)
    @textview_for_the_string.css_class('BG_Black')
    @textview_for_the_string.css_class('white')
    @textview_for_the_string.css_class('main_textview')
    @textview_for_the_string.set_left_margin(5)
    @textview_for_the_string.set_right_margin(5)
    @textview_for_the_string.indent = 2
    @textview_for_the_string.signal_connect(:key_press_event) {|widget, event|
      do_test_the_regex
      false
    }
    hbox1 = hbox
    hbox1.maximal(@textview_for_the_string)
    hbox1.bblack3
    hbox1.pad4px
    hbox1.css_class('BG_Black')
    hbox1.css_class('white')
    @grid.left(
      hbox1
    )
    @grid.right(
      return_match_result_box
    )
    @grid.new_line
    @grid.left(
      return_the_regex_quick_reference_text
    )
    @grid.new_line
    @grid.full_row(
      return_the_textview_containing_the_quickref
    )
    @grid.new_line
    @grid.hcenter
    minimal(@grid, 4)
    hbox1 = hbox
    hbox1.minimal(@button_add_special_symbol, 2)
    hbox1.minimal(@button_test_regex,         2)
    hbox1.minimal(@button_quit,               2)
    hbox1.hcenter
    minimal(hbox1, 2)
  end

  # ========================================================================= #
  # === textview_for_the_string_text?
  # ========================================================================= #
  def textview_for_the_string_text?
    @textview_for_the_string.text?
  end

  # ========================================================================= #
  # === entry2?
  # ========================================================================= #
  def entry2?
    @textview_for_the_string
  end

  # ========================================================================= #
  # === return_the_textview_containing_the_quickrefreturn_the_textview_containing_the_quickref
  # ========================================================================= #
  def return_the_textview_containing_the_quickref
    _ = textview
    _.set_text('

<b>[abc]</b>    A single character of: a, b, or c            <b>.</b>  Any single character                            <b>(...)</b>  Capture everything enclosed
<b>[^abc]</b>   Any single character except: a, b, or c      <b>\s</b> Any whitespace character                        <b>(a|b)</b>  a or b
<b>[a-z]</b>    Any single character in the range a-z        <b>\S</b> Any non-whitespace character                    <b>a?</b>     Zero or one of a
<b>[a-zA-Z]</b> Any single character in the range a-z or A-Z <b>\d</b> Any digit                                       <b>a*</b>     Zero or more of a
<b>^</b>        Start of line                                <b>\D</b> Any non-digit                                   <b>a+</b>     One or more of a
<b>$</b>        End of line                                  <b>\w</b> Any word character (letter, number, underscore) <b>a{3}</b>   Exactly 3 of a
<b>\A</b>       Start of string                              <b>\W</b> Any non-word character                          <b>a{3,}</b>  3 or more of a
<b>\z</b>       End of string                                <b>\b</b> Any word boundary                               <b>a{3,6}</b> Between 3 and 6 of a


    '.strip)
    _.use_this_font = :hack_13
    _.do_markify
    #_.pad8px
    hbox1 = hbox
    hbox1.maximal(_, 38)
    hbox1.css_class('big_grey_border')
    return hbox1
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === return_match_result_box
  # ========================================================================= #
  def return_match_result_box
    _ = vbox
    _.minimal(
      left_aligned_bold_text('Match result')
    )
    @textview_for_the_match_result = textview
    @textview_for_the_match_result.width_height(500, 150)
    hbox1 = hbox
    hbox1.bblack3
    hbox1.css_class('BG_Black')
    hbox1.maximal(@textview_for_the_match_result)
    _.minimal(hbox1)
    return _
  end

  # ========================================================================= #
  # === do_test_the_regex
  # ========================================================================= #
  def do_test_the_regex
    use_this_regex = Regexp.new('('+entry1?.text?+')')
    test_against_this_string = textview_for_the_string_text?
    case $1
    when nil
      set_match_result('')
    else
      set_match_result($1.to_s.dup) 
    end
  end

  # ========================================================================= #
  # === set_match_result
  # ========================================================================= #
  def set_match_result(i)
    i = i.to_s
    case i
    when ''
      i = '[No match result]'
      @textview_for_the_match_result.css_class('tomato')
    end
    @textview_for_the_match_result.set_text(i)
  end

  # ========================================================================= #
  # === entry3?
  # ========================================================================= #
  def entry3?
    @textview_for_the_match_result
  end

  # ========================================================================= #
  # === do_add_a_special_regex_symbol
  # ========================================================================= #
  def do_add_a_special_regex_symbol
    old_text = entry1?.text?
    new_symbol = combobox?.selection?
    entry1?.set_text("#{old_text}#{new_symbol}")
  end

  # ========================================================================= #
  # === Gtk::RegexMatcher.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::RegexMatcher.new(i)
    r = ::Gtk.run
    r << _
    # ======================================================================= #
    # Add two convenience keybindings next:
    # ======================================================================= #
    r.alt_key(1, 'focus_entry(1)') # alt+1
    r.alt_key(2, 'focus_entry(2)') # alt+2
    r.alt_key(3, 'focus_entry(3)') # alt+3
    r.automatic_size_then_automatic_title
    r.background(:white)
    r.enable_quick_exit
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::RegexMatcher.run
end