#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::RenameWidget
#
# This widget can be used to rename files / directories.
#
# For this to work, the file / directory has to exist. Furthermore the
# user has to be able to give the new name into a second entry.
# Last but not least, a button should be used on that second entry
# that allows the user to "commit" to the action at hand.
#
# There should also be some kind of explanation, to explain to the
# user how to use this widget.
#
# =========================================================================== #
# require 'gtk_paradise/gui/gtk3/rename_widget/rename_widget.rb'
# Gtk::RenameWidget.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class RenameWidget < ::Gtk::Box # === Gtk::RenameWidget 

  require 'fileutils'

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Rename Widget'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_18

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '80% or minimum 800px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '20% or minimum 100px'

  # ========================================================================= #
  # === BUTTON_WIDTH
  # ========================================================================= #
  BUTTON_WIDTH = 170

  # ========================================================================= #
  # === BUTTON_HEIGHT
  # ========================================================================= #
  BUTTON_HEIGHT = 58

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = ARGV,
      run_already = true
    )
    super(:vertical)
    set_my_title(TITLE) if respond_to? :set_my_title
    reset
    set_commandline_arguments(i)
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    title_width_height_font(TITLE, WIDTH, HEIGHT, :hack_22)
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === create_buttons                                          (buttons tag)
  # ========================================================================= #
  def create_buttons
    # ======================================================================= #
    # === @button_do_the_rename_action
    #
    # Create the rename-button:
    # ======================================================================= #
    @button_do_the_rename_action = button('_Rename')
    @button_do_the_rename_action.set_size_request(BUTTON_WIDTH, BUTTON_HEIGHT)
    @button_do_the_rename_action.bblack2
    @button_do_the_rename_action.hint = "Click on this button "\
      "in order to rename the file or directory chosen above. \n\n"\
      "Provide the new name in the entry field to the ← left "\
      "side of this button."
    @button_do_the_rename_action.on_clicked {
      do_the_rename_action
    }
    # ========================================================================= #
    # === @button_open_file
    # ======================================================================= #
    @button_open_file = gtk_button('_Open file')
    @button_open_file.set_size_request(BUTTON_WIDTH, BUTTON_HEIGHT)
    @button_open_file.bblack2
    @button_open_file.hint = 'Open a file that you wish to rename.'
    @button_open_file.on_clicked {
      do_handle_the_file_selection_action
    }
  end

  # ========================================================================= #
  # === create_entries                               (entries tag, entry tag)
  # ========================================================================= #
  def create_entries
    # ======================================================================= #
    # The first entry is the read-only entry.
    # ======================================================================= #
    @entry_on_top = gtk_entry
    @entry_on_top.set_text('/Depot/j/foobar.md')
    @entry_on_top.light_yellow_background
    # ======================================================================= #
    # The following entry is the one that the user can modify.
    # ======================================================================= #
    @entry_on_bottom = gtk_entry
    @entry_on_bottom.allow_enter
    @entry_on_bottom.on_enter {
      do_the_rename_action
    }
    @entry_on_bottom.light_yellow_background
    @entry_on_bottom.hint = 'This will become the new filename.'
    @entry_on_bottom.ghost = 'Input the new filename here ...'
  end

  # ========================================================================= #
  # === entry_bottom?
  # ========================================================================= #
  def entry_bottom?
    @entry_on_bottom
  end

  # ========================================================================= #
  # === entry_bottom_text?
  # ========================================================================= #
  def entry_bottom_text?
    @entry_on_bottom.text?.to_s
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_buttons
    create_entries
    @label_explanation_of_this_widget = gtk_label('The two entries below '\
      'can be used to rename a file or a directory.'
    )
  end

  # ========================================================================= #
  # === do_the_rename_action
  # ========================================================================= #
  def do_the_rename_action(
      input_file_or_directory = @entry_on_top.text.to_s
    )
    input_file_or_directory = input_file_or_directory.to_s
    # ======================================================================= #
    # === Check whether the file or directory exists (File.exist? works
    #     on directories as well)
    # ======================================================================= #
    if File.exist?(input_file_or_directory)
      new_name = @entry_on_bottom.text.to_s
      unless new_name.include? '/'
        new_name = File.absolute_path(new_name)
        @entry_on_bottom.set_text(new_name)
      end
      e 'Now renaming `'+input_file_or_directory.to_s+
        '` to `'+new_name+'`.'
      begin
        FileUtils.mv(input_file_or_directory, new_name)
      rescue Exception => error
        pp error
      end
    else
      e 'No file exists at `'+input_file_or_directory.to_s+'`'
    end
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    minimal(@label_explanation_of_this_widget, 1)
    hbox_on_top = gtk_hbox
    hbox_on_top.maximal(@entry_on_top,     5)
    hbox_on_top.minimal(@button_open_file, 5)
    minimal(hbox_on_top, 1)
    minimal(horizontal_separator, 1)
    bottom_hbox = gtk_hbox
    bottom_hbox.maximal(@entry_on_bottom,             5)
    bottom_hbox.minimal(@button_do_the_rename_action, 5)
    minimal(bottom_hbox, 1)
  end

  # ========================================================================= #
  # === do_handle_the_file_selection_action
  # ========================================================================= #
  def do_handle_the_file_selection_action
    @extra_button = gtk_button('_Extra button') 
    @extra_button.on_clicked { e ' > Extra button was clicked' }
    create_a_new_file_chooser_dialog {{
      # filter_for: '.md',
      additional_folders:
        [
          '/Depot/j/',
          ENV['DEPOT_PKG'],
          ENV['MY_TEMP'],
          ENV['MY_SONGS'],
          ENV['RUBY_SRC'],
          return_pwd,
          'C:\ingrid'
       ],
      start_dir: '/tmp/',
      show_hidden: :yes,
      extra_widget: @extra_button
    }}
    _ = main_file?.to_s.dup
    if File.exist? _
      filename = File.absolute_path(_).to_s
      @entry_on_top.set_text(filename)
      e 'The chosen file was: '+filename
      if !entry_bottom_text?.include?('/')
        entry_bottom?.set_text(
          (File.dirname(filename)+'/').squeeze('/')
        )
      end
    end
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::RenameWidget[]
  # ========================================================================= #
  def self.[](i = '')
    new
  end

  # ========================================================================= #
  # === Gtk::RenameWidgetModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = ::Gtk::RenameWidget.new(ARGV)
    r = ::Gtk.run
    r.set_size_request(_.width? / 2, _.height? / 2)
    r.signal_connect(:delete_event) { |_widget| Gtk.main_quit }
    r << _
    r.top_left_then_run
  end

  # ========================================================================= #
  # === Gtk::RenameWidgetModule.widget
  # ========================================================================= #
  def self.widget(&block)
    ::Gtk::RenameWidget.new(&block)
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::RenameWidget.run
end # gtk_rename_widget