#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Stopwatch
# =========================================================================== #
# require 'gtk_paradise/gui/gtk3/stopwatch/stopwatch.rb'
# Gtk::Stopwatch.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class Stopwatch < ::Gtk::BaseModuleBox # === Gtk::Stopwatch 

  # ========================================================================= #
  # === DEFAULT_LABEL_TEXT
  # ========================================================================= #
  DEFAULT_LABEL_TEXT = '00:00:00.0'

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Stopwatch'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '25% or minimum 240px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '25% or minimum 280px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_20

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical, 2)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file 
    infer_the_size_automatically
    # ======================================================================= #
    # === @accumulated
    # ======================================================================= #
    reset_accumulated
    # ======================================================================= #
    # === @elapsed
    # ======================================================================= #
    @elapsed = 0
    # ======================================================================= #
    # === @array_all_buttons
    # ======================================================================= #
    @array_all_buttons = []
    do_stop_the_stopwatch_counter
  end

  # ========================================================================= #
  # === Stopwatch.width?
  # ========================================================================= #
  def self.width?
    WIDTH
  end

  # ========================================================================= #
  # === Stopwatch.height?
  # ========================================================================= #
  def self.height?
    HEIGHT
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    12
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === populate_and_then_colourize_the_array_containing_the_buttons
  # ========================================================================= #
  def populate_and_then_colourize_the_array_containing_the_buttons
    @array_all_buttons << @button_start << @button_stop << @button_reset
    @array_all_buttons.each {|this_button|
      this_button.clear_background
      this_button.on_hover_lightblue
    }
  end

  # ========================================================================= #
  # === create_buttons                                          (buttons tag)
  # ========================================================================= #
  def create_buttons
    # ======================================================================= #
    # The "start the count-down" button:
    # ======================================================================= #
    @button_start = button('_Start')
    @button_start.hint = ' <b>Start</b> the stopwatch counter '
    @button_start.on_clicked {
      do_start_the_stopwatch_counter
      @button_start.toggle_grey
      @button_stop.toggle_grey
    }
    # ======================================================================= #
    # The stop-button:
    # ======================================================================= #
    @button_stop = button('S_top')
    @button_stop.hint = ' <b>Stop</b> the stopwatch counter '
    @button_stop.on_clicked {
      do_stop_the_stopwatch_counter
      @button_start.toggle_grey
      @button_stop.toggle_grey
      change_the_label_of_the_start_button_to_resume
    }
    @button_stop.grey_out
    # ======================================================================= #
    # The reset-button:
    # ======================================================================= #
    @button_reset = button('_Reset')
    @button_reset.hint = ' <b>Reset</b> the stopwatch to its '\
                         'initial (startup) state '
    @button_reset.on_clicked {
      do_reset_the_main_label
      do_stop_the_stopwatch_counter
      reset_accumulated
      initial_grey_state
      change_the_label_of_the_start_button_to_start
    }
    array_buttons = [
      @button_start, @button_stop, @button_reset
    ]
    array_buttons.each {|entry|
      entry.enable_markup
      entry.bblack2
    }
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    :tabble
  end

  # ========================================================================= #
  # === do_reset_the_main_label
  # ========================================================================= #
  def do_reset_the_main_label
    label?.set_text(return_default_label_text)
  end

  # ========================================================================= #
  # === return_default_label_text
  # ========================================================================= #
  def return_default_label_text
    DEFAULT_LABEL_TEXT
  end

  # ========================================================================= #
  # === initial_grey_state
  # ========================================================================= #
  def initial_grey_state
    @button_start.grey_in
    @button_stop.grey_out
  end

  # ========================================================================= #
  # === change_the_label_of_the_start_button_to_resume
  # ========================================================================= #
  def change_the_label_of_the_start_button_to_resume
    @button_start.set_text('R_esume')
  end

  # ========================================================================= #
  # === change_the_label_of_the_start_button_to_start
  # ========================================================================= #
  def change_the_label_of_the_start_button_to_start
    @button_start.set_text('Start')
  end

  # ========================================================================= #
  # === do_stop_the_stopwatch_counter
  # ========================================================================= #
  def do_stop_the_stopwatch_counter
    @timer_stopped = true
    @accumulated += @elapsed
  end

  # ========================================================================= #
  # === reset_accumulated
  # ========================================================================= #
  def reset_accumulated
    @accumulated = 0
  end

  # ========================================================================= #
  # === do_start_the_stopwatch_counter
  # ========================================================================= #
  def do_start_the_stopwatch_counter
    @accumulated ||= 0
    @elapsed = 0
    @start_time = Time.now # "Capture" the time here.
    @timer_stopped = false
    @timer = Thread.new {
      until @timer_stopped do
        sleep(0.1)
        tick unless @timer_stopped
      end
    }
  end

  # ========================================================================= #
  # === label?
  # ========================================================================= #
  def label?
    @label_showing_the_time
  end

  # ========================================================================= #
  # === create_labels                                 (labels tag, label tag)
  # ========================================================================= #
  def create_labels
    @label_showing_the_time = create_label(DEFAULT_LABEL_TEXT)
    @label_showing_the_time.set_size_request(400, 50)
    @label_showing_the_time.align_to_the_left
  end

  # ========================================================================= #
  # === tick
  #
  # This is the method that is called for keeping track of the time that
  # has passed, in seconds.
  # ========================================================================= #
  def tick
    @elapsed = Time.now - @start_time
    time = @accumulated + @elapsed # This will be in n seconds.
    # ======================================================================= #
    # Format the time data next:
    # ======================================================================= #
    h  = sprintf("\%02i", (time.to_i / 3600))
    m  = sprintf("\%02i", ((time.to_i % 3600) / 60))
    s  = sprintf("\%02i", (time.to_i % 60))
    mt = sprintf("\%1i", ((time - time.to_i) * 10).to_i)
    label?.set_text(h+':'+m+':'+s+'.'+mt)
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    hbox_with_buttons = create_hbox
    hbox_with_buttons.minimal(@button_start)
    hbox_with_buttons.minimal(@button_stop)
    hbox_with_buttons.minimal(@button_reset)
    vbox = create_vbox
    vbox.add(@label_showing_the_time)
    vbox.minimal(hbox_with_buttons, 8)
    # ======================================================================= #
    # === Add an expander next.
    # ======================================================================= #
    expander = gtk_expander('Stopwatch')
    expander.is_expanded
    expander.attach_to_this_widget(vbox)
    maximal(expander, 5)
  end

  # ========================================================================= #
  # === create_skeleton                            (create tag, skeleton tag)
  # ========================================================================= #
  def create_skeleton
    create_labels
    create_buttons
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton
    populate_and_then_colourize_the_array_containing_the_buttons
    connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::Stopwatch.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run/run.rb'
    _ = Gtk::Stopwatch.new(i)
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::Stopwatch.run
end