#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Tabble::ImageAndButton
#
# This widget represents an image and a button. The image can be clicked
# as well; if the user clicks on it then we will show a popover tooltip,
# if applicable. If the button is clicked then the corresponding application
# will be run.
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/tabble/image_and_button.rb'
# Gtk::Tabble::ImageAndButton
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class Tabble < ::Gtk::Box

class ImageAndButton < ::Gtk::Box # === Gtk::Tabble::ImageAndButton

  require 'gtk_paradise/widgets/gtk3/tabble/constants.rb'

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  require 'convert_global_env' rescue LoadError

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Application Image and Button'

  alias e puts

  # ========================================================================= #
  # === Gtk::ApplicationMenu::ApplicationImageAndButtonModule.try_to_require_relevant_files_from_the_rbt_project
  # ========================================================================= #
  def self.try_to_require_relevant_files_from_the_RBT_project
    unless Object.const_defined? :RBT
      %w(
        rbt/requires/require_the_installer_class.rb
        rbt/requires/require_the_cookbook_class.rb
        rbt/toplevel_methods/available_programs.rb
        rbt/cookbooks/sanitize_cookbook/class_methods.rb
      ).each {|file|
        begin
          require(file)
        rescue LoadError; end
      }
    end
  end; try_to_require_relevant_files_from_the_RBT_project # Call it at once.

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = ::Gtk::Tabble::USE_THIS_FONT

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      run_already = true
    )
    super(:vertical)
    reset
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      if yielded.is_a? Hash
        # =================================================================== #
        # === :use_this_as_image
        # =================================================================== #
        if yielded.has_key? :use_this_as_image
          set_use_this_as_image(
            yielded.delete(:use_this_as_image)
          )
        end
        # =================================================================== #
        # === :text_for_the_button
        # =================================================================== #
        if yielded.has_key? :text_for_the_button
          set_use_this_text_for_the_button(
            yielded.delete(:text_for_the_button)
          )
        end
        # =================================================================== #
        # === :command_to_run
        # =================================================================== #
        if yielded.has_key? :command_to_run
          set_command_to_run(
            yielded.delete(:command_to_run)
          )
        end
        # =================================================================== #
        # === :use_this_icon
        # =================================================================== #
        if yielded.has_key? :use_this_icon
          set_use_this_icon(yielded.delete(:use_this_icon))
        end
      end
    end
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === @title
    # ======================================================================= #
    @title  = TITLE
    # ======================================================================= #
    # === @width
    # ======================================================================= #
    set_width(WIDTH)
    # ======================================================================= #
    # === @height
    # ======================================================================= #
    set_height(HEIGHT)
    set_use_this_font(USE_THIS_FONT)
    use_gtk_paradise_project_css_file 
    infer_the_size_automatically
    # ======================================================================= #
    # === @use_this_image
    # ======================================================================= #
    @use_this_image = image_applications_system_symbolic_symbolic
    # ======================================================================= #
    # === @use_this_text_for_the_button
    # ======================================================================= #
    @use_this_text_for_the_button = 'button'
    set_command_to_run
    # ======================================================================= #
    # === @tooltip_message
    #
    # This variable can store the tooltip-message text, if we use
    # such a message. By default this is nil, meaning that we
    # will not make use of such a tooltip message.
    # ======================================================================= #
    @tooltip_message = nil
    more_CSS_then_apply_it '
    tooltip {
      padding:      8px;
      margin:       5px;
      border-width: 8px;
      border-style: solid;
      border-color: steelblue;
    }'
  end

  # ========================================================================= #
  # === set_parent_widget
  # ========================================================================= #
  def set_parent_widget(i)
    @parent_widget = i
  end

  # ========================================================================= #
  # === set_use_this_text_for_the_button
  # ========================================================================= #
  def set_use_this_text_for_the_button(i)
    @use_this_text_for_the_button = i
  end

  # ========================================================================= #
  # === set_use_this_image
  # ========================================================================= #
  def set_use_this_image(i)
    if i.is_a? Array
      i = i.join(' ').strip
    end
    if i.include?('$')
      i = replace_global_variables_in_this_string(i)
    end
    case i
    when :default, nil, ''
      i = STAR_IMAGE
    end
    @use_this_image = i
  end; alias set_use_this_as_image set_use_this_image # === set_use_this_as_image
       alias set_use_this_icon     set_use_this_image # === set_use_this_icon
       alias set_image_location    set_use_this_image # === set_image_location

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def run
    set_size_request(50, 50)
    minimal(return_the_appropriate_image, 1)
    minimal(return_the_application_button, 1)
  end

  # ========================================================================= #
  # === program_exists?
  #
  # Use this method to find out if a program exists or is installed, or 
  # available, on our given host. If the input includes a ' ' then we
  # will only check the second part.
  #
  # Usage Example:
  #
  #   program_exists? 'gimp'
  #
  # ========================================================================= #
  def program_exists?(this_program)
    original_input = this_program.dup
    does_program_exist = false # default is false
    if this_program.include? ' '
      # this_program = this_program.split(' ').first
      true
    else
      this_program = replace_global_variables_in_this_string(this_program)
      # this_program = this_program.downcase # We must be careful with .downcase.
      if File.exist? ENV['MY_SYSBIN'].to_s+'/'+this_program
        does_program_exist = true
      elsif File.exist? ENV['UBIN'].to_s+'/'+this_program
        does_program_exist = true
      elsif File.exist? this_program
        does_program_exist = true
      elsif File.exist? original_input
        does_program_exist = true
      end
    end
    return does_program_exist
  end

  # ========================================================================= #
  # === replace_global_variables_in_this_string
  # ========================================================================= #
  def replace_global_variables_in_this_string(i)
    if Object.const_defined?(:ConvertGlobalEnv)
      i = ConvertGlobalEnv[i]
    else
      i =~ /(\$.+?)\//
      _ = $1.gsub(/\$/,'').dup
      unless ENV[_].nil?
        i = i.dup if i.frozen?
        i.gsub!("$#{_}", ENV[_])
      end
    end
    return i
  end

  # ========================================================================= #
  # === return_the_appropriate_image
  # ========================================================================= #
  def return_the_appropriate_image
    image = gtk_image
    if @use_this_image.is_a?(String) and File.exist?(@use_this_image)
      image.set_from_file(@use_this_image)
    end
    image.set_padding(1, 12)
    @eventbox = gtk_eventbox(image)
    @eventbox.on_clicked {
      if @tooltip_message and !@tooltip_message.empty?
        popup_over_this_widget(eventbox?, @tooltip_message.to_s)
      end
    }
    hbox = gtk_hbox(@eventbox)
    hbox.width_height(84, 84)
    hbox.hcenter
    return hbox
  end

  # ========================================================================= #
  # === eventbox?
  # ========================================================================= #
  def eventbox?
    @eventbox
  end

  # ========================================================================= #
  # === use_this_image?
  # ========================================================================= #
  def use_this_image?
    @use_this_image
  end; alias image_location  use_this_image? # === image_location
       alias image_location? use_this_image? # === image_location?
       alias image?          use_this_image? # === image?

  # ========================================================================= #
  # === run_this_command
  # ========================================================================= #
  def run_this_command(i)
    i = i.dup
    if i.include? 'PROJECT_BASE_DIRECTORY'
      i.gsub!(
        /PROJECT_BASE_DIRECTORY/,
        ::Gtk::PROJECT_BASE_DIRECTORY
      )
    end
    if @parent_widget and 
       @parent_widget.respond_to?(:admin_panel) and
       @parent_widget.admin_panel.compile_program and # If we wish to compile it:
       is_the_RBT_project_is_available?
      begin
        # =============================================================== #
        # Instantiate a new cookbook-instance and compile-instance next.
        # =============================================================== #
        @compile = RBT::Compile.new(i.to_s.downcase)
      rescue LoadError; end
    end
    if @parent_widget and
       @parent_widget.respond_to?(:add_status_message)
      @parent_widget.add_status_message(i) # Sync it back to the parent-widget.
    end
    Opn.opn; e i
    if program_exists?(i)
      esystem i
    else
      opn; ewarn "But `#{sfancy(i)}#{swarn('` does not exist.')}"
      opn; ewarn 'Please compile or otherwise install it.'
    end
  end

  # ========================================================================= #
  # === set_command_to_run
  # ========================================================================= #
  def set_command_to_run(i = 'ls')
    @command_to_run = i
  end

  # ========================================================================= #
  # === is_the_RBT_namespace_available?
  # ========================================================================= #
  def is_the_RBT_namespace_available?
    Object.const_defined?(:RBT)
  end; alias the_RBT_project_is_available?    is_the_RBT_namespace_available? # === the_RBT_project_is_available?
       alias is_the_RBT_project_is_available? is_the_RBT_namespace_available? # === is_the_RBT_project_is_available?

  # ========================================================================= #
  # === return_the_application_button
  # ========================================================================= #
  def return_the_application_button(
      use_this_text = @use_this_text_for_the_button
    )
    @button = gtk_button(use_this_text.to_s)
    @button.set_size_request(80, 40)
    @button.width_height(280, 40)
    @button.modify_background(
      :prelight, :mediumspringgreen
    )
    if the_RBT_project_is_available? and RBT.does_include?(@use_this_text_for_the_button.to_s.downcase)
      # ===================================================================== #
      # If the RBT project is available, and the program at hand is
      # registered within the RBT project, then we will make use of a
      # tooltip for this button, by using the description available
      # from the RBT project as-is.
      #
      # The .downcase action is done to convert from e. g. 'Gwenview'
      # to 'gwenview'.
      # ===================================================================== #
      @tooltip_message = RBT.return_description_of(
        @use_this_text_for_the_button.to_s.downcase
      )
      @button.hint = " #{@tooltip_message} "
    end
    @button.set_name(
      'button_lightgreen_to_olivedrab'
    )
    @button.on_clicked {
      Thread.new {
        run_this_command @command_to_run.to_s
      }
    }
    return @button
  end

end; end; end