#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Tabble
#
# This is essentially just "a tabble clone"; see this webpage to understand
# what tabble is all about:
#
#   http://www.rillion.net/tabble/index.html
#
# Gtk::Tabble represents the main application; rather than calling it tabble
# we could also simply call it "application-menu".
#
# Tabble allows the users to start programs in a simple manner, by simply
# clicking on an icon or a button.
#
# The custom menu I use is distributed as part of the gtk_paradise project,
# since as of October 2020. Keep in mind that this custom menu will
# NOT work for everyone else, as it is adjusted to my use case(s).
#
# If you want to customize it, simply change that .yml file to your
# liking. It resides under the yaml/ subdirectory of this gem.
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/tabble/tabble.rb'
# Gtk::Tabble.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class Tabble < ::Gtk::Box # === Gtk::Tabble 

  require 'gtk_paradise/widgets/gtk3/tabble/admin_panel.rb'
  require 'gtk_paradise/widgets/gtk3/tabble/constants.rb'
  require 'gtk_paradise/widgets/gtk3/tabble/image_and_button.rb'

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Tabble - an application menu'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1540

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 760

  # ========================================================================= #
  # === RUBY_SRC
  #
  # Specify a hardcoded path to my home directory - this is, however had,
  # not as important anymore, as tabble is now part of gtk_paradise and
  # we can query the given host directory dynamically via the method
  # Gtk.project_base_dir?.
  # ========================================================================= #
  RUBY_SRC = '/home/x/programming/ruby/src/'

  # ========================================================================= #
  # === THIS_FILE_HERE
  # ========================================================================= #
  THIS_FILE_HERE =
    "#{RUBY_SRC}gtk_paradise/lib/gtk_paradise/widgets/gtk3/"\
    "tabble/tabble.rb"

  # ========================================================================= #
  # === N_ELEMENTS_PER_ROW
  # ========================================================================= #
  N_ELEMENTS_PER_ROW = 5

  # ========================================================================= #
  # === @menu_file
  #
  # This is where we can store the main menu.yml file. It is presently
  # hardcoded for my home system - may have to become more flexibly
  # in the future. See the setter-method defined in this file.
  # ========================================================================= #
  # YAML.load_file('/home/x/programming/ruby/src/gtk_paradise/lib/gtk_paradise/yaml/menu.yml')
  # ========================================================================= #
  @menu_file = "#{::Gtk::PROJECT_BASE_DIRECTORY}yaml/menu.yml"

  # ========================================================================= #
  # === Gtk::Tabble.menu_file?
  # ========================================================================= #
  def self.menu_file?
    @menu_file
  end

  # ========================================================================= #
  # === Gtk::Tabble.set_menu_file
  # ========================================================================= #
  def self.set_menu_file(i)
    @menu_file = i
  end

  # ========================================================================= #
  # === Gtk::Tabble.path_to_logo_file?
  #
  # This method will return the hardcoded path to the tabble-image, on
  # my home system.
  # ========================================================================= #
  def self.path_to_logo_file?
    "#{PROJECT_BASE_DIRECTORY}images/misc/tabble.png"
  end

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === @title
    # ======================================================================= #
    @title  = TITLE
    # ======================================================================= #
    # === @width
    # ======================================================================= #
    set_width(WIDTH)
    # ======================================================================= #
    # === @height
    # ======================================================================= #
    set_height(HEIGHT)
    set_use_this_font(USE_THIS_FONT)
    use_gtk_paradise_project_css_file # or use use_project_css_file 
    infer_the_size_automatically
    @yaml_file = Gtk::Tabble.menu_file?
    load_the_yaml_dataset
  end

  # ========================================================================= #
  # === admin_panel?
  # ========================================================================= #
  def admin_panel?
    @admin_panel
  end; alias admin_panel admin_panel? # === admin_panel

  # ========================================================================= #
  # === create_admin_panel
  # ========================================================================= #
  def create_admin_panel
    @admin_panel = ::Gtk::Tabble::AdminPanel.new
  end

  # ========================================================================= #
  # === editor?
  #
  # Denote which editor is to be used here.
  # ========================================================================= #
  def editor?
    'bluefish'
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    0
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === add_status_message
  #
  # Add a message to the StatusBar.
  # ========================================================================= #
  def add_status_message(i)
    @status_bar.push(@status_bar_context_id, i)
  end; alias add_message add_status_message # === add_message

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
    create_admin_panel
    create_the_main_notebook
    create_menu
    create_the_statusbar
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    parse_the_commandline_arguments
    create_skeleton_then_connect_skeleton
    populate_the_main_grid_as_well_as_the_notebook_with_the_correct_entries
    white_background
  end

  # ========================================================================= #
  # === path_to_logo_file?
  # ========================================================================= #
  def path_to_logo_file?
    ::Gtk::Tabble.path_to_logo_file?
  end; alias favicon? path_to_logo_file? # === favicon?

  # ========================================================================= #
  # === parse_the_commandline_arguments
  # ========================================================================= #
  def parse_the_commandline_arguments(
      i = @commandline_arguments
    )
    if i.is_a? Array
      i.each {|entry| menu(entry) } 
    else
      case i # case tag
      # ===================================================================== #
      # === --open
      #
      # rb tabble.rb --open
      # ===================================================================== #
      when /^-?-?open$/i
        open_this_file_in_the_main_editor
        exit
      end
    end
  end; alias menu parse_the_commandline_arguments # === menu

  # ========================================================================= #
  # === Gtk::Tabble.run
  #
  # This is the main run-method, if you wish to start Gtk::Tabble.
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::Tabble.new(i)
    r = ::Gtk.run
    r << _
    r.modify_background(:normal, :white)
    upto = 9
    1.upto(upto) { |t|
      r.add_shortcut(t, 'focus_tab('+t.to_s+')', :alt)
    }
    r.add_shortcut(0, 'focus_tab(10)', :alt)
    # r.fullscreen # If we want full-screen support.
    r.automatic_size_then_automatic_title
    r.top_left_then_run
  end

  require 'opn' rescue LoadError
  # ========================================================================= #
  # === open_this_file_in_the_editor
  # ========================================================================= #
  def open_this_file_in_the_editor(i)
    Opn.opn; e "Now opening #{sfile(i)} in the editor #{sfile(editor?)}."
    system "#{editor?} #{i} &"
  end

  # ========================================================================= #
  # === create_menu                                                (menu tag)
  #
  # This will create the top-bar menu.
  # ========================================================================= #
  def create_menu
    @menubar = gtk_menu_bar
    @top_menu = gtk_menu_item('_Menu')
    menu = gtk_menu
    # ========================================================================= #
    # Here we give the entry of the Menu a name.
    # ========================================================================= #
    menu_item = gtk_menu_item("open #{path_to_yaml_file?}")
    menu_item.on_button_press_event { |widget, event|
      open_this_file_in_the_editor(path_to_yaml_file?)
    }
    menu.append(menu_item)
    # ========================================================================= #
    # And next we add code that opens the .rb file here.
    # ========================================================================= #
    menu_item = gtk_menu_item("open #{THIS_FILE_HERE}")
    menu_item.on_button_press_event { |widget, event|
      open_this_file_in_the_editor(THIS_FILE_HERE)
    }
    menu.append(menu_item)
    @top_menu.set_submenu(menu)
    @menubar << @top_menu

    if ::Gtk.is_on_roebe?
      # ===================================================================== #
      # Add a debug-entry on my home system.
      # ===================================================================== #
      @top_menu2 = gtk_menu_item('_Debug')
      menu = gtk_menu
      # ===================================================================== #
      # Here we give the entry of the Menu a name.
      # ===================================================================== #
      menu_item = gtk_menu_item('debug')
      menu_item.on_button_press_event { |widget, event|
        debug?
      }
      menu.append(menu_item)
      @top_menu2.set_submenu(menu)
      @menubar << @top_menu2
    end
  end

  # ========================================================================= #
  # === path_to_yaml_file?
  # ========================================================================= #
  def path_to_yaml_file?
    ::Gtk::Tabble.menu_file?
  end; alias yaml_file? path_to_yaml_file? # === yaml_file?

  # ========================================================================= #
  # === open_this_file_in_the_main_editor
  # ========================================================================= #
  def open_this_file_in_the_main_editor(
      i = path_to_yaml_file?
    )
    begin
      require 'open'
      Open.in_editor(i)
    rescue LoadError; end
  end

  # ========================================================================= #
  # === load_the_yaml_dataset
  # ========================================================================= #
  def load_the_yaml_dataset(
      i = @yaml_file
    )
    @dataset_from_the_yaml_file = YAML.load_file(i)
    # ======================================================================= #
    # We obtain the titles, in a sorted manner, next. However had, in
    # January 2021 I decided that sorting isn't ideal - let's keep
    # the original arrangement as specified in the .yml file as-is; the
    # user can thus decide how this should appear..
    # ======================================================================= #
    @tab_titles = @dataset_from_the_yaml_file.keys
    @tab_titles.delete('fluxbox') # Delete unnecessary entries.
    @tab_titles.delete('tabble')
  end; alias read_the_dataset_from_the_yaml_file load_the_yaml_dataset # === read_the_dataset_from_the_yaml_file

  # ========================================================================= #
  # === return_the_main_grid
  # ========================================================================= #
  def return_the_main_grid
    grid = default_grid
    grid.css_class('pad8px')
    grid.row_spacing    = 8
    grid.column_spacing = 8
    grid.bblack1
    grid.set_border_width(5)
    return grid
  end

  # ========================================================================= #
  # === append_the_special_tab
  #
  # This is a special tab where we can add other custom widgets and similar
  # to the notebook tab.
  # ========================================================================= #
  def append_the_special_tab
    vbox = gtk_vbox
    # ======================================================================= #
    # === @button_play_20_simpsons
    # ======================================================================= #
    @button_play_20_simpsons = gtk_button(
      'Play 20 different simpson-movies'
    )
    @button_play_20_simpsons.make_bold
    @button_play_20_simpsons.bblack1
    @button_play_20_simpsons.hint = 'This can be used to play simpsons '\
      'files via <b>vlc</b>. Logically this only works if <b>vlc</b> '\
      'is available.'
    @button_play_20_simpsons.on_clicked {
      do_play_a_simpson_movie(1)
    }
    vbox.minimal(@button_play_20_simpsons, 4)
    @notebook.add_tab(
      gtk_button(vbox), modify_label('Special tab', 'darkblue')
    )
  end

  # ========================================================================= #
  # === do_play_a_simpson_movie
  # ========================================================================= #
  def do_play_a_simpson_movie(
      n_times                      = 2,
      base_dir_for_the_video_files = '/home/x/video/Cartoons/'
    )
    n_times.to_i.times {
      random_simpson_movie = Dir[base_dir_for_the_video_files+'**/**'].sample
      esystem 'vlc "'+random_simpson_movie+'" &'
    }
  end

  # ========================================================================= #
  # === create_the_main_notebook
  #
  # Since as of 14.06.2007 the notebook also allows drag-and-drop
  # reordering.
  # ========================================================================= #
  def create_the_main_notebook
    @notebook = default_notebook
    @notebook.do_show_the_border
    @notebook.focus_on_the_first_tab
  end

  # ========================================================================= #
  # === populate_the_main_grid_as_well_as_the_notebook_with_the_correct_entries
  # ========================================================================= #
  def populate_the_main_grid_as_well_as_the_notebook_with_the_correct_entries
    @tab_titles.each {|entry|
      hash_to_fill_up = @dataset_from_the_yaml_file[entry]

      grid = return_the_main_grid
      grid.reset_all_counters # Make sure that the counters are properly reset.
      hash_to_fill_up.each_pair.each_with_index {|(_key, inner_hash), index|
        vbox = ::Gtk::Tabble::ImageAndButton.new {{
          text_for_the_button: inner_hash['name'],
          command_to_run:      inner_hash['cmd'],
          use_this_icon:       inner_hash['img_dir']+
                               inner_hash['icon']
        }}
        vbox.set_parent_widget(self)
        case (index % N_ELEMENTS_PER_ROW)
        when 0
          grid.left(vbox)
        when (N_ELEMENTS_PER_ROW - 1)
          grid.right(vbox)
          grid.new_row
        else
          grid.middle(vbox)
        end
      }
      scrolled_window = gtk_scrolled_window(grid) { :only_up_and_down }
      scrolled_window.set_border_width(12)
      scrolled_window.width_height(880, 820)
      @notebook.add_tab(
        scrolled_window,
        entry
      )
    }
    append_the_special_tab
  end

  # ========================================================================= #
  # === focus_tab
  #
  # Use this method consistently when you wish to focus on a tab of the 
  # Notebook.
  # ========================================================================= #
  def focus_tab(i = 1)
    if i.is_a? Symbol
      case i
      # ===================================================================== #
      # === :first_tab
      # ===================================================================== #
      when :first_tab
        i = 0
      end
    else
      i = i.to_i - 1
    end
    @notebook.set_page(i)
  end

  # ========================================================================= #
  # === create_the_statusbar
  # ========================================================================= #
  def create_the_statusbar
    @status_bar = gtk_status_bar { :with_a_resize_grip }
    @status_bar_context_id = @status_bar.get_context_id('Application-Menu')
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    @vpaned_top = gtk_vpaned(@menubar, @notebook)
    # ======================================================================= #
    # Next add the bottom-components:
    # ======================================================================= #
    bottom_vbox = gtk_vbox
    bottom_vbox.minimal(@admin_panel, 0)
    bottom_vbox.minimal(@status_bar,  0)
    bigger_vpaned = gtk_vpaned(@vpaned_top, bottom_vbox)
    bigger_vpaned.position = 680
    add(bigger_vpaned)
  end

end

# =========================================================================== #
# === Gtk.start_tabble
# =========================================================================== #
def self.start_tabble(i = ARGV)
  ::Gtk::Tabble.run(i)
end

end

if __FILE__ == $PROGRAM_NAME
  Gtk.start_tabble(ARGV) 
end # rtabble