#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::CdripperModule
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/cdripper/cdripper_module.rb'
# include ::Gtk::CdripperModule
# =========================================================================== #
module Gtk

module CdripperModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  begin
    include PangoFonts # Need some font constants
  rescue Exception; end

  # ========================================================================= #
  # === ARRAY_ADD_THESE_SHORTCUT_FOLDERS
  #
  # This Array specifies some ENV variables that the user may have
  # set. It is catered to my use case, though.
  # ========================================================================= #
  ARRAY_ADD_THESE_SHORTCUT_FOLDERS = %w(
    DEP_ISO
    MY_TEMP
    MY_SONGS
    RSRC
  )

  # ========================================================================= #
  # === DEPOT_DIR
  # ========================================================================= #
  DEPOT_DIR = ENV['DEPOT'].to_s+'/'

  # ========================================================================= #
  # === RUN_SIMULATION
  # ========================================================================= #
  RUN_SIMULATION = false # if true we wont burn.

  # ========================================================================= #
  # === DEFAULT_STATUS_BAR_MESSAGE
  # ========================================================================= #
  DEFAULT_STATUS_BAR_MESSAGE = ' Not burning.'

  # ========================================================================= #
  # === USE_THIS_BURN_PROGRAM
  # ========================================================================= #
  USE_THIS_BURN_PROGRAM = 'cdrecord' # 'wodim'

  # ========================================================================= #
  # === DEFAULT_ADDON_BURN_COMMAND
  # ========================================================================= #
  DEFAULT_ADDON_BURN_COMMAND = ' dev=/dev/cdrom  -pad speed=8  -v gracetime=2  '

  # ========================================================================= #
  # === DEFAULT_BURN_COMMAND
  # ========================================================================= #
  DEFAULT_BURN_COMMAND = USE_THIS_BURN_PROGRAM + DEFAULT_ADDON_BURN_COMMAND

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE  = ' CD Ripper '

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH  = '65% or minimum 500px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '45% or minimum 400px'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      optional_file = nil,
      run_already   = true
    )
    super(title?)
    reset
    make_bold
    set_optional_file(optional_file)
    run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height(TITLE, WIDTH, HEIGHT)
    # ======================================================================= #
    # === @use_this_burn_program
    # ======================================================================= #
    @use_this_burn_program = USE_THIS_BURN_PROGRAM
    # ======================================================================= #
    # === @default_burn_command
    # ======================================================================= #
    @default_burn_command  = DEFAULT_BURN_COMMAND
    # ======================================================================= #
    # === @cd_tray_is_closed
    # ======================================================================= #
    @cd_tray_is_closed     = true # at startup, cd tray is closed.
    # ======================================================================= #
    # === @sys_cmd
    # ======================================================================= #
    @sys_cmd               = '' # stores the sys cmd for burning a CD
    # ======================================================================= #
    # === @result_of_sys_cmd
    # ======================================================================= #
    @result_of_sys_cmd     = '' # stores result of sys command
    handle_CSS_rules
    set_font_size(20)
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    8
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    4
  end

  # ========================================================================= #
  # === set_optional_file
  # ========================================================================= #
  def set_optional_file(optional_file)
    @optional_file = optional_file
  end

  # ========================================================================= #
  # === create_file_filters
  #
  # The filter is applied when the user lets popup @dialog_file_chooser.
  # ========================================================================= #
  def create_file_filters
    @file_filter = gtk_file_filter
    @file_filter.name = 'ISO Files'
    @file_filter.add_pattern('*.iso')
  end

  # ========================================================================= #
  # === add_to_status_bar
  # ========================================================================= #
  def add_to_status_bar(this_text)
    @status_bar.push(@status_bar_context_id, this_text)    
  end

  # ========================================================================= #
  # === burn_iso
  #
  # This burns the iso, thus invoke it only if you want to burn a CD.
  # ========================================================================= #
  def burn_iso
    add_to_status_bar " Burning #{@entry_this_iso.text.to_s} next..."
    @sys_cmd = @default_burn_command # see top of this file
    @sys_cmd << @entry_this_iso.text
    efancy @sys_cmd
    e 'Now please be patient ... burning'
    @result_of_sys_cmd = `#{@sys_cmd}` unless RUN_SIMULATION.true?
    e @result_of_sys_cmd
    # revert back to default again.
    @status_bar.push(@status_bar_context_id, DEFAULT_STATUS_BAR_MESSAGE)
  end

  # ========================================================================= #
  # === toggle
  #
  # This method helps toggle the CD tray status.
  # ========================================================================= #  
  def toggle(value = @cd_tray_is_closed)
    if value
      false
    else
      true
    end
  end

  # ========================================================================= #
  # === set_burn_program
  # ========================================================================= #
  def set_burn_program(
      this_program = USE_THIS_BURN_PROGRAM
    )
    case this_program
    # ======================================================================= #
    # === :cdrecord
    # ======================================================================= #
    when :cdrecord
      @use_this_burn_program = this_program.to_s
      @default_burn_command = @use_this_burn_program + DEFAULT_ADDON_BURN_COMMAND
    # ======================================================================= #
    # === :wodim
    # ======================================================================= #
    when :wodim
      @use_this_burn_program = this_program.to_s
      @default_burn_command = @use_this_burn_program + DEFAULT_ADDON_BURN_COMMAND  
    else
      e 'Error!'
    end
    add_to_status_bar(@default_burn_command)
  end

  # ========================================================================= #
  # === create_checkboxes
  # ========================================================================= #
  def create_checkboxes
    @hbox_with_checkboxes = gtk_hbox

    checkbox_cdrecord = gtk_check_button('use cdrecord')
    checkbox_wodim    = gtk_check_button('use wodim')

    case @use_this_burn_program
    when 'cdrecord'
      checkbox_cdrecord.set_active(true)
    when 'wodim'
      checkbox_wodim.set_active(true)
    end

    checkbox_cdrecord.on_toggled { |widget|
      if checkbox_cdrecord.active?
        set_burn_program(:cdrecord)
        checkbox_wodim.set_active(false)
      else
        checkbox_wodim.set_active(true)
      end
    }

    checkbox_wodim.on_toggled { |widget|
      if checkbox_wodim.active?
        set_burn_program(:wodim)
        checkbox_cdrecord.set_active(false)
      else
        checkbox_cdrecord.set_active(true)
      end
    }
    @hbox_with_checkboxes.add(checkbox_cdrecord).add(checkbox_wodim)
  end

  # ========================================================================= #
  # === create_vbox_and_pack_it
  # ========================================================================= #
  def create_vbox_and_pack_it
    @v_box = gtk_vbox
    hpaned = gtk_hpaned(@button_burn_this_iso, @button_rip_cd)
    @v_box.maximal(
      # ===================================================================== #
      # Next we will embed the three buttons on top.
      # ===================================================================== #
      create_paned_hbox(
        hpaned, @button_eject_cd
      ), 1
    )
    _ = gtk_hbox
    _.minimal(@entry_this_iso, 4)
    _.minimal(@button_clear,   4)
    @v_box.minimal(_,          5)
  end

  # ========================================================================= #
  # === rip_audio
  # ========================================================================= #
  def rip_audio
    cd "#{DEPOT_DIR}Audio" # Damit es in dieses dir gerippt wird
    e return_pwd
    sys 'cdparanoia -B'
  end

  # ========================================================================= #
  # === create_buttons                             (Buttons tag. Button tag.)
  # ========================================================================= #
  def create_buttons
    @button_quit  = gtk_button('_Quit')
    # ======================================================================= #
    # === @button_clear
    #
    # The clear-button, for erasing the old content.
    # ======================================================================= #
    @button_clear = gtk_button('_Clear')
    @button_clear.hint = ' ← <b>Clear</b> the entry to the '\
      'left by clicking on this button. '
    @image_clear  = gtk_image
    if use_gtk2?
      @image_clear.set(:'gtk-clear', ::Gtk::IconSize::BUTTON)
    end
    @button_clear.set_image(@image_clear)

    @button_clear.on_clicked {
      @entry_this_iso.set_text( '' )
    }
    @button_clear.set_border_width(0)

    @button_rip_cd = gtk_button('_Rip CD')
    @button_rip_cd.set_font('Sans Bold 18')

    # ======================================================================= #
    # The rip-CD button:
    # ======================================================================= #
    @button_rip_cd.hint = 'Use this button to <b>burn a CD</b>.'
    @button_rip_cd.on_clicked { rip_audio }
    @button_rip_cd.set_border_width(1)
    @button_rip_cd.set_size_request(26, 60)

    # ======================================================================= #
    # The eject CD button:
    # ======================================================================= #
    @button_eject_cd = gtk_button('_Eject CD-Tray')
    @button_eject_cd.on_clicked { open_or_close_cd_tray }
    @button_eject_cd.hint = '<b>Eject</b> the mounted CD'

    # ======================================================================= #
    # The select/burn iso button:
    # ======================================================================= #
    @button_burn_this_iso = gtk_button('_Select/Burn ISO')
    @button_burn_this_iso.hint = 'This button can be used '\
                                 'to burn an .iso file.'

    @button_burn_this_iso.on_clicked {
      if @entry_this_iso.text?.empty?
        # if empty, run dialog
        create_and_run_file_dialog_chooser
      else
        burn_iso
      end
    }
    @button_quit.set_border_width(1)
    @button_eject_cd.set_border_width(1)
    @button_eject_cd.set_size_request(26, 60)    
    @button_burn_this_iso.set_border_width(1)
    # ======================================================================= #
    # Gather all used buttons into an Array next:
    # ======================================================================= #
    @array_buttons = [
      @button_quit,
      @button_eject_cd,
      @button_burn_this_iso,
      @button_clear,
      @button_rip_cd
    ]
    @array_buttons.each {|this_button|
      this_button.bblack2
    }
  end

  # ========================================================================= #
  # === create_statusbar
  # ========================================================================= #
  def create_statusbar
    @status_bar = gtk_statusbar
    @status_bar.has_resize_grip = true if @status_bar.respond_to? :has_resize_grip
    @status_bar_context_id = @status_bar.get_context_id('id_burning')
    @status_bar.push(@status_bar_context_id, DEFAULT_STATUS_BAR_MESSAGE)
    @status_bar_in_event_box = gtk_event_box.add(@status_bar)
    @status_bar_in_event_box.on_button_press_event { |widget, event|
      e 'The status-bar was clicked.'
    }
  end

  # ========================================================================= #
  # === Cdripper.width?
  # ========================================================================= #
  def self.width?
    WIDTH
  end

  # ========================================================================= #
  # === Cdripper.height?
  # ========================================================================= #
  def self.height?
    HEIGHT
  end

  # ========================================================================= #
  # === create_and_run_file_dialog_chooser
  #
  # Will always create file filters.
  # ========================================================================= #
  def create_and_run_file_dialog_chooser
    @dialog_file_chooser = standard_file_chooser_dialog(
      'Select This .iso File', gtk_window
    )
    create_file_filters # also creates @file_filter
    @dialog_file_chooser.add_filter(@file_filter)
    ARRAY_ADD_THESE_SHORTCUT_FOLDERS.each {|entry|
      if ENV.has_key? entry
        entry = ENV[entry]
        @dialog_file_chooser.add_shortcut_folder(entry)
      else
        e 'No ENV entry for '+::Colours.sfancy(entry)+' is known.'
      end
    }
    # ======================================================================= #
    # Next we will add the DEPOT_ISO environment variable. However had,
    # not every user may have this set, so we must use a conditional
    # check here.
    # ======================================================================= #
    if ENV.has_key? 'DEPOT_ISO'
      @dialog_file_chooser.current_folder = ENV['DEPOT_ISO']
    end

    #if use_gtk2?
      accept_type = ::Gtk::Dialog::RESPONSE_ACCEPT
    #else
    #  accept_type = :accept
    #end
    result = @dialog_file_chooser.run
    case result
    when accept_type
      e
      e "Selected file: #{@dialog_file_chooser.filename}"
      e
      @entry_this_iso.set_text(@dialog_file_chooser.filename)
    end
    @dialog_file_chooser.destroy  
  end

  # ========================================================================= #
  # === open_or_close_cd_tray
  #
  # ejects or closes CD Tray.
  # ========================================================================= #
  def open_or_close_cd_tray
    if @cd_tray_is_closed
      _ = `eject` # open it
      button_eject_cd?.set_text('C_lose CD-Tray')
    else
      _ = `eject -t` # close it
      button_eject_cd?.set_text('_Eject CD-Tray')
    end
    @cd_tray_is_closed = toggle 
  end

  # ========================================================================= #
  # === button_eject_cd?
  # ========================================================================= #
  def button_eject_cd?
    @button_eject_cd
  end

  # ========================================================================= #
  # === label_eject_cd?
  # ========================================================================= #
  def label_eject_cd?
    @button_eject_cd.label
  end

  # ========================================================================= #
  # === connect_skeleton
  #
  # Connect the skeleton of the widgets.
  # ========================================================================= #
  def connect_skeleton
    modify_background(:normal, :mediumaquamarine)
    @v_box.maximal(@hbox_with_checkboxes,    2)
    @v_box.maximal(@button_quit,             2)
    @v_box.maximal(@status_bar_in_event_box, 2)
    @v_box.show_all
    add(@v_box)
    show_all
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    :audio_favicon
  end

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
    create_entries
    create_buttons
    create_statusbar
    create_vbox_and_pack_it
    create_checkboxes
  end

  # ========================================================================= #
  # === start_the_application
  # ========================================================================= #
  def start_the_application
    @entry_this_iso.set_text(@optional_file) if @optional_file
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
    start_the_application
  end

  # ========================================================================= #
  # === create_entries                               (entries tag, entry tag)
  # ========================================================================= #
  def create_entries
    # ======================================================================= #
    # === @entry_this_iso
    # ======================================================================= #
    @entry_this_iso = gtk_entry
    # @entry_this_iso.signal_connect( :activate ) { |entry| enter_callback(entry) }
    @entry_this_iso.on_activate { |entry| enter_callback(entry) }
    @entry_this_iso.width_height(500, 22)
    @entry_this_iso.pad8px
    @entry_this_iso.bblack2
    @entry_this_iso.light_yellowish_background
    @entry_this_iso.set_name('entry1')
  end

  # ========================================================================= #
  # === handle_CSS_rules
  # ========================================================================= #
  def handle_CSS_rules
    use_gtk_paradise_project_css_file
  end

  # ========================================================================= #
  # === GtkParadise::GUI::Gtk::CdripperModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = Gtk::Cdripper.new(ARGV.first)
    r = ::Gtk.run
    r << _
    r.automatic_size
    r.top_left_then_run
  end

end; end