#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ColourPaletteModule
#
# If you need to obtain the current active colour then you can use the
# method .current_colour?().
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/colour_palette/colour_palette_module.rb'
# include ::Gtk::ColourPaletteModule
# =========================================================================== #
module Gtk

module ColourPaletteModule # === Gtk::ColourPaletteModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Colour Palette'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '4% or minimum 60px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '5%'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :hack_16 # dejavu_condensed_20

  # ========================================================================= #
  # === SMALLER_FONT
  # ========================================================================= #
  SMALLER_FONT = :hack_12

  # ========================================================================= #
  # === SMALLEST_FONT
  # ========================================================================= #
  SMALLEST_FONT = :hack_10

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      run_already = true
    )
    super(:vertical)
    reset
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :do_not_run_yet
      # ===================================================================== #
      when :do_not_run_yet
        run_already = false
      else
        # =================================================================== #
        # === Handle Hashes next
        # =================================================================== #
        if yielded.is_a? Hash
          # ================================================================= #
          # === :dimension
          # ================================================================= #
          if yielded.has_key? :dimension
            set_dimension(yielded[:dimension])
          end
        end
      end
    end
    case run_already
    when :do_not_run_yet
      run_already = false
    end
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height(TITLE, WIDTH, HEIGHT)
    # ======================================================================= #
    # === @parent_widget
    #
    # This widget can be used to call back actions onto the parent widget,
    # if non-nil.
    # ======================================================================= #
    @parent_widget = nil
    # ======================================================================= #
    # === @array_use_these_colours
    # ======================================================================= #
    @array_use_these_colours = []
    # ======================================================================= #
    # === @n_times_x_axis
    # ======================================================================= #
    @n_times_x_axis = 5
    @n_times_y_axis = 5
    # ======================================================================= #
    # === @output_the_colour_on_toggled_event
    # ======================================================================= #
    @output_the_colour_on_toggled_event = false
    set_use_this_font(USE_THIS_FONT)
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    1
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
    create_buttons
  end

  # ========================================================================= #
  # === untoggle_all_widgets_except_this_one
  # ========================================================================= #
  def untoggle_all_widgets_except_this_one(widget)
    return_all_button_boxes.each {|inner_children|
      inner_children.each {|a_widget|
        unless a_widget == widget
          a_widget.set_active(false)
        end
      }
    }
  end

  # ========================================================================= #
  # === return_all_button_boxes
  # ========================================================================= #
  def return_all_button_boxes
    children?.select {|entry| entry.is_a?(Gtk::ButtonBox)}
  end

  # ========================================================================= #
  # === current_colour?
  #
  # This method can return the active colour of the colour-palette.
  # ========================================================================= #
  def current_colour?
    children?.each {|inner_children|
      inner_children.each {|a_widget|
        if a_widget.active?
          _ = a_widget.background_colour?
          return _
        end
      }
    }
    return nil # Or nil.
  end; alias active_colour? current_colour? # === active_colour?

  # ========================================================================= #
  # === parent_widget?
  # ========================================================================= #
  def parent_widget?
    @parent_widget
  end

  # ========================================================================= #
  # === do_output_the_colour_on_toggled_event
  # ========================================================================= #
  def do_output_the_colour_on_toggled_event
    @output_the_colour_on_toggled_event = true
  end

  # ========================================================================= #
  # === select_twentyfive_random_colours
  # ========================================================================= #
  def select_twentyfive_random_colours(n_times = 25)
    @array_use_these_colours.clear
    all_html_colours = ::Colours.html_colours
    n_times.times {
      all_html_colours.shuffle!
      _ = all_html_colours.shift
      @array_use_these_colours << _
    }
  end

  # ========================================================================= #
  # === set_dimension
  #
  # Here you can determine how many boxes should appear. For a small widget
  # you could use something like '3x3', aka 3 buttons per row. For a larger
  # widget higher values may be useful.
  # ========================================================================= #
  def set_dimension(i = '3x3')
    if i.is_a?(String) and i.include?('x')
      splitted = i.split('x')
      @n_times_x_axis = splitted.first.to_i
      @n_times_y_axis = splitted.last.to_i
    end
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    select_twentyfive_random_colours
    index = 0
    @n_times_y_axis.times {
      # ===================================================================== #
      # Create the main button box next.
      # ===================================================================== #
      button_box = button_box(:horizontal)
      button_box.spacing = 1
      button_box.layout = :center # :expand
      button_box.disallow_resizing
      @n_times_x_axis.times {
        button_toggle = gtk_toggle_button
        button_toggle.disallow_resizing
        button_toggle.clear_background
        button_toggle.set_border_width(0)
        button_toggle.width_height(12, 12)
        button_toggle.use_these_css_classes('no_background_image bblack2')
        button_toggle.on_toggled { |widget|
          if button_toggle.active?
            button_toggle.remove_css_class('bblack2')
            button_toggle.css_class('btomato2')
            # ================================================================= #
            # Careful - we must avoid infinite loops.
            # ================================================================= #
            untoggle_all_widgets_except_this_one(button_toggle)
            if parent_widget? and parent_widget?.respond_to?(:set_colour)
              parent_widget?.set_colour(active_colour?)
            end
            e current_colour? if @output_the_colour_on_toggled_event
          elsif not button_toggle.active?
            button_toggle.remove_css_class('btomato2')
            button_toggle.bblack2
          end
        }
        # =================================================================== #
        # Keep track of the colour in use next.
        # =================================================================== #
        button_toggle.background_colour(
          @array_use_these_colours[index]
        )
        button_box.pack_start(
          button_toggle, fill: false, expand: false, padding: 0
        )
        index +=1
      }
      pack_start(button_box, fill: false, expand: false, padding: 1)
    }
    minimal(@button_shuffle_the_colour_palette, 0)
  end

  # ========================================================================= #
  # === create_buttons                              (buttons tag, button tag)
  # ========================================================================= #
  def create_buttons
    # ======================================================================= #
    # === @button_shuffle_the_colour_palette
    # ======================================================================= #
    @button_shuffle_the_colour_palette = button(
      'Shuffle the colour palette'
    )
    @button_shuffle_the_colour_palette.disallow_resizing
    @button_shuffle_the_colour_palette.on_hover(:lightblue)
    @button_shuffle_the_colour_palette.make_bold
    @button_shuffle_the_colour_palette.hint =
      'Clicking on this button will reshuffle (change) the '\
      'above colour palette.'
    @button_shuffle_the_colour_palette.clear_background
    @button_shuffle_the_colour_palette.set_size_request(32, 15)
    @button_shuffle_the_colour_palette.css_class('no_background_image')
    @button_shuffle_the_colour_palette.bblack2
    @button_shuffle_the_colour_palette.use_this_font = SMALLEST_FONT
    @button_shuffle_the_colour_palette.on_clicked {
      do_shuffle_the_colour_palette
    }
  end

  # ========================================================================= #
  # === do_shuffle_the_colour_palette
  #
  # This method will shuffle (change) the colour-palette in use.
  # ========================================================================= #
  def do_shuffle_the_colour_palette
    # ======================================================================= #
    # First re-assign @array_use_these_colours.
    # ======================================================================= #
    select_twentyfive_random_colours
    toggle_buttons?.each_with_index {|this_toggle_button, index|
      this_toggle_button.background_colour(
        @array_use_these_colours[index]
      )
    }
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === Gtk::ColourPaletteModule.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::ColourPalette.new(i)
    _.do_output_the_colour_on_toggled_event
    r = ::Gtk.run
    r << _
    #r.set_size_request(_.width?, _.height?)
    r.infer_the_title_from_the_child_widget
    r.top_left_then_run
  end

end; end