#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ConkyModule
#
# This shall be a clone of conky eventually.
#
# For now I just copy/paste what I see on the www.
#
# Examples that seem interesting:
#
#   https://i0.wp.com/itsfoss.com/wp-content/uploads/2013/07/default-conky.jpg?resize=800%2C400&ssl=1
#   https://i.imgur.com/6ymEF7D.png
#
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/conky/conky_module.rb'
# include ::Gtk::ConkyModule
# =========================================================================== #
module Gtk

module ConkyModule # For a gtk-vbox.

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Conky'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1400

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 520

  # ========================================================================= #
  # === ONE_HOUR
  # ========================================================================= #
  ONE_HOUR = 60 * 60 # in n seconds

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :hack_18 # :dejavu_condensed_20

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      optional_file = nil,
      run_already   = true
    )
    super(:vertical)
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
  end

  # ========================================================================= #
  # === add_update_information
  # ========================================================================= #
  def add_update_information
    # ======================================================================= #
    # The outcome of `uptime` would be like this:
    #
    #   "18:49:08  up 12 days 13:10,  14 users,  load average: 1.37, 1.21, 1.20"
    #
    # ======================================================================= #
    _ = `uptime`.split(',').first.to_s.split(' ')[1 .. -1].
        join(' ').strip.dup.sub(/up/,'').strip
    n_seconds = _.split(' ').first.to_i * 24 * 60 * 60 # "12 days 13:12"
    n_hours = 0
    loop {
      if n_seconds >= ONE_HOUR
        n_seconds -= ONE_HOUR
        n_hours += 1
      else
        break
      end 
    }
    # ======================================================================= #
    # The display is not 100% correct for now; I just wanted to get a
    # rough estimate for it.
    # ======================================================================= #
    label = text(
      'Uptime:          <b>'+n_hours.to_s+'h 0m 0s</b>'
    )
    label.set_name('darkblue')
    label.left_align
    add(label)
  end

  # ========================================================================= #
  # === add_processor_information
  # ========================================================================= #
  def add_processor_information
    _ = `lscpu`.split(N).select {|entry| entry.include? 'Architecture:' }
    _ = _.first.to_s.split(':').last.strip
    label = text_left_aligned(
      'Processor type: <b> '+_+' </b> '
    )
    label.set_name('darkblue')
    hbox = gtk_hbox
    hbox.minimal(label)
    label.set_size_request(200, 50)
    add(hbox)
  end

  # ========================================================================= #
  # === add_RAM_usage
  # ========================================================================= #
  def add_RAM_usage
    result = `free -h`.split(N)[1]
    # ======================================================================= #
    # This may look like this:
    #
    #   "Mem:           14Gi       4.4Gi       2.1Gi       106Mi       8.1Gi       9.8Gi"
    #
    # ======================================================================= #
    splitted = result.split(' ')
    lower_bound = splitted[2].to_s.delete('i').to_f
    upper_bound = splitted[1].to_s.delete('i').to_f

    label = text_left_aligned(
      'RAM usage:       <b>'+
      lower_bound.to_s+' GB / '+
      upper_bound.to_s+
      ' GB </b> '
    )
    label.set_name('darkblue')
    hbox = gtk_hbox
    hbox.minimal(label)
    add(hbox)
  end

  # ========================================================================= #
  # === return_cpu_usage
  #
  # We will look for this line:
  #
  #   "%Cpu(s): 12.7 us,  3.2 sy,  0.0 ni, 84.1 id,  0.0 wa,  0.0 hi,  0.0 si,  0.0 st"
  #
  # ========================================================================= #
  def return_cpu_usage
    result = `top -b -n 1`.split("\n")
    line = result.select {|entry| entry.start_with? '%Cpu' }
    if line and line.is_a?(Array)
      line = line.first
    end
    if line.include? '%Cpu('
      line = line.split(':')[1].strip
      if line.include? ' us'
        line = line.split(' us').first
      end
    end
    return line.strip  
  end

  # ========================================================================= #
  # === add_filesystem_information
  #
  # This entry point will add filesystem information to the conky-widget.
  # ========================================================================= #
  def add_filesystem_information
    _ = `df -h`.split(N).select {|entry|
      entry.strip.end_with?('/')
    }.first.to_s
    # _ is now something like: "/dev/sda3       2.7T  445G  2.2T  18% /"
    splitted = _.split(' ')
    mounted_on = splitted.last
    total_size = splitted[1]
    occupied   = splitted[2]
    label = text_left_aligned(
      'File system:     mounted on <b>'+
      mounted_on+
      '</b>'
    )
    label.set_name('darkblue')
    add(label)
    label = text_left_aligned(
      '                     Total size: <b>'+total_size+'</b>'+
      ' Occupied: <b>'+occupied.to_s+'</b>'
    )
    label.set_name('darkblue')
    hbox = create_hbox
    hbox.minimal(label)

    coloured_bar_widget = gtk_coloured_bar
    coloured_bar_widget.set_colour1(:royalblue)
    coloured_bar_widget.set_colour2(:forestgreen)
    if total_size.is_a?(String) and total_size.end_with?('T')
      total_size.delete!('T')
      total_size = total_size.to_f * 1000 # e. g. '2.7T'
    end
    _ = (
      (occupied.to_f * 100.0) / total_size.to_f
    ).round(2)
    coloured_bar_widget.set_percentage_value(_)
    coloured_bar_widget.set_width(150)
    coloured_bar_widget.set_height(15)
    coloured_bar_widget.set_size_request(150, 15)
    coloured_bar_widget.do_rescale
    hbox.minimal(coloured_bar_widget, 10)
    add(hbox)
  end

  # ========================================================================= #
  # === add_CPU_usage
  # ========================================================================= #
  def add_CPU_usage
    # ======================================================================= #
    # === CPU usage
    # ======================================================================= #
    cpu_value = return_cpu_usage.to_s
    label_cpu_usage = gtk_text_left_aligned(
      'CPU usage: <b>'+cpu_value+'%</b> '
    )
    label_cpu_usage.set_name('darkblue')
    hbox = gtk_hbox
    hbox.minimal(label_cpu_usage)
    coloured_bar_widget = gtk_coloured_bar
    coloured_bar_widget.set_colour1(:darkblue)
    coloured_bar_widget.set_colour2(:azure)
    coloured_bar_widget.set_percentage_value(cpu_value)
    coloured_bar_widget.set_width(150)
    coloured_bar_widget.set_height(15)
    coloured_bar_widget.set_size_request(150, 15)
    coloured_bar_widget.do_rescale
    hbox.minimal(coloured_bar_widget, 10)
    add(hbox)
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    if use_gtk3?
      _ = return_pixbuf_based_on_the_default_icon_theme(:image_computer)
      # ======================================================================= #
      # Scale the image up a bit.
      # ======================================================================= #
      _ = _.scale_simple(60, 60, :bilinear)
      _ = gtk_image_from_pixbuf(_)
      _.align_left
      hbox = create_hbox
      hbox.minimal(_, 5)
      require 'gtk_paradise/widgets/gtk3/mini_clock/mini_clock.rb'
      hbox.minimal(image_emblem_urgent, 8)
      hbox.minimal(::Gtk::MiniClock.new { :do_not_show_the_seconds })
      add(hbox) # Add the computer image here.
      add_horizontal_spacer
    end
    add_processor_information
    add_update_information
    add_RAM_usage
    add_filesystem_information
    add_CPU_usage
    add_number_of_processes_currently_running
  end

  # ========================================================================= #
  # === add_number_of_processes_currently_running
  # ========================================================================= #
  def add_number_of_processes_currently_running
    hbox = gtk_hbox
    @label_number_of_processes_currently_running = create_label
    @label_number_of_processes_currently_running.set_name('darkblue')
    @label_number_of_processes_currently_running.left_align
    @label_number_of_processes_currently_running.set_text(
      'Number of processes currently running: <b>'+
      return_the_number_of_processes_currently_running+'</b>'
    )
    @label_number_of_processes_currently_running.do_markify
    hbox.minimal(@label_number_of_processes_currently_running)
    add(hbox)
    GLib::Timeout.add(5_000) {
      do_update_the_label_number_of_processes_currently_running
    }
  end

  # ========================================================================= #
  # === do_update_the_label_number_of_processes_currently_running
  # ========================================================================= #
  def do_update_the_label_number_of_processes_currently_running
    @label_number_of_processes_currently_running.set_text(
      'Number of processes currently running: <b>'+
      return_the_number_of_processes_currently_running+
      '</b>'
    )
    @label_number_of_processes_currently_running.do_markify
  end

  # ========================================================================= #
  # === return_the_number_of_processes_currently_running
  # ========================================================================= #
  def return_the_number_of_processes_currently_running
    `ps -e | wc -l`.to_s.strip
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === Gtk::ConkyModule.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::Conky.new(i)
    r = ::Gtk.run
    r << _
    r.automatic_title_and_automatic_size
    r.top_left_then_run
  end

end; end