#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ContactsModule
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/contacts/contacts_module.rb'
# include ::Gtk::ContactsModule
# =========================================================================== #
module Gtk

module ContactsModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  begin
    require 'cliner'
  rescue LoadError; end

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === N
  # ========================================================================= #
  N = "\n"

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Contacts'

  # ========================================================================= #
  # === CONTACTS_FILE
  #
  # This file is hardcoded, defaulting to my home setup.
  #
  # Here the contact information from:
  #
  #   bl $PERSONAL_YAML/yaml/contacts.yml
  #
  # ========================================================================= #
  CONTACTS_FILE =
    '/home/x/data/personal/yaml/contacts.yml'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :hack_22 # :dejavu_condensed_20

  # ========================================================================= #
  # === DEBUG
  # ========================================================================= #
  DEBUG = false

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH  = '80% or minimum 1640px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 680

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      run_already = true
    )
    super(:vertical)
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    # ======================================================================= #
    # === @array_contact_info
    # ======================================================================= #
    @array_contact_info = []
    load_the_contacts_file
    use_gtk_paradise_project_css_file
    handle_ad_hoc_CSS_rules
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === handle_ad_hoc_CSS_rules
  # ========================================================================= #
  def handle_ad_hoc_CSS_rules
    more_CSS_then_apply_it '

.css_for_the_main_scrolled_window {
  border: 10px solid darkblue;
  border-radius: 14px;
  padding: 2px;
  margin:  2px;
}
'
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    0
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === load_the_contacts_file
  # ========================================================================= #
  def load_the_contacts_file(
      i = CONTACTS_FILE
    )
    if File.exist? i
      @array_contact_info = YAML.load_file(i)
    elsif is_on_roebe?
      e "No file exists at `#{i}`."
    end
  end

  # ========================================================================= #
  # === create_cell_renderer
  # ========================================================================= #
  def create_cell_renderer
    @renderer = gtk_cell_renderer_text
    @renderer.editable = true
    @renderer.set_property('background', 'white')
    @renderer.set_cell_background('slateblue') # This makes the blue border.
  end

  # ========================================================================= #
  # === create_and_fill_list_store
  #
  # Create our @list_store data here in this method.
  # ========================================================================= #
  def create_and_fill_list_store
    @list_store = ::Gtk::ListStore.new(
      *[String] * 6
    )
    @array_contact_info.each { |entry|
      show_debug_information(entry) # Show debug information here.
      iter = @list_store.append
      iter.set_value(0, entry[0])
      iter.set_value(1, entry[1].to_s) unless entry[1].nil?
      iter.set_value(2, entry[2].to_s) unless entry[2].nil?
      iter.set_value(3, entry[3].to_s) unless entry[3].nil?
      # ===================================================================== #
      # Entry number 4 is the "Wohnort", but we may have done this
      # incorrectly for some entries.
      # ===================================================================== #
      unless entry[4].nil?
        wohnort = entry[4]
        if wohnort.is_a? Hash
          if wohnort.has_key? 'phone'
            iter.set_value(1, wohnort.delete('phone'))
          end
        end
        wohnort = wohnort.to_s
        iter.set_value(4, wohnort) 
      end
      _ = entry[5].to_s
      _ = '---' if _.empty?        
      iter.set_value(5, _)
    }
  end

  # ========================================================================= #
  # === show_debug_information
  # ========================================================================= #
  def show_debug_information(e)
    pp e if DEBUG
  end; alias debug show_debug_information # === debug

  # ========================================================================= #
  # === create_note_viewer_at_bottom
  # ========================================================================= #
  def create_note_viewer_at_bottom
    @text_buffer = gtk_text_buffer
    @text_buffer.set_text('')
    @text_view_at_bottom = gtk_text_view(@text_buffer)
    @text_view_at_bottom.set_editable(true)
  end

  # ========================================================================= #
  # === create_scrolled_window
  # ========================================================================= #
  def create_scrolled_window
    widget = gtk_resize_grip(@tree_view, @text_view_at_bottom)
    @scrolled_window = gtk_scrolled_window(widget) { :automatic }
    @scrolled_window.css_class('css_for_the_main_scrolled_window')
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_and_fill_list_store
    create_cell_renderer
    create_tree_view
    create_note_viewer_at_bottom
    create_scrolled_window
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    maximal(@scrolled_window, 4)
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === Gtk::ContactsModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = ::Gtk::Contacts.new
    r = ::Gtk.run
    r << _
    r.modify_background :normal, :white
    r.automatic
    r.top_left_then_run
  end

  # ========================================================================= #
  # === create_tree_view
  #
  # Our tree view.
  # ========================================================================= #
  def create_tree_view
    @tree_view = gtk_tree_view(@list_store)
    @tree_view.set_name('treeview1')
    @tree_view << ['Name',           @renderer, text: 0]
    @tree_view << ['Telefon Nummer', @renderer, text: 1]
    ::Gtk.custom_treeview_widget( # This will be inserted in an absolute way.
      @tree_view, 1, :phone, 'Telefon Nummer'
    )
    @tree_view << ['Email',          @renderer, text: 2]
    @tree_view << ['Geburtstag',     @renderer, text: 3]
    @tree_view << ['Wohnort',        @renderer, text: 4]
    @tree_view << ['ICQ Nummer',     @renderer, text: 5]
    @tree_view.hadjustment = ::Gtk::Adjustment.new(0.0, 0.0, 105.0, 0.2, 1.0, 1.0)
    @tree_view.clickable_headers
    @tree_view.enable_drag_and_drop # Enable drag and drop support of the treeview.
    @tree_view.on_button_press_event { |widget, event|
      # ===================================================================== #
      # Show some information if the user clicks on this.
      # ===================================================================== #
      if use_gtk2?
        button_press_event = Gdk::Event::BUTTON2_PRESS
      else
        button_press_event = Gdk::EventType::BUTTON2_PRESS
      end
      if (event.event_type == button_press_event) and (event.button == 1) # event.event_type.name
        Cliner.cliner
        s = @tree_view.selection.selected
        string = ''.dup
        string << s[0]+N if s[0]
        string << s[1]+N if s[1]
        string << s[2]+N if s[2]
        string << s[3]+N if s[3]
        string << s[4]+N if s[4]
        string << s[5]+N if s[5]
        e string
        Cliner.cliner
        e
      end
    }
    @tree_view.on_key_press_event { |widget, event|
      case Gdk::Keyval.to_name(event.keyval)
      when 'Return','KP_Enter' # enter drücken
        e 'ENTER DR*CT - brauch ma des?'
      end
    }
    @tree_view.on_button_press_event { |widget, event|
      if ::Gtk.use_gtk2?
        this_gdk_event = Gdk::Event::BUTTON_PRESS
      else
        this_gdk_event = :button_press
      end
      if event.event_type == this_gdk_event # normaler mouse button
        @tree_view.selection.unselect_all if event.button == 3
      end
    }
    @tree_view.the_header_can_be_dragged
    columns = @tree_view.columns?
    handle_sorting(columns)
  end

  # ========================================================================= #
  # === handle_sorting
  # ========================================================================= #
  def handle_sorting(
      columns = @tree_view.columns?
    )
    # ======================================================================= #
    # Enable sorting for some of the entries next.
    # ======================================================================= #
    columns[0].sortable_based_on(0)
    columns[2].sortable_based_on(2) # This sorts based on the email.
    columns[3].sortable_based_on(3) # This sorts based on the date-of-birth.
    columns[4].sortable_based_on(4) # This sorts based on the "Wohnort" aka location.
  end

end; end