#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::FileViewerModule
#
# This module can be used as a simpler file-viewer widget.
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/file_viewer/file_viewer_module.rb'
# include ::Gtk::FileViewerModule
# =========================================================================== #
module Gtk

module FileViewerModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  require 'gtk_paradise/toplevel_methods/toplevel_methods.rb'

  begin
    require 'opn'
  rescue LoadError; end

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'File Viewer'

  # ========================================================================= #
  # === LIMIT
  # ========================================================================= #
  LIMIT = 80

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1400

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 800

  # ========================================================================= #
  # === FONT_TO_USE
  # ========================================================================= #
  FONT_TO_USE = :dejavu_condensed_22 # 'Hack 20'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = ARGV,
      run_already = true
    )
    super(:vertical)
    reset
    set_display_this_file(i) if i
    run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #  
    @configuration = [true, __dir__, NAMESPACE]
    set_use_this_font(FONT_TO_USE)
    # ======================================================================= #
    # === @display_this_file
    # ======================================================================= #
    @display_this_file = nil
    # ======================================================================= #
    # === @title
    # ======================================================================= #
    @title  = TITLE
    # ======================================================================= #
    # === @width
    # ======================================================================= #
    set_width(WIDTH)
    # ======================================================================= #
    # === @height
    # ======================================================================= #
    set_height(HEIGHT)
    # ======================================================================= #
    # === @use_numbered_lines_widget
    # ======================================================================= #
    @use_numbered_lines_widget = true
    use_gtk_paradise_project_css_file
    increase_font_size
  end

  # ========================================================================= #
  # === set_display_this_file
  # ========================================================================= #
  def set_display_this_file(i)
    i = i.first.to_s if i.is_a? Array
    @display_this_file = i
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === do_open_a_file
  # ========================================================================= #
  def do_open_a_file
    this_file = ::Gtk.file_selection
    this_file = this_file.first
    if File.exist? this_file
      @file_path_widget << this_file
      file_content = File.read(this_file)
      @source_buffer.set_text(file_content)
    end
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    @source_buffer = create_text_buffer
    file_handle = ''.dup
    if File.exist? @display_this_file
      file_handle << File.readlines(@display_this_file).join
    end
    # ======================================================================= #
    # Zwischen 80 und 300 denken wir, das es eine sehr lange command line is.
    # ======================================================================= #
    if file_handle.size > 80 and file_handle.size < 300
      file_handle.insert(80, "\n")
    end
    @source_buffer.text = file_handle
    @text_field = create_text_view(@source_buffer)
    if @text_field.respond_to? :show_line_numbers=
      @text_field.show_line_numbers = true
    end
    if @text_field.respond_to? :tab_width=
      @text_field.tab_width = 2
    end
    if @text_field.respond_to? :auto_indent=
      @text_field.auto_indent = true
    end
    # @text_field.show_line_markers = true
    # @text_field.smart_home_end = true
    # @text_field.show_margin=true 
    if @use_numbered_lines_widget
      require 'gtk_paradise/widgets/gtk3/numbered_lines/numbered_lines.rb'
      hbox = gtk_hbox
      n_lines = @text_field.buffer?.text.split("\n").size
      @numbered_lines_widget = ::Gtk::NumberedLines.new {{ specify_n_lines: n_lines }}
      @numbered_lines_widget.set_size_request(50, 200)
      hbox.minimal(@numbered_lines_widget, 1)
      hbox.maximal(@text_field, 0)
      @scrolled_window = create_scrolled_window(hbox)
    else
      @scrolled_window = create_scrolled_window(@text_field)
    end
    create_buttons
    create_file_path_widget
  end

  # ========================================================================= #
  # === create_file_path_widget
  # ========================================================================= #
  def create_file_path_widget
    @file_path_widget = ::Gtk::FilePathWidget.new
  end

  # ========================================================================= #
  # === create_buttons
  # ========================================================================= #
  def create_buttons
    # ======================================================================= #
    # Next, add the quit-button:
    # ======================================================================= #
    @button_quit = quit_button('_Quit') { :exit_on_click_event }
  end

  # ========================================================================= #
  # === try_to_load_gtksourceview2
  # ========================================================================= #
  def try_to_load_gtksourceview2
    begin
      require 'gtksourceview2'
    rescue LoadError
      if ::Gtk.is_on_roebe?
        puts 'gtksourceview2 is not available.'
      end
    end
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    try_to_load_gtksourceview2
    create_skeleton_then_connect_skeleton
  end; alias start run # === start

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    minimal(@menu,             0)
    minimal(@file_path_widget, 0)
    maximal(@scrolled_window,  0)
    minimal(@button_quit,      0)
  end

  # ========================================================================= #
  # === Gtk::FileViewerModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = ::Gtk::FileViewer.new
    r = ::Gtk.run
    r << _
    r.infer_the_size
    r.automatic
    r.top_left_then_run
  end

end

# =========================================================================== #
# === Gtk.file_viewer
# =========================================================================== #
def self.file_viewer(i = nil)
  ::Gtk::FileViewer.new(i)
end

end