#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::LsmodModule
#
# This module can be used to display the lsmod-result in a graphical manner.
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/lsmod/lsmod_module.rb'
# include ::Gtk::LsmodModule
# =========================================================================== #
module Gtk

module LsmodModule

  include ::Gtk::BaseModule

  # ========================================================================= #
  # === N
  # ========================================================================= #
  N = "\n"

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'lsmod'

  # ========================================================================= #
  # === FONT_SIZE
  # ========================================================================= #
  FONT_SIZE = 20

  # ========================================================================= #
  # === PADDING
  # ========================================================================= #
  PADDING = 8

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = "Liberation serif #{FONT_SIZE}"

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH  = 1440

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT =  900

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      run_already = true,
      &block
    )
    super(:vertical)
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    # ======================================================================= #
    # === @array_lsmod_result
    # ======================================================================= #
    @array_lsmod_result = []
    # ======================================================================= #
    # === @lsmod_result
    # ======================================================================= #
    @lsmod_result = `lsmod`
    # ======================================================================= #
    # === @debug
    # ======================================================================= #
    @debug = false # True if we wish to debug this application.
    use_gtk_paradise_project_css_file
  end

  # ========================================================================= #
  # === create_vbox
  # ========================================================================= #
  def create_vbox
    @v_box = gtk_vbox
  end

  # ========================================================================= #
  # === create_renderer
  # ========================================================================= #
  def create_renderer
    @renderer = gtk_cell_renderer_text
    @renderer.editable = true
    @renderer.set_property('background', 'white')
    @renderer.set_property('foreground', 'darkgreen')
  end
  
  # ========================================================================= #
  # === update_listing
  # ========================================================================= #
  def update_listing
    @lsmod_result = `lsmod`
    sanitize_data
    update_liststore
  end

  # ========================================================================= #
  # === update_liststore
  # ========================================================================= #
  def update_liststore
    array = [] # tmp ary
    @array_lsmod_result.each_with_index { |line, index|
      # we skip the first line cuz we dont need it in 
      # the listing
      unless index == 0
        _ = line.squeeze.split(' ') 
        array << _
      end  
    }
    array = array.sort
    @array_lsmod_result = array
    @array_lsmod_result.each { |entry|
      iter = @list_store.append
      iter.set_value(0, entry[0])
      iter.set_value(1, entry[1])
      iter.set_value(2, entry[2])
    }
  end

  # ========================================================================= #
  # === create_scrolled_window
  #
  # Contains our lsmod.
  # ========================================================================= #
  def create_scrolled_window
    @scrolled_window = gtk_scrolled_window { :always }
    @scrolled_window.set_shadow_type_in
    @scrolled_window.bblack2
    @scrolled_window.pad8px
  end

  # ========================================================================= #
  # === create_treeview
  # ========================================================================= #
  def create_treeview
    @tree_view = gtk_treeview(@list_store)
    # ======================================================================= #
    # We want to make the headers of this tree-view clickable.
    # ======================================================================= #
    @tree_view.headers_clickable = true
    # @tree_view.rules_hint = true 
    @tree_view.append('Module',  @renderer, text: 0)
    @tree_view.append('Size',    @renderer, text: 1)
    @tree_view.append('Used by', @renderer, text: 2)
    @tree_view.drag_action_copy_and_action_move
  end

  # ========================================================================= #
  # === sanitize_data
  # ========================================================================= #
  def sanitize_data
    @array_lsmod_result = @lsmod_result.split(N)
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    8
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    :audio_favicon
  end

  # ========================================================================= #
  # === create_buttons
  # ========================================================================= #
  def create_buttons
    # ======================================================================= #
    # === @button_update
    #
    # Create the update-button first.
    # ======================================================================= #
    @button_update = gtk_bold_button('_Update')
    @button_update.bblack2
    @button_update.pad8px
    @button_update.on_clicked {
      e 'Updating Listing' if @debug
      update_listing
    }
    @button_update.hint = ' <b>Update</b> the module listing seen above. '
    @button_update.modify_background(:active,   :coral)
    @button_update.modify_background(:normal,   :mediumaquamarine)
    @button_update.modify_background(:prelight, :darkgreen) 
  end

  # ========================================================================= #
  # === create_liststore
  # ========================================================================= #
  def create_liststore
    @list_store = ::Gtk::ListStore.new(String, String, String)
    update_liststore
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    @v_box.maximal(@tree_view)
    @v_box.maximal(@button_update)
    # Das geht nicht!!!
    @tree_view.focus_vadjustment = @scrolled_window.vadjustment
    @tree_view.focus_hadjustment = @scrolled_window.hadjustment
    @scrolled_window.add_with_viewport(@v_box)
    pack_maximal(@scrolled_window, 2)
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_renderer
    create_vbox
    create_scrolled_window
    create_liststore
    create_treeview
    create_buttons
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    sanitize_data
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::LsmodModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    r = ::Gtk.run
    _ = ::Gtk::Lsmod.new(r)
    r << _
    r.automatic
    r.top_left_then_run
  end

  # ========================================================================= #
  # === Gtk::LsmodModule[]
  # ========================================================================= #
  def self.[](i = '')
    new(i)
  end

end; end